% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est_map_hmm.R
\name{est_rf_hmm_sequential}
\alias{est_rf_hmm_sequential}
\title{Multipoint analysis using Hidden Markov Models: Sequential phase elimination}
\usage{
est_rf_hmm_sequential(
  input.seq,
  twopt,
  start.set = 4,
  thres.twopt = 5,
  thres.hmm = 50,
  extend.tail = NULL,
  phase.number.limit = 20,
  sub.map.size.diff.limit = Inf,
  info.tail = TRUE,
  reestimate.single.ph.configuration = FALSE,
  tol = 0.1,
  tol.final = 0.001,
  verbose = TRUE,
  detailed.verbose = FALSE,
  high.prec = FALSE
)
}
\arguments{
\item{input.seq}{an object of class \code{mappoly.sequence}}

\item{twopt}{an object of class \code{poly.est.two.pts.pairwise}
containing the two-point information}

\item{start.set}{number of markers to start the phasing procedure (default = 4)}

\item{thres.twopt}{the LOD threshold used to determine if the linkage
phases compared via two-point analysis should be considered 
for the search space reduction (A.K.A. \eqn{\eta} in 
\cite{Mollinari and Garcia (2019)}, default = 5)}

\item{thres.hmm}{the LOD threshold used to determine if the linkage
phases compared via hmm analysis should be evaluated in the 
next round of marker inclusion (default = 50)}

\item{extend.tail}{the length of the chain's tail that should
be used to calculate the likelihood of the map. If \code{NULL} (default), 
the function uses all markers positioned. Even if \code{info.tail = TRUE}, 
it uses at least \code{extend.tail} as the tail length}

\item{phase.number.limit}{the maximum number of linkage phases of the sub-maps defined 
by arguments \code{info.tail} and \code{extend.tail}. Default is 20. If the
size exceeds this limit, the marker will not be inserted. If
\code{Inf}, then it will insert all markers.}

\item{sub.map.size.diff.limit}{the maximum accepted length
difference between the current and the previous sub-map defined 
by arguments \code{info.tail} and \code{extend.tail}. If the
size exceeds this limit, the marker will not be inserted. If
\code{NULL}(default), then it will insert all markers.}

\item{info.tail}{if \code{TRUE} (default), it uses the complete informative tail
of the chain (i.e. number of markers where all homologous 
(\eqn{ploidy x 2}) can be distinguished) to calculate the map likelihood}

\item{reestimate.single.ph.configuration}{logical. If \code{FALSE} (default)
returns a map without re-estimating the map parameters in cases
where there are only one possible linkage phase configuration}

\item{tol}{the desired accuracy during the sequential phase (default = 10e-02)}

\item{tol.final}{the desired accuracy for the final map (default = 10e-04)}

\item{verbose}{If \code{TRUE} (default), current progress is shown; if
\code{FALSE}, no output is produced}

\item{detailed.verbose}{If \code{TRUE}, the expansion of the current 
submap is shown;}

\item{high.prec}{logical. If \code{TRUE} uses high precision 
(long double) numbers in the HMM procedure implemented in C++,
which can take a long time to perform (default = FALSE)}
}
\value{
A list of class \code{mappoly.map} with two elements: 

i) info:  a list containing information about the map, regardless of the linkage phase configuration:
\item{m}{the ploidy level}
\item{n.mrk}{number of markers}
\item{seq.num}{a vector containing the (ordered) indices of markers in the map, 
               according to the input file}
\item{mrk.names}{the names of markers in the map}
\item{seq.dose.p}{a vector containing the dosage in parent 1 for all markers in the map}
\item{seq.dose.q}{a vector containing the dosage in parent 2 for all markers in the map}
\item{sequence}{a vector indicating the sequence (usually chromosome) each marker belongs 
                as informed in the input file. If not available, 
                \code{sequence = NULL}}
\item{sequence.pos}{physical position (usually in megabase) of the markers into the sequence}
\item{seq.ref}{reference base used for each marker (i.e. A, T, C, G). If not available, 
                \code{seq.ref = NULL}}
\item{seq.alt}{alternative base used for each marker (i.e. A, T, C, G). If not available, 
                \code{seq.ref = NULL}}
\item{chisq.pval}{a vector containing p-values of the chi-squared test of Mendelian 
                  segregation for all markers in the map}
\item{data.name}{name of the dataset of class \code{mappoly.data}}
\item{ph.thres}{the LOD threshold used to define the linkage phase configurations to test}

ii) a list of maps with possible linkage phase configuration. Each map in the list is also a 
   list containing
\item{seq.num}{a vector containing the (ordered) indices of markers in the map, 
               according to the input file}
\item{seq.rf}{a vector of size (\code{n.mrk - 1}) containing a sequence of recombination 
              fraction between the adjacent markers in the map}
\item{seq.ph}{linkage phase configuration for all markers in both parents}
\item{loglike}{the hmm-based multipoint likelihood}
}
\description{
Performs the multipoint analysis proposed by \cite{Mollinari and
 Garcia (2019)} in a sequence of markers removing unlikely phases
using sequential multipoint information.
}
\details{
This function sequentially includes markers into a map given an
ordered sequence. It uses two-point information to eliminate
unlikely linkage phase configurations given \code{thres.twopt}. The
search is made within a window of size \code{extend.tail}. For the
remaining configurations, the HMM-based likelihood is computed and
the ones that pass the HMM threshold (\code{thres.hmm}) are eliminated.
}
\examples{
 \donttest{
    mrk.subset<-make_seq_mappoly(hexafake, 1:20)
    red.mrk<-elim_redundant(mrk.subset)
    unique.mrks<-make_seq_mappoly(red.mrk)
    subset.pairs<-est_pairwise_rf(input.seq = unique.mrks,
                                  ncpus = 1,
                                  verbose=TRUE)
    subset.map <- est_rf_hmm_sequential(input.seq = unique.mrks,
                                        thres.twopt = 5,
                                        thres.hmm = 10,
                                        extend.tail = 10,
                                        tol = 0.1,
                                        tol.final = 10e-3,
                                        phase.number.limit = 5,
                                        twopt = subset.pairs,
                                        verbose = TRUE)
     print(subset.map, detailed = TRUE)
     plot(subset.map)
     plot(subset.map, left.lim = 0, right.lim = 1, mrk.names = TRUE)
     plot(subset.map, phase = FALSE)
     
     ## Retrieving simulated linkage phase
     ph.P <- maps.hexafake[[1]]$maps[[1]]$seq.ph$P
     ph.Q <- maps.hexafake[[1]]$maps[[1]]$seq.ph$Q
     ## Estimated linkage phase
     ph.P.est <- subset.map$maps[[1]]$seq.ph$P
     ph.Q.est <- subset.map$maps[[1]]$seq.ph$Q
     compare_haplotypes(m = 6, h1 = ph.P[names(ph.P.est)], h2 = ph.P.est)
     compare_haplotypes(m = 6, h1 = ph.Q[names(ph.Q.est)], h2 = ph.Q.est)
   }

}
\references{
Mollinari, M., and Garcia, A.  A. F. (2019) Linkage
    analysis and haplotype phasing in experimental autopolyploid
    populations with high ploidy level using hidden Markov
    models, _G3: Genes, Genomes, Genetics_. 
    \url{https://doi.org/10.1534/g3.119.400378}
}
\author{
Marcelo Mollinari, \email{mmollin@ncsu.edu}
}
