% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RidgePEER.R
\name{RidgePEER}
\alias{RidgePEER}
\title{Graph-constrained estimation with enhanced regulariazation parameter selection}
\usage{
RidgePEER(Q, y, Z, X = NULL, Q.normalized = FALSE, add.Ridge = TRUE,
  add.PEER = TRUE, optim.function = "sbplx", x0 = c(1, 1),
  lambda.Q.lo = 10^(-5), lambda.Q.up = 1e+06, lambda.R.lo = 10^(-5),
  lambda.R.up = 1e+06, verbose = TRUE, compute.Grace = FALSE)
}
\arguments{
\item{Q}{graph-originated penalty matrix \eqn{(p \times p)}, typically: a graph Laplacian matrix}

\item{y}{response values matrix \eqn{(n \times 1)}}

\item{Z}{design matrix \eqn{(n \times p)} modeled as random effects variables (to be penalized in regression modeling)}

\item{X}{design matrix \eqn{(n \times k)} modeled as fixed effects variables (not to be penalized in regression modeling);
note that an additional column representing intercept will always be added to \code{X} for the sake of computational simplicity 
(particularly, if \code{X} is initially \code{NULL}, it will be transformed so as to contain a single column of 1's)}

\item{Q.normalized}{logical whether or not to use normalized version of a graph-originated penalty matrix}

\item{add.Ridge}{logical whether or not to include Ridge penalty term in \code{RidgePEER} model penalty term}

\item{add.PEER}{logical whether or not to include PEER penalty term in \code{RidgePEER} model penalty term}

\item{optim.function}{signature of optimization algorithm used in MLE estimation (see: Details)}

\item{x0}{2-elements vector of initial values for \eqn{\lambda_Q, \lambda_R} parameters MLE 
estimation in optimization algorithm}

\item{lambda.Q.lo}{lower boundary for values space in which we search for optimal value of \eqn{\lambda_Q}}

\item{lambda.Q.up}{upper boundary for values space in which we search for optimal value of \eqn{\lambda_Q}}

\item{lambda.R.lo}{lower boundary for values space in which we search for optimal value of \eqn{\lambda_R}}

\item{lambda.R.up}{upper boundary for values space in which we search for optimal value of \eqn{\lambda_R}}

\item{verbose}{logical whether or not message out information from function execution}

\item{compute.Grace}{logical whether or not to compute \href{https://arxiv.org/abs/1506.08339}{Grace test} 
(a significance test for graph-constrained estimation) results}
}
\value{
\item{b.est}{vector with estimated values of \eqn{b} coefficients}
\item{beta.est}{vector with estimated values of \eqn{\beta} coefficients}
\item{opt.out}{output from the \eqn{\lambda_Q, \lambda_R} parameters optimization algorithm}
\item{opt.bestval}{objective function value from the \eqn{\lambda_Q, \lambda_R} parameters optimization algorithm}
\item{lambda.Q}{ML estimator value of \eqn{\lambda_Q} regularization parameter}
\item{lambda.R}{ML estimator value of \eqn{\lambda_2} regularization parameter}
\item{lambda.2}{\code{lambda.R}/\code{lambda.Q} value}
\item{grace.test.res}{output from the Grace significance test for graph-constrained estimation computation}
}
\description{
Perform penalized estimation where penalty term is a linear combination of 
graph-originated penalty and Ridge-originated penalty terms. Corresponding regularization
parameter values \eqn{\lambda} (\eqn{\lambda_Q, \lambda_R}) are estimated as ML 
estimators from corresponding Linear Mixed Model solution.
}
\details{
We assume a model: \deqn{y = X\beta + Zb + \varepsilon} and we 
estimate its coefficients  \eqn{\beta, b} as follows: 
\deqn{\widehat{\beta}, \widehat{b}= arg \; min_{\beta,b}\left \{ (y - X\beta - Zb)^T(y - X\beta - Zb) + \lambda_Qb^TQb +  \lambda_Rb^Tb\right \}, }
or, equivalently: 
\deqn{\widehat{\beta}, \widehat{b}= arg \; min_{\beta,b}\left \{ (y - X\beta - Zb)^T(y - X\beta - Zb) + \lambda_Q (b^TQb +  \lambda_2 b^Tb\right) \}, }
  where: 
\itemize{
 \item \eqn{y} - a response variable,  
 \item \eqn{X} - matrix of variables whose coefficients we do not want to penalize in the estimation process (assumed to be standarized to have mean equal 0 and variance equal 1),  
 \item \eqn{Z} - matrix of variables whose coefficients we want to penalize in the estimation process (assumed to be standarized to have mean equal 0 and variance equal 1),  
 \item \eqn{Q} - a graph-originated penalty matrix, typically: a graph Laplacian matrix, 
 \item \eqn{\lambda_Q, \lambda_R, \lambda_2} - regularization parameters (\eqn{\lambda_R = \lambda_Q  \lambda_2}). 
} 

In the coefficient estimation formula, \eqn{b^TQb} denotes a graph-originated model penalty term, 
\eqn{b^Tb} denotes a Ridge-originated model penalty term.

A graph-originated penalty term allows imposing similarity between coefficients 
of variables which are similar (or connected), based on some graph information given. 
Additional Ridge-originated penalty term facilitates parameters estimation: 
it reduces computational issues (arising from singularity in a graph-originated penalty matrix) 
and yields plausible results in situations when graph information is not informative or when 
it is unclear whether connectivities represented by a graph reflect similarities among corresponding coefficients.

Implementation of model regularization parameters \eqn{\lambda} (\eqn{\lambda_Q, \lambda_R}) selection 
utilizes the known fact of equivalence between penalized regression and Linear Mixed Model solutions, 
and provides values of the two regularization parameters that are Maximum Likelihood estimators of the latter.

Several optimization algorithms for minimization objective function in MLE estimation 
are available: \code{sbplx}, \code{bobyqa}, \code{cobyla}, \code{lbfgs}, \code{neldermead}, 
\code{auglag}, \code{slsqp}, \code{varmetric}. Simulation study conducted shown that \code{sbplx} 
algorithm was of the best performance in the cases investigated. This algorithm is set as default
optimization algorithm choice and is strongly recommended by the package authors for 
the \code{RidgePEER} function. All the optimization algorithms available are implemented
in the \code{nloptr} package. Please refer to the \code{nloptr} package for 
details.
}
\examples{
# Example 1. model without covariates

set.seed(1234)
# graph Adjacency matrix
n <- 200 
p1 <- 10
p2 <- 90
p <- p1 + p2
A <- matrix(rep(0, p*p), nrow = p, ncol = p)
A[1:p1, 1:p1] <- 1
A[(p1+1):p, (p1+1):p] <- 1
# graph Laplacian matrix
L <- Adj2Lap(A)
L.norm <- L2L.normalized(L)
# Z design matrix 
Z <- matrix(rnorm(n*p), nrow = n, ncol = p)
# True b coefficients 
b.true<- c(rep(1, p1), rep(0, p2))
beta.true <- runif(3)
intercept <- 0
eta <- intercept + Z \%*\% b.true 
R2 <- 0.5 # assumed variance explained 
sd.eps <- sqrt(var(eta) * (1 - R2) / R2)
error <- rnorm(n, sd = sd.eps)
# Y observed 
Y <- eta + error

# model estimation
RidgePEER.fit <- RidgePEER(Q = L.norm, y = Y, Z = Z, X = NULL)

# b coefficient estimates
RidgePEER.fit$b.est
# MLE of regularization parameters lambda
RidgePEER.fit$lambda.Q
RidgePEER.fit$lambda.R


# Example 2.: model with non-penalized covariates 

# X design matrix (covariates which are not to be penalized)
X <- matrix(rnorm(n*3), nrow = n, ncol = 3)
beta.true <- runif(3)
intercept <- 0
eta <- intercept + Z \%*\% b.true + X \%*\% beta.true
R2 <- 0.5 # assumed variance explained 
sd.eps <- sqrt(var(eta) * (1 - R2) / R2)
error <- rnorm(n, sd = sd.eps)
# Y observed 
Y <- eta + error

# model estimation
RidgePEER.fit <- RidgePEER(Q = L.norm, y = Y, Z = Z, X = X)

# b coefficient estimates
RidgePEER.fit$b.est
# beta coefficient estimates
RidgePEER.fit$beta.est
# MLE of regularization parameters lambda
RidgePEER.fit$lambda.Q
RidgePEER.fit$lambda.R

}
\references{
Brumback, B. A., Ruppert, D., Wand, M. P., Comment on 'Variable selection and function estimation in
additive nonparametric regression using a data-based prior'. Journal of the American Statistical
Association (1999): 94, 794–797.

Karas, M., Brzyski, D., Randolph, T., Harezlak, D. Brain connectivity-informed regularization methods for regression. Paper in progress, to be submited as an invited paper on CCNS for a special issue of  Statistics in Biosciences by Nov 30, 2016 (reference will be updated).

Li, C., Li, H., Network-constrained regularization and variable selection for analysis of genomic data. 
Bioinformatics (2008): 24(9), 1175-1182.

Li, C., Li, H., Variable selection and regression analysis for graph-structured covariates with an application to genomics. 
The Annals of Applied Statistics (2010): 4(3), 1498–1516.

Randolph, T., Harezlak, J., Feng, Z., Structured penalties for functional linear models—partially empirical 
eigenvectors for regression. The Electronic Journal of Statistics (2012): 6, 323-353.
}

