<!-- README.md is generated from README.Rmd. Please edit that file -->




# mi4p <img src="man/figures/logo.png" align="right" width="200"/>

# A multiple imputation framework for proteomics
## Marie Chion, Christine Carapito and Frédéric Bertrand

<!-- badges: start -->
[![Lifecycle: stable](https://img.shields.io/badge/lifecycle-stable-green.svg)](https://lifecycle.r-lib.org/articles/stages.html)
[![Project Status: Active – The project has reached a stable, usable state and is being actively developed.](https://www.repostatus.org/badges/latest/active.svg)](https://www.repostatus.org/#active)
[![R-CMD-check](https://github.com/mariechion/mi4p/workflows/R-CMD-check/badge.svg)](https://github.com/mariechion/mi4p/actions)
<!--[![Codecov test coverage](https://codecov.io/gh/mariechion/mi4p/branch/master/graph/badge.svg)](https://codecov.io/gh/mariechion/mi4p?branch=master)-->
[![CRAN status](https://www.r-pkg.org/badges/version/mi4p)](https://cran.r-project.org/package=mi4p)
[![CRAN RStudio mirror downloads](https://cranlogs.r-pkg.org/badges/mi4p)](https://cran.r-project.org/package=mi4p)
[![GitHub Repo stars](https://img.shields.io/github/stars/mariechion/mi4p?style=social)](https://github.com/mariechion/mi4p)
<!--[![DOI](https://zenodo.org/badge/18441799.svg)](https://zenodo.org/badge/latestdoi/18441799)-->
<!-- badges: end -->

This repository contains the R code and package for the _mi4p_ methodology (Multiple Imputation for Proteomics), proposed by Marie Chion, Christine Carapito and Frédéric Bertrand (2021) in *Accounting for multiple imputation-induced variability for differential analysis in mass spectrometry-based label-free quantitative proteomics*, [https://arxiv.org/abs/2108.07086](https://arxiv.org/abs/2108.07086). 

The following material is available on the Github repository of the package [https://github.com/mariechion/mi4p/](https://github.com/mariechion/mi4p/).

1. The `Functions` folder contains all the functions used for the workflow.  

2. The `Simulation-1`, `Simulation-2` and `Simulation-3` folders contain all the R scripts and data used to conduct simulated experiments and evaluate our methodology. 

3. The  `Arabidopsis_UPS` and `Yeast_UPS` folders contain all the R scripts and data used to challenge our methodology on real proteomics datasets. Raw experimental data were deposited with the ProteomeXchange Consortium via the PRIDE partner repository with the dataset identifiers PXD003841 and PXD027800.

This website and these examples were created by M. Chion, C. Carapito and F. Bertrand.

## Installation

You can install the released version of mi4p from [CRAN](https://CRAN.R-project.org) with:


```r
install.packages("mi4p")
```

You can install the development version of mi4p from [github](https://github.com) with:


```r
devtools::install_github("mariechion/mi4p")
```

## Examples

### First section


```r
library(mi4p)
```


```r
set.seed(4619)
datasim <- protdatasim()
str(datasim)
#> 'data.frame':	200 obs. of  11 variables:
#>  $ id.obs: int  1 2 3 4 5 6 7 8 9 10 ...
#>  $ X1    : num  99.6 99.9 100.2 99.8 100.4 ...
#>  $ X2    : num  97.4 101.3 100.3 100.2 101.7 ...
#>  $ X3    : num  100.3 100.9 99.1 101.2 100.6 ...
#>  $ X4    : num  99.4 99.2 98.5 99.1 99.5 ...
#>  $ X5    : num  98.5 99.7 100 100.2 100.7 ...
#>  $ X6    : num  200 199 199 200 199 ...
#>  $ X7    : num  200 200 202 199 199 ...
#>  $ X8    : num  202 199 200 199 201 ...
#>  $ X9    : num  200 200 199 201 200 ...
#>  $ X10   : num  200 198 200 201 199 ...
#>  - attr(*, "metadata")='data.frame':	10 obs. of  3 variables:
#>   ..$ Sample.name: chr [1:10] "X1" "X2" "X3" "X4" ...
#>   ..$ Condition  : Factor w/ 2 levels "A","B": 1 1 1 1 1 2 2 2 2 2
#>   ..$ Bio.Rep    : int [1:10] 1 2 3 4 5 6 7 8 9 10
```

It s the dataset shipped with package.

```r
save(datasim, file="datasim.RData", compress = "xz")
```


```r
attr(datasim, "metadata")
#>    Sample.name Condition Bio.Rep
#> 1           X1         A       1
#> 2           X2         A       2
#> 3           X3         A       3
#> 4           X4         A       4
#> 5           X5         A       5
#> 6           X6         B       6
#> 7           X7         B       7
#> 8           X8         B       8
#> 9           X9         B       9
#> 10         X10         B      10
```


## AMPUTATION

```r
MV1pct.NA.data <- MVgen(dataset = datasim[,-1], prop_NA = 0.01)
MV1pct.NA.data
#>            X1        X2        X3
#> 1    99.62136  97.36890 100.28075
#> 2    99.86342 101.27309 100.86058
#> 3   100.21425 100.30241  99.07865
#> 4    99.84381 100.20091 101.18119
#> 5   100.36412 101.70192 100.59543
#> 6   100.70214        NA 101.65048
#> 7    99.65595  99.31420 100.02326
#> 8    97.98251  97.59303 100.60547
#> 9    99.56975 100.09034  99.30467
#> 10   99.66258  99.27055  98.28584
#> 11  100.67448  98.76728  99.03524
#> 12   99.70519 101.44341 100.27238
#> 13  100.22737  98.20463 100.17269
#> 14   99.58471  99.56201 100.68163
#> 15   98.88906 100.91857 100.40506
#> 16   99.72203 100.13020 100.38206
#> 17   98.66712  99.87493  98.73070
#> 18  101.74826  99.50606  99.06766
#> 19  100.37070  99.92320 101.00412
#> 20   99.67479  99.88322  99.71573
#> 21  101.22486 101.49935 100.51086
#> 22  100.94777 100.74609  98.16971
#> 23   99.23466  98.98854  97.70679
#> 24  100.73409 100.11633  99.34654
#> 25   99.42769 101.44334 100.31472
#> 26   97.98284  98.58111 100.72714
#> 27  100.77200  99.87912 100.64746
#> 28  101.21120 101.21477  98.66129
#> 29   99.89863 100.70173  99.77229
#> 30  100.06061 100.37637  99.99705
#> 31   98.30618  98.66601  99.96445
#> 32   99.20158  99.72202 100.56677
#> 33  102.69643  99.43994  99.81578
#> 34   99.64883  99.69805 100.82740
#> 35  100.52176 100.35496 101.18851
#> 36   99.14024 100.18434 101.25128
#> 37   98.71673 100.27563 101.26047
#> 38   99.60300  99.96101  97.69730
#> 39  100.95813  99.81860 101.42839
#> 40  100.94308 101.05768 101.33514
#> 41  100.08233  99.25436 100.18024
#> 42   98.08118  98.05493 101.38092
#> 43  100.73244 101.01459  99.31688
#> 44  101.37057 100.74243 100.25070
#> 45  100.57588  98.59631  99.63513
#> 46  102.32726 100.72725 100.83068
#> 47  100.66396 101.24997 102.74212
#> 48  101.18708  99.14272 100.97788
#> 49  100.29781 100.05292 101.81405
#> 50   98.90440  98.88690 100.10132
#> 51   99.13211 101.27766 100.35375
#> 52   99.68964  98.53949        NA
#> 53  100.20759 100.75656 101.83451
#> 54  100.81381 100.86708  98.98648
#> 55   97.62262 100.96918 100.88207
#> 56   99.87966  99.11009 100.62407
#> 57  100.81265 101.40022 101.64736
#> 58   99.42476 100.95793  98.97635
#> 59   98.54276 100.53223 100.52489
#> 60  101.65724 100.08529 100.82927
#> 61  100.22370  99.10401 101.10075
#> 62  100.01483  98.93389 101.42779
#> 63   98.03262  98.51941 100.64758
#> 64  102.48775 100.54013 100.22541
#> 65  101.08856  99.96487  99.35269
#> 66   99.28101 100.31202  99.58869
#> 67   99.94741  99.59370 101.31019
#> 68   99.11708  99.87691 100.41860
#> 69   99.83367  99.48486 100.47715
#> 70  100.26151  98.90834  99.27541
#> 71  101.42782        NA  99.47898
#> 72   98.37754  97.58114  98.54586
#> 73   98.43999 102.07782  99.31780
#> 74   99.21218  99.76310 100.58792
#> 75  100.31827 100.87590 100.00028
#> 76  101.33601  99.30129 101.45846
#> 77   98.71098  98.45163 100.50164
#> 78  100.54735 100.13119  99.49891
#> 79  100.62054  99.90850  99.83848
#> 80  100.87418  99.07000  99.39502
#> 81  100.58658 100.55023  98.79955
#> 82  100.65936  99.83262 100.37488
#> 83  101.05624  99.81584  97.94433
#> 84   99.76635  99.48239 100.50835
#> 85   99.61604  99.94210  99.92007
#> 86  102.65635  98.83253 100.94177
#> 87   99.16549  98.59323  98.50025
#> 88   99.80314 102.83451  98.62313
#> 89  100.81485 100.59225  99.68692
#> 90  101.15941  99.94095  99.38685
#> 91   99.60963  99.75646  99.79006
#> 92   98.67055 101.07326  99.37645
#> 93   97.18646 100.47113 101.30328
#> 94   99.76743 100.12888 100.75160
#> 95   98.92139 100.45915  99.08668
#> 96  100.14744  99.68685 101.91843
#> 97   97.81162  99.63666 100.95298
#> 98   99.35723 100.81867  99.36142
#> 99  100.44599 101.11855 101.18338
#> 100  99.86970  99.79042  99.41267
#>            X4        X5        X6
#> 1    99.37889  98.48006 200.28026
#> 2    99.19046  99.71866 199.26177
#> 3    98.51259 100.02675 199.21966
#> 4    99.05420 100.18467 200.06153
#> 5    99.49536 100.65887 199.25196
#> 6    99.11372 100.94608 201.74710
#> 7   101.56787 100.01322 199.91695
#> 8    97.81128 101.07808 199.94233
#> 9    99.69204 100.78160 200.79724
#> 10   99.76234 100.15253 199.05707
#> 11  100.05411  99.65590  99.17835
#> 12  100.05246 100.22091 100.72780
#> 13   99.25860  97.97791 100.25975
#> 14  100.80704 101.25322 100.22989
#> 15   98.21385  99.92004 101.20937
#> 16  100.30495 100.50996  99.77492
#> 17  100.21022 100.16050  98.70055
#> 18  100.88594 100.89332  98.20664
#> 19   98.77845  99.51780 100.51887
#> 20   98.72810 100.61274  97.97179
#> 21  100.64889 101.37527 100.31040
#> 22  100.51673 100.34439 100.46798
#> 23   99.66054  99.63939 100.49429
#> 24   99.01384 101.20664  99.83183
#> 25   99.85531 100.97558  98.07407
#> 26  100.77219  99.50288 101.16027
#> 27  100.26483  99.76495 101.61521
#> 28  100.28078  99.24424  99.46075
#> 29   98.53766 100.04275  99.75579
#> 30  100.47969  99.08339  99.36436
#> 31  100.75600 101.78651  99.57463
#> 32  101.86184  98.29285 100.69424
#> 33  100.45453 100.66046 101.42928
#> 34  100.94480 101.13936 100.43190
#> 35  100.98460  99.39586 100.72377
#> 36   99.24923  99.25243 101.26967
#> 37  100.17444 100.22413  99.39826
#> 38  100.69089 100.56642 100.59508
#> 39  101.09042  99.95280 100.05541
#> 40   98.67754 100.78029 101.33639
#> 41   99.49072 101.11272  99.46501
#> 42   98.98588 100.38570 101.43547
#> 43  100.70114  99.63081 100.12055
#> 44  100.36574 100.12416  99.61283
#> 45   99.14692 100.54662 100.38188
#> 46  100.69548  99.14269  99.33752
#> 47  100.21230 100.15852 100.11243
#> 48  100.43447 100.20844  98.35361
#> 49   99.51584 100.27605  99.18655
#> 50   98.60870  99.54416 100.53822
#> 51   99.01066  98.65777  99.59689
#> 52   99.63273  99.82911 100.20886
#> 53  100.83636 101.20224 101.46609
#> 54   97.94897  99.05776 100.18764
#> 55   99.72121 101.16145  98.67508
#> 56  101.51961  99.07788  98.87347
#> 57  100.72335  98.98892  99.42230
#> 58  101.74751  99.12457 100.38954
#> 59  100.72203 100.67585  98.81364
#> 60   99.95037  98.87261  99.72656
#> 61  100.55131  98.18042  98.01345
#> 62   99.12747 100.93053 100.67624
#> 63   98.84928 100.03873 100.56751
#> 64  101.07624 100.45880 100.28039
#> 65  100.88600  99.88268  99.03970
#> 66   98.36661  98.88559 100.42823
#> 67  102.00179  99.71711 100.71517
#> 68  100.05917  98.87877 100.05524
#> 69  100.42913  99.81200  99.04303
#> 70  101.30210 100.63369  98.32762
#> 71   99.69239 101.13070 101.38540
#> 72   99.71640  99.18235 102.10023
#> 73   97.19951 100.03653  98.74474
#> 74  100.51794  98.67775 100.30000
#> 75   99.41642  99.73776 100.05191
#> 76  100.57889  99.82176  99.35001
#> 77   99.49692 100.56487  97.99646
#> 78  101.94890 100.94310  99.79346
#> 79  101.83898  99.71147  98.91599
#> 80   99.07229  99.84750  98.21041
#> 81  101.56748  99.77204 101.46388
#> 82  100.96218  98.40581 101.40310
#> 83  100.75954  99.44069 100.06661
#> 84   97.81779  99.47231  99.00190
#> 85  100.83718 100.31995 100.67056
#> 86  100.24468 100.60003 101.03815
#> 87  101.09024  98.26549  99.63777
#> 88  100.24650  99.66112 100.47485
#> 89  101.42699 100.49860 100.32587
#> 90   99.12203  99.97329 100.26401
#> 91  101.97879  99.59668  98.13397
#> 92  101.50588  99.35769  99.36484
#> 93  100.19676 100.38262 100.46563
#> 94  100.49422 100.80529 100.52150
#> 95  101.39591  99.88206 100.57992
#> 96   99.53541  99.69815 100.77521
#> 97   98.41874  99.46149 101.44124
#> 98   99.72925 100.37592  99.54315
#> 99  100.11626  98.76637 100.35712
#> 100 101.35380 101.48962  98.37529
#>            X7        X8        X9
#> 1   199.80422 201.60954 199.74508
#> 2   200.39344 199.48269 199.72970
#> 3   202.28551 199.71608 198.88835
#> 4   199.34245 199.18647 200.95961
#> 5   199.47146 200.96823 199.98379
#> 6   198.70242 201.80299 199.40298
#> 7   199.59548 201.39032 200.26089
#> 8   199.46429 199.78429 199.69111
#> 9   199.30691 199.78971 201.88039
#> 10  200.98124 200.12561 201.45554
#> 11   98.58828  99.22494  98.54141
#> 12   99.40128  99.56137 102.23024
#> 13  100.44203  99.57370  99.16147
#> 14  101.69888 101.43233 100.21297
#> 15  101.01489  99.91842 100.53775
#> 16  101.92973 100.30625 100.44396
#> 17   99.66331  99.55839  98.62409
#> 18   99.92798  97.68238  98.89593
#> 19  100.65552 101.80858  99.24056
#> 20         NA  99.61030  98.69528
#> 21   99.45380 100.46323 100.27759
#> 22  101.21767  99.35589  98.66862
#> 23   99.84157 101.00604  99.00019
#> 24  100.18725  98.13992 101.70280
#> 25   99.12720  99.53977 101.19468
#> 26   97.92605  99.47466  99.10165
#> 27  100.04600 100.55987  99.16342
#> 28  101.14263 101.04240  99.83072
#> 29  101.32759 101.84453  99.57541
#> 30   99.03801  99.57686  99.59102
#> 31   97.63290  99.90373  99.99344
#> 32   99.51856  99.06693  99.07289
#> 33  100.05872  99.89433  99.72724
#> 34   98.86824  99.96153 100.54200
#> 35  100.78591  99.62805  99.01663
#> 36   99.00934  99.24133  99.41116
#> 37  101.40395 101.84888 100.57555
#> 38  100.64289  99.49468  99.37011
#> 39  100.25045 101.35255 100.22298
#> 40   99.06949 101.42822  99.97315
#> 41   99.42307  99.08463 100.73347
#> 42   98.73351  98.25351  99.17284
#> 43   98.27283  99.99836  98.28623
#> 44   99.72949  99.58595 100.59102
#> 45   99.85310 100.14963  99.37502
#> 46  100.98748  99.37594 100.35493
#> 47   98.59912  97.65577 100.91458
#> 48   98.66029 102.58168 100.31207
#> 49   99.93729 100.35470 101.07800
#> 50   99.79307 102.11009  98.73965
#> 51  101.53856 100.48919  99.52892
#> 52   99.59244  99.42755 100.85116
#> 53   99.25368  99.62083 101.46474
#> 54  101.23422  99.56874 100.23675
#> 55   99.75286 101.30089 100.41263
#> 56  100.09431 102.23957 101.24945
#> 57  100.87794  99.81577  99.51272
#> 58   98.87197  99.89209 100.84835
#> 59   98.79910 100.81882 100.46032
#> 60   99.90579  98.94344  99.89661
#> 61         NA 100.70073 100.97489
#> 62   98.69724  99.95896  99.09162
#> 63   99.63268  98.33434 101.74138
#> 64   98.74043  98.55782 100.48805
#> 65  100.72421 100.55019 101.13522
#> 66  100.11403  99.46400  99.54982
#> 67   99.53402 100.01518 100.47384
#> 68  100.47340  98.63438  98.65675
#> 69   99.16834 100.31664  99.89835
#> 70   99.53554 101.22029 101.03583
#> 71   97.77069  99.56227  99.91556
#> 72  101.28421        NA  97.77383
#> 73   98.03535 102.24238  99.67195
#> 74  100.60982 100.15740  99.96954
#> 75  100.53130 100.67662 101.86988
#> 76  101.78012 100.04703 101.89692
#> 77  101.16713  99.18905  99.04889
#> 78  102.36457 101.22784 100.68876
#> 79  100.72897  99.38159  98.90303
#> 80   98.99321 102.33368  98.56857
#> 81   99.25508  98.14549  98.95842
#> 82  100.64276 100.57123 101.04280
#> 83   99.26098  99.98457  99.77443
#> 84  101.11238  99.94912  98.83581
#> 85  101.00432  99.40805  99.39592
#> 86   99.16664  99.64301  98.89459
#> 87   99.77638 101.57385 100.94446
#> 88   99.82777 100.45078 100.22406
#> 89  100.42828 100.69815 100.49905
#> 90  101.33879  99.97464 100.31493
#> 91  100.54690  99.71475 102.65816
#> 92  101.20923 101.66662 101.03311
#> 93   98.08284 101.39629  99.03853
#> 94   98.69169 100.96759  98.70530
#> 95   99.18996 100.04035  99.50948
#> 96   99.18343 101.07734  99.54734
#> 97   99.59299 100.52423 100.40615
#> 98  100.70182 100.95334 101.53602
#> 99   99.79178  99.84165 101.68890
#> 100 100.23905  98.64045 100.19707
#>           X10
#> 1   199.75432
#> 2   198.39845
#> 3   200.40936
#> 4   201.27893
#> 5   198.73031
#> 6   199.90329
#> 7   201.08186
#> 8   199.70172
#> 9   199.08733
#> 10  199.41868
#> 11  101.83299
#> 12   99.57105
#> 13   99.52041
#> 14  100.80091
#> 15  100.03770
#> 16   99.33812
#> 17  100.24479
#> 18  100.56774
#> 19  101.23897
#> 20  100.35563
#> 21  100.04180
#> 22   98.92921
#> 23  101.20803
#> 24   98.46646
#> 25  101.62890
#> 26   99.97932
#> 27  101.53302
#> 28   97.23290
#> 29   99.84934
#> 30   99.93157
#> 31  101.02664
#> 32  101.44852
#> 33  100.61599
#> 34  101.70919
#> 35  100.32899
#> 36   99.87241
#> 37   98.78104
#> 38   99.50465
#> 39  100.33084
#> 40   98.71295
#> 41  100.14049
#> 42  101.87870
#> 43   99.04798
#> 44  100.78820
#> 45   99.16579
#> 46  100.13561
#> 47  100.99970
#> 48   99.56147
#> 49   99.64434
#> 50  100.96517
#> 51   99.00256
#> 52   99.45526
#> 53   99.94194
#> 54   99.41421
#> 55   99.82159
#> 56   99.21182
#> 57  101.15604
#> 58  100.76270
#> 59  101.25812
#> 60  101.12487
#> 61  101.00311
#> 62  100.28762
#> 63   99.17981
#> 64   99.06127
#> 65   99.01184
#> 66   98.45192
#> 67   97.41584
#> 68  101.44591
#> 69   99.59414
#> 70   99.05709
#> 71  100.34972
#> 72  100.94120
#> 73  100.73581
#> 74   99.92478
#> 75   98.25933
#> 76  100.17572
#> 77   98.77554
#> 78   99.42993
#> 79  102.18627
#> 80  100.57919
#> 81   99.50920
#> 82  100.22187
#> 83  100.01202
#> 84  100.49362
#> 85  100.10986
#> 86  101.33294
#> 87  100.69335
#> 88   99.59000
#> 89   98.37901
#> 90   98.57134
#> 91   99.01875
#> 92  102.42760
#> 93   98.16374
#> 94   99.84676
#> 95  100.51410
#> 96   98.55898
#> 97   99.65020
#> 98  101.36138
#> 99   99.68010
#> 100  98.82971
#>  [ reached 'max' / getOption("max.print") -- omitted 100 rows ]
```

## IMPUTATION

```r
MV1pct.impMLE <- multi.impute(data = MV1pct.NA.data, conditions = attr(datasim,"metadata")$Condition, method = "MLE", parallel = FALSE)
```

## ESTIMATION

```r
print(paste(Sys.time(), "Dataset", 1, "out of", 1))
#> [1] "2021-08-12 01:38:43 Dataset 1 out of 1"
MV1pct.impMLE.VarRubin.Mat <- rubin2.all(data = MV1pct.impMLE, metacond = attr(datasim, "metadata")$Condition) 
```

## PROJECTION

```r
print(paste("Dataset", 1, "out of",1, Sys.time()))
#> [1] "Dataset 1 out of 1 2021-08-12 02:05:38"
MV1pct.impMLE.VarRubin.S2 <- as.numeric(lapply(MV1pct.impMLE.VarRubin.Mat, function(aaa){
    DesMat = DAPAR::make.design(attr(datasim, "metadata"))
    return(max(diag(aaa)%*%t(DesMat)%*%DesMat))
  }))
```

## MODERATED T-TEST

```r
MV1pct.impMLE.mi4limma.res <- mi4limma(qData = apply(MV1pct.impMLE,1:2,mean), 
                 sTab = attr(datasim, "metadata"), 
                 VarRubin = sqrt(MV1pct.impMLE.VarRubin.S2))
MV1pct.impMLE.mi4limma.res
#> $logFC
#>      A_vs_B_logFC
#> 1   -1.012127e+02
#> 2   -9.927197e+01
#> 3   -1.004769e+02
#> 4   -1.000728e+02
#> 5   -9.911801e+01
#> 6   -9.982359e+01
#> 7   -1.003342e+02
#> 8   -1.007027e+02
#> 9   -1.002846e+02
#> 10  -1.007809e+02
#> 11   1.642084e-01
#> 12   4.052473e-02
#> 13  -6.232310e-01
#> 14  -4.972736e-01
#> 15  -8.743117e-01
#> 16  -1.487572e-01
#> 17   1.704648e-01
#> 18   1.364113e+00
#> 19  -7.736504e-01
#> 20   5.630360e-01
#> 21   9.424812e-01
#> 22   4.170600e-01
#> 23  -1.264042e+00
#> 24   4.178347e-01
#> 25   4.904022e-01
#> 26  -1.515887e-02
#> 27  -3.178319e-01
#> 28   3.805760e-01
#> 29  -6.799174e-01
#> 30   4.990594e-01
#> 31   2.695631e-01
#> 32  -3.121658e-02
#> 33   2.683149e-01
#> 34   1.491150e-01
#> 35   3.924688e-01
#> 36   5.472297e-02
#> 37  -2.712528e-01
#> 38  -2.177595e-01
#> 39   2.072179e-01
#> 40   4.547053e-01
#> 41   2.547398e-01
#> 42  -5.170840e-01
#> 43   1.133979e+00
#> 44   5.092196e-01
#> 45  -8.491259e-02
#> 46   7.063772e-01
#> 47   1.349054e+00
#> 48   4.962941e-01
#> 49   3.511585e-01
#> 50  -1.220144e+00
#> 51  -3.448379e-01
#> 52  -5.184845e-01
#> 53   6.179962e-01
#> 54  -5.934908e-01
#> 55   7.869669e-02
#> 56  -2.914607e-01
#> 57   5.575441e-01
#> 58  -1.067051e-01
#> 59   1.695508e-01
#> 60   3.595042e-01
#> 61  -2.846456e-01
#> 62   3.445642e-01
#> 63  -6.736214e-01
#> 64   1.532072e+00
#> 65   1.427288e-01
#> 66  -3.148159e-01
#> 67   8.832298e-01
#> 68  -1.830318e-01
#> 69   4.032647e-01
#> 70   2.409381e-01
#> 71   1.233375e+00
#> 72  -2.051791e+00
#> 73  -4.717118e-01
#> 74  -4.405299e-01
#> 75  -2.080830e-01
#> 76  -1.506797e-01
#> 77   3.097923e-01
#> 78  -8.702237e-02
#> 79   3.604256e-01
#> 80  -8.521368e-02
#> 81   7.887639e-01
#> 82  -7.293849e-01
#> 83  -1.639307e-02
#> 84  -4.691277e-01
#> 85   9.325053e-03
#> 86   6.400077e-01
#> 87  -1.402221e+00
#> 88   1.201843e-01
#> 89   5.378503e-01
#> 90  -1.762344e-01
#> 91   1.318162e-01
#> 92  -1.143514e+00
#> 93   4.786437e-01
#> 94   6.429156e-01
#> 95  -1.772527e-02
#> 96   3.687938e-01
#> 97  -1.066664e+00
#> 98  -8.906438e-01
#> 99   5.419909e-02
#> 100  1.126928e+00
#> 101 -8.616428e-01
#> 102 -4.405730e-01
#> 103 -9.865894e-01
#> 104 -1.041152e+00
#> 105  6.753898e-01
#> 106  1.474734e-01
#> 107  1.111835e-01
#> 108  1.681135e-01
#> 109 -7.067925e-02
#> 110 -9.789653e-01
#> 111 -9.668787e-01
#> 112  3.390004e-01
#> 113 -5.472008e-01
#> 114 -3.418560e-01
#> 115 -4.138748e-01
#> 116 -6.507368e-02
#> 117  2.593238e-01
#> 118  4.881388e-01
#> 119  5.347332e-01
#> 120  1.084537e+00
#> 121 -1.291370e+00
#> 122 -7.893439e-01
#> 123  2.287629e-01
#> 124 -2.975810e-01
#> 125 -9.810067e-01
#> 126  7.173674e-01
#> 127  1.373830e+00
#> 128  1.012382e+00
#> 129 -2.565706e-01
#> 130 -1.647907e-01
#> 131 -6.508092e-02
#> 132  5.175271e-01
#> 133  2.800798e-01
#> 134 -1.044942e+00
#> 135  1.260690e-01
#> 136  6.445337e-01
#> 137  1.646906e-01
#> 138 -6.823872e-01
#> 139 -6.108207e-01
#> 140 -3.258216e-01
#> 141 -4.868820e-01
#> 142  5.149480e-02
#> 143  5.364877e-01
#> 144  5.671157e-01
#> 145 -7.546692e-01
#> 146  1.110852e-01
#> 147 -6.772738e-01
#> 148  1.336448e+00
#> 149 -3.241622e-01
#> 150  6.122240e-01
#> 151 -8.587254e-01
#> 152  4.453814e-01
#> 153  9.925005e-01
#> 154 -2.654850e-01
#> 155  4.487299e-01
#> 156  4.818324e-01
#> 157  8.746962e-01
#> 158  5.247776e-02
#> 159  9.903067e-01
#> 160 -7.137321e-01
#> 161  3.478403e-01
#> 162 -2.509245e-01
#> 163 -1.043022e+00
#> 164  3.484151e-02
#> 165 -1.089838e-01
#> 166  9.315078e-01
#> 167  4.570214e-01
#> 168  6.675017e-01
#> 169 -2.731210e-01
#> 170 -5.864457e-01
#> 171  4.800880e-02
#> 172 -9.480365e-01
#> 173  7.188479e-01
#> 174  4.299004e-01
#> 175  7.732633e-01
#> 176  5.982326e-02
#> 177 -1.878878e+00
#> 178  8.313797e-02
#> 179  7.412046e-01
#> 180  1.244863e-01
#> 181 -4.439316e-02
#> 182  4.082467e-01
#> 183 -8.122601e-01
#> 184  3.380700e-01
#> 185  6.184746e-01
#> 186 -4.462906e-02
#> 187 -1.780221e-01
#> 188  7.422867e-01
#> 189  8.504415e-01
#> 190 -1.447240e+00
#> 191 -5.486541e-01
#> 192 -6.742420e-01
#> 193 -1.743576e-01
#> 194 -3.833636e-01
#> 195  1.066416e+00
#> 196 -4.101028e-01
#> 197 -5.133464e-01
#> 198  3.714431e-01
#> 199 -6.904036e-01
#> 200 -2.796176e-01
#> 
#> $P_Value
#>     A_vs_B_pval
#> 1   0.000000000
#> 2   0.000000000
#> 3   0.000000000
#> 4   0.000000000
#> 5   0.000000000
#> 6   0.000000000
#> 7   0.000000000
#> 8   0.000000000
#> 9   0.000000000
#> 10  0.000000000
#> 11  0.796941098
#> 12  0.949367396
#> 13  0.328845493
#> 14  0.435896381
#> 15  0.170804559
#> 16  0.815686143
#> 17  0.789383602
#> 18  0.032678839
#> 19  0.225507500
#> 20  0.377692370
#> 21  0.139852178
#> 22  0.513447430
#> 23  0.047758669
#> 24  0.512665570
#> 25  0.442263162
#> 26  0.981049169
#> 27  0.618473149
#> 28  0.550961487
#> 29  0.286774215
#> 30  0.434250458
#> 31  0.672742627
#> 32  0.960986570
#> 33  0.674170532
#> 34  0.815250906
#> 35  0.538586073
#> 36  0.931665593
#> 37  0.670811526
#> 38  0.732939800
#> 39  0.745408176
#> 40  0.476182828
#> 41  0.689774526
#> 42  0.417839744
#> 43  0.075726038
#> 44  0.424954526
#> 45  0.894149436
#> 46  0.268439313
#> 47  0.034647029
#> 48  0.436800667
#> 49  0.582161204
#> 50  0.056024524
#> 51  0.588971036
#> 52  0.416580176
#> 53  0.332923108
#> 54  0.352443711
#> 55  0.901857291
#> 56  0.647888970
#> 57  0.382361413
#> 58  0.867209930
#> 59  0.790486401
#> 60  0.573226392
#> 61  0.655583701
#> 62  0.589266718
#> 63  0.291258749
#> 64  0.016459648
#> 65  0.823029024
#> 66  0.621807695
#> 67  0.166485512
#> 68  0.774263458
#> 69  0.527474211
#> 70  0.705775614
#> 71  0.053416061
#> 72  0.001327678
#> 73  0.459847684
#> 74  0.490038170
#> 75  0.744382370
#> 76  0.813347884
#> 77  0.627379172
#> 78  0.891535505
#> 79  0.572243878
#> 80  0.893776333
#> 81  0.216578083
#> 82  0.253165324
#> 83  0.979506563
#> 84  0.462309373
#> 85  0.988341624
#> 86  0.315996655
#> 87  0.028121277
#> 88  0.850622027
#> 89  0.399393706
#> 90  0.782431621
#> 91  0.836360270
#> 92  0.073299806
#> 93  0.453280658
#> 94  0.313803559
#> 95  0.977841563
#> 96  0.563358277
#> 97  0.094779581
#> 98  0.162957862
#> 99  0.932318203
#> 100 0.077562329
#> 101 0.177083821
#> 102 0.489995750
#> 103 0.122253377
#> 104 0.102936427
#> 105 0.289994394
#> 106 0.817248618
#> 107 0.861691774
#> 108 0.792221635
#> 109 0.911812677
#> 110 0.125164854
#> 111 0.129891010
#> 112 0.595292753
#> 113 0.391250628
#> 114 0.592196448
#> 115 0.516668791
#> 116 0.918781532
#> 117 0.684490200
#> 118 0.444371933
#> 119 0.402130788
#> 120 0.089380260
#> 121 0.043150430
#> 122 0.216240509
#> 123 0.720000187
#> 124 0.641010594
#> 125 0.124380027
#> 126 0.261066023
#> 127 0.031460293
#> 128 0.112795260
#> 129 0.687662205
#> 130 0.796236910
#> 131 0.918772527
#> 132 0.417441001
#> 133 0.660759379
#> 134 0.101690383
#> 135 0.843400388
#> 136 0.312587589
#> 137 0.796357915
#> 138 0.285027866
#> 139 0.338565258
#> 140 0.609677681
#> 141 0.445545325
#> 142 0.935687696
#> 143 0.400588765
#> 144 0.374246784
#> 145 0.237090770
#> 146 0.861812910
#> 147 0.288651505
#> 148 0.036371683
#> 149 0.611499869
#> 150 0.337457060
#> 151 0.178553900
#> 152 0.485271972
#> 153 0.120032717
#> 154 0.677412026
#> 155 0.481996940
#> 156 0.450277631
#> 157 0.170616625
#> 158 0.934462815
#> 159 0.120853136
#> 160 0.263489240
#> 161 0.585731624
#> 162 0.694184376
#> 163 0.102320308
#> 164 0.956460519
#> 165 0.864401320
#> 166 0.144519990
#> 167 0.473939550
#> 168 0.295662704
#> 169 0.668679110
#> 170 0.358187287
#> 171 0.940032877
#> 172 0.137533834
#> 173 0.260083586
#> 174 0.500568547
#> 175 0.225739623
#> 176 0.925314553
#> 177 0.003280237
#> 178 0.896349025
#> 179 0.245558770
#> 180 0.845341413
#> 181 0.944541547
#> 182 0.522386143
#> 183 0.203206394
#> 184 0.596303267
#> 185 0.332549080
#> 186 0.944247318
#> 187 0.780280965
#> 188 0.244870442
#> 189 0.182777701
#> 190 0.023453268
#> 191 0.389994048
#> 192 0.290814582
#> 193 0.784691114
#> 194 0.548048245
#> 195 0.094856436
#> 196 0.520497055
#> 197 0.421212303
#> 198 0.560559032
#> 199 0.279409182
#> 200 0.661284332
(simplify2array(MV1pct.impMLE.mi4limma.res)$P_Value.A_vs_B_pval)[1:10]
#>  [1] 0 0 0 0 0 0 0 0 0 0

(simplify2array(MV1pct.impMLE.mi4limma.res)$P_Value.A_vs_B_pval)[11:200]<=0.05
#>   [1] FALSE FALSE FALSE FALSE FALSE
#>   [6] FALSE FALSE  TRUE FALSE FALSE
#>  [11] FALSE FALSE  TRUE FALSE FALSE
#>  [16] FALSE FALSE FALSE FALSE FALSE
#>  [21] FALSE FALSE FALSE FALSE FALSE
#>  [26] FALSE FALSE FALSE FALSE FALSE
#>  [31] FALSE FALSE FALSE FALSE FALSE
#>  [36] FALSE  TRUE FALSE FALSE FALSE
#>  [41] FALSE FALSE FALSE FALSE FALSE
#>  [46] FALSE FALSE FALSE FALSE FALSE
#>  [51] FALSE FALSE FALSE  TRUE FALSE
#>  [56] FALSE FALSE FALSE FALSE FALSE
#>  [61] FALSE  TRUE FALSE FALSE FALSE
#>  [66] FALSE FALSE FALSE FALSE FALSE
#>  [71] FALSE FALSE FALSE FALSE FALSE
#>  [76] FALSE  TRUE FALSE FALSE FALSE
#>  [81] FALSE FALSE FALSE FALSE FALSE
#>  [86] FALSE FALSE FALSE FALSE FALSE
#>  [91] FALSE FALSE FALSE FALSE FALSE
#>  [96] FALSE FALSE FALSE FALSE FALSE
#> [101] FALSE FALSE FALSE FALSE FALSE
#> [106] FALSE FALSE FALSE FALSE FALSE
#> [111]  TRUE FALSE FALSE FALSE FALSE
#> [116] FALSE  TRUE FALSE FALSE FALSE
#> [121] FALSE FALSE FALSE FALSE FALSE
#> [126] FALSE FALSE FALSE FALSE FALSE
#> [131] FALSE FALSE FALSE FALSE FALSE
#> [136] FALSE FALSE  TRUE FALSE FALSE
#> [141] FALSE FALSE FALSE FALSE FALSE
#> [146] FALSE FALSE FALSE FALSE FALSE
#> [151] FALSE FALSE FALSE FALSE FALSE
#> [156] FALSE FALSE FALSE FALSE FALSE
#> [161] FALSE FALSE FALSE FALSE FALSE
#> [166] FALSE  TRUE FALSE FALSE FALSE
#> [171] FALSE FALSE FALSE FALSE FALSE
#> [176] FALSE FALSE FALSE FALSE  TRUE
#> [181] FALSE FALSE FALSE FALSE FALSE
#> [186] FALSE FALSE FALSE FALSE FALSE
```

True positive rate

```r
sum((simplify2array(MV1pct.impMLE.mi4limma.res)$P_Value.A_vs_B_pval)[1:10]<=0.05)/10
#> [1] 1
```

False positive rate

```r
sum((simplify2array(MV1pct.impMLE.mi4limma.res)$P_Value.A_vs_B_pval)[11:200]<=0.05)/190
#> [1] 0.05789474
```



```r
MV1pct.impMLE.dapar.res <-limmaCompleteTest.mod(qData = apply(MV1pct.impMLE,1:2,mean), sTab = attr(datasim, "metadata"))
MV1pct.impMLE.dapar.res
#> $res.l
#> $res.l$logFC
#>      A_vs_B_logFC
#> 1   -1.012127e+02
#> 2   -9.927197e+01
#> 3   -1.004769e+02
#> 4   -1.000728e+02
#> 5   -9.911801e+01
#> 6   -9.982359e+01
#> 7   -1.003342e+02
#> 8   -1.007027e+02
#> 9   -1.002846e+02
#> 10  -1.007809e+02
#> 11   1.642084e-01
#> 12   4.052473e-02
#> 13  -6.232310e-01
#> 14  -4.972736e-01
#> 15  -8.743117e-01
#> 16  -1.487572e-01
#> 17   1.704648e-01
#> 18   1.364113e+00
#> 19  -7.736504e-01
#> 20   5.630360e-01
#> 21   9.424812e-01
#> 22   4.170600e-01
#> 23  -1.264042e+00
#> 24   4.178347e-01
#> 25   4.904022e-01
#> 26  -1.515887e-02
#> 27  -3.178319e-01
#> 28   3.805760e-01
#> 29  -6.799174e-01
#> 30   4.990594e-01
#> 31   2.695631e-01
#> 32  -3.121658e-02
#> 33   2.683149e-01
#> 34   1.491150e-01
#> 35   3.924688e-01
#> 36   5.472297e-02
#> 37  -2.712528e-01
#> 38  -2.177595e-01
#> 39   2.072179e-01
#> 40   4.547053e-01
#> 41   2.547398e-01
#> 42  -5.170840e-01
#> 43   1.133979e+00
#> 44   5.092196e-01
#> 45  -8.491259e-02
#> 46   7.063772e-01
#> 47   1.349054e+00
#> 48   4.962941e-01
#> 49   3.511585e-01
#> 50  -1.220144e+00
#> 51  -3.448379e-01
#> 52  -5.184845e-01
#> 53   6.179962e-01
#> 54  -5.934908e-01
#> 55   7.869669e-02
#> 56  -2.914607e-01
#> 57   5.575441e-01
#> 58  -1.067051e-01
#> 59   1.695508e-01
#> 60   3.595042e-01
#> 61  -2.846456e-01
#> 62   3.445642e-01
#> 63  -6.736214e-01
#> 64   1.532072e+00
#> 65   1.427288e-01
#> 66  -3.148159e-01
#> 67   8.832298e-01
#> 68  -1.830318e-01
#> 69   4.032647e-01
#> 70   2.409381e-01
#> 71   1.233375e+00
#> 72  -2.051791e+00
#> 73  -4.717118e-01
#> 74  -4.405299e-01
#> 75  -2.080830e-01
#> 76  -1.506797e-01
#> 77   3.097923e-01
#> 78  -8.702237e-02
#> 79   3.604256e-01
#> 80  -8.521368e-02
#> 81   7.887639e-01
#> 82  -7.293849e-01
#> 83  -1.639307e-02
#> 84  -4.691277e-01
#> 85   9.325053e-03
#> 86   6.400077e-01
#> 87  -1.402221e+00
#> 88   1.201843e-01
#> 89   5.378503e-01
#> 90  -1.762344e-01
#> 91   1.318162e-01
#> 92  -1.143514e+00
#> 93   4.786437e-01
#> 94   6.429156e-01
#> 95  -1.772527e-02
#> 96   3.687938e-01
#> 97  -1.066664e+00
#> 98  -8.906438e-01
#> 99   5.419909e-02
#> 100  1.126928e+00
#> 101 -8.616428e-01
#> 102 -4.405730e-01
#> 103 -9.865894e-01
#> 104 -1.041152e+00
#> 105  6.753898e-01
#> 106  1.474734e-01
#> 107  1.111835e-01
#> 108  1.681135e-01
#> 109 -7.067925e-02
#> 110 -9.789653e-01
#> 111 -9.668787e-01
#> 112  3.390004e-01
#> 113 -5.472008e-01
#> 114 -3.418560e-01
#> 115 -4.138748e-01
#> 116 -6.507368e-02
#> 117  2.593238e-01
#> 118  4.881388e-01
#> 119  5.347332e-01
#> 120  1.084537e+00
#> 121 -1.291370e+00
#> 122 -7.893439e-01
#> 123  2.287629e-01
#> 124 -2.975810e-01
#> 125 -9.810067e-01
#> 126  7.173674e-01
#> 127  1.373830e+00
#> 128  1.012382e+00
#> 129 -2.565706e-01
#> 130 -1.647907e-01
#> 131 -6.508092e-02
#> 132  5.175271e-01
#> 133  2.800798e-01
#> 134 -1.044942e+00
#> 135  1.260690e-01
#> 136  6.445337e-01
#> 137  1.646906e-01
#> 138 -6.823872e-01
#> 139 -6.108207e-01
#> 140 -3.258216e-01
#> 141 -4.868820e-01
#> 142  5.149480e-02
#> 143  5.364877e-01
#> 144  5.671157e-01
#> 145 -7.546692e-01
#> 146  1.110852e-01
#> 147 -6.772738e-01
#> 148  1.336448e+00
#> 149 -3.241622e-01
#> 150  6.122240e-01
#> 151 -8.587254e-01
#> 152  4.453814e-01
#> 153  9.925005e-01
#> 154 -2.654850e-01
#> 155  4.487299e-01
#> 156  4.818324e-01
#> 157  8.746962e-01
#> 158  5.247776e-02
#> 159  9.903067e-01
#> 160 -7.137321e-01
#> 161  3.478403e-01
#> 162 -2.509245e-01
#> 163 -1.043022e+00
#> 164  3.484151e-02
#> 165 -1.089838e-01
#> 166  9.315078e-01
#> 167  4.570214e-01
#> 168  6.675017e-01
#> 169 -2.731210e-01
#> 170 -5.864457e-01
#> 171  4.800880e-02
#> 172 -9.480365e-01
#> 173  7.188479e-01
#> 174  4.299004e-01
#> 175  7.732633e-01
#> 176  5.982326e-02
#> 177 -1.878878e+00
#> 178  8.313797e-02
#> 179  7.412046e-01
#> 180  1.244863e-01
#> 181 -4.439316e-02
#> 182  4.082467e-01
#> 183 -8.122601e-01
#> 184  3.380700e-01
#> 185  6.184746e-01
#> 186 -4.462906e-02
#> 187 -1.780221e-01
#> 188  7.422867e-01
#> 189  8.504415e-01
#> 190 -1.447240e+00
#> 191 -5.486541e-01
#> 192 -6.742420e-01
#> 193 -1.743576e-01
#> 194 -3.833636e-01
#> 195  1.066416e+00
#> 196 -4.101028e-01
#> 197 -5.133464e-01
#> 198  3.714431e-01
#> 199 -6.904036e-01
#> 200 -2.796176e-01
#> 
#> $res.l$P_Value
#>     A_vs_B_pval
#> 1   0.000000000
#> 2   0.000000000
#> 3   0.000000000
#> 4   0.000000000
#> 5   0.000000000
#> 6   0.000000000
#> 7   0.000000000
#> 8   0.000000000
#> 9   0.000000000
#> 10  0.000000000
#> 11  0.796941098
#> 12  0.949367396
#> 13  0.328845493
#> 14  0.435896381
#> 15  0.170804559
#> 16  0.815686143
#> 17  0.789383602
#> 18  0.032678839
#> 19  0.225507500
#> 20  0.377692370
#> 21  0.139852178
#> 22  0.513447430
#> 23  0.047758669
#> 24  0.512665570
#> 25  0.442263162
#> 26  0.981049169
#> 27  0.618473149
#> 28  0.550961487
#> 29  0.286774215
#> 30  0.434250458
#> 31  0.672742627
#> 32  0.960986570
#> 33  0.674170532
#> 34  0.815250906
#> 35  0.538586073
#> 36  0.931665593
#> 37  0.670811526
#> 38  0.732939800
#> 39  0.745408176
#> 40  0.476182828
#> 41  0.689774526
#> 42  0.417839744
#> 43  0.075726038
#> 44  0.424954526
#> 45  0.894149436
#> 46  0.268439313
#> 47  0.034647029
#> 48  0.436800667
#> 49  0.582161204
#> 50  0.056024524
#> 51  0.588971036
#> 52  0.416580176
#> 53  0.332923108
#> 54  0.352443711
#> 55  0.901857291
#> 56  0.647888970
#> 57  0.382361413
#> 58  0.867209930
#> 59  0.790486401
#> 60  0.573226392
#> 61  0.655583701
#> 62  0.589266718
#> 63  0.291258749
#> 64  0.016459648
#> 65  0.823029024
#> 66  0.621807695
#> 67  0.166485512
#> 68  0.774263458
#> 69  0.527474211
#> 70  0.705775614
#> 71  0.053416061
#> 72  0.001327678
#> 73  0.459847684
#> 74  0.490038170
#> 75  0.744382370
#> 76  0.813347884
#> 77  0.627379172
#> 78  0.891535505
#> 79  0.572243878
#> 80  0.893776333
#> 81  0.216578083
#> 82  0.253165324
#> 83  0.979506563
#> 84  0.462309373
#> 85  0.988341624
#> 86  0.315996655
#> 87  0.028121277
#> 88  0.850622027
#> 89  0.399393706
#> 90  0.782431621
#> 91  0.836360270
#> 92  0.073299806
#> 93  0.453280658
#> 94  0.313803559
#> 95  0.977841563
#> 96  0.563358277
#> 97  0.094779581
#> 98  0.162957862
#> 99  0.932318203
#> 100 0.077562329
#> 101 0.177083821
#> 102 0.489995750
#> 103 0.122253377
#> 104 0.102936427
#> 105 0.289994394
#> 106 0.817248618
#> 107 0.861691774
#> 108 0.792221635
#> 109 0.911812677
#> 110 0.125164854
#> 111 0.129891010
#> 112 0.595292753
#> 113 0.391250628
#> 114 0.592196448
#> 115 0.516668791
#> 116 0.918781532
#> 117 0.684490200
#> 118 0.444371933
#> 119 0.402130788
#> 120 0.089380260
#> 121 0.043150430
#> 122 0.216240509
#> 123 0.720000187
#> 124 0.641010594
#> 125 0.124380027
#> 126 0.261066023
#> 127 0.031460293
#> 128 0.112795260
#> 129 0.687662205
#> 130 0.796236910
#> 131 0.918772527
#> 132 0.417441001
#> 133 0.660759379
#> 134 0.101690383
#> 135 0.843400388
#> 136 0.312587589
#> 137 0.796357915
#> 138 0.285027866
#> 139 0.338565258
#> 140 0.609677681
#> 141 0.445545325
#> 142 0.935687696
#> 143 0.400588765
#> 144 0.374246784
#> 145 0.237090770
#> 146 0.861812910
#> 147 0.288651505
#> 148 0.036371683
#> 149 0.611499869
#> 150 0.337457060
#> 151 0.178553900
#> 152 0.485271972
#> 153 0.120032717
#> 154 0.677412026
#> 155 0.481996940
#> 156 0.450277631
#> 157 0.170616625
#> 158 0.934462815
#> 159 0.120853136
#> 160 0.263489240
#> 161 0.585731624
#> 162 0.694184376
#> 163 0.102320308
#> 164 0.956460519
#> 165 0.864401320
#> 166 0.144519990
#> 167 0.473939550
#> 168 0.295662704
#> 169 0.668679110
#> 170 0.358187287
#> 171 0.940032877
#> 172 0.137533834
#> 173 0.260083586
#> 174 0.500568547
#> 175 0.225739623
#> 176 0.925314553
#> 177 0.003280237
#> 178 0.896349025
#> 179 0.245558770
#> 180 0.845341413
#> 181 0.944541547
#> 182 0.522386143
#> 183 0.203206394
#> 184 0.596303267
#> 185 0.332549080
#> 186 0.944247318
#> 187 0.780280965
#> 188 0.244870442
#> 189 0.182777701
#> 190 0.023453268
#> 191 0.389994048
#> 192 0.290814582
#> 193 0.784691114
#> 194 0.548048245
#> 195 0.094856436
#> 196 0.520497055
#> 197 0.421212303
#> 198 0.560559032
#> 199 0.279409182
#> 200 0.661284332
#> 
#> 
#> $fit.s2
#>   [1] 1.017841 1.017841 1.017841
#>   [4] 1.017841 1.017841 1.017841
#>   [7] 1.017841 1.017841 1.017841
#>  [10] 1.017841 1.017841 1.017841
#>  [13] 1.017841 1.017841 1.017841
#>  [16] 1.017841 1.017841 1.017841
#>  [19] 1.017841 1.017841 1.017841
#>  [22] 1.017841 1.017841 1.017841
#>  [25] 1.017841 1.017841 1.017841
#>  [28] 1.017841 1.017841 1.017841
#>  [31] 1.017841 1.017841 1.017841
#>  [34] 1.017841 1.017841 1.017841
#>  [37] 1.017841 1.017841 1.017841
#>  [40] 1.017841 1.017841 1.017841
#>  [43] 1.017841 1.017841 1.017841
#>  [46] 1.017841 1.017841 1.017841
#>  [49] 1.017841 1.017841 1.017841
#>  [52] 1.017841 1.017841 1.017841
#>  [55] 1.017841 1.017841 1.017841
#>  [58] 1.017841 1.017841 1.017841
#>  [61] 1.017841 1.017841 1.017841
#>  [64] 1.017841 1.017841 1.017841
#>  [67] 1.017841 1.017841 1.017841
#>  [70] 1.017841 1.017841 1.017841
#>  [73] 1.017841 1.017841 1.017841
#>  [76] 1.017841 1.017841 1.017841
#>  [79] 1.017841 1.017841 1.017841
#>  [82] 1.017841 1.017841 1.017841
#>  [85] 1.017841 1.017841 1.017841
#>  [88] 1.017841 1.017841 1.017841
#>  [91] 1.017841 1.017841 1.017841
#>  [94] 1.017841 1.017841 1.017841
#>  [97] 1.017841 1.017841 1.017841
#> [100] 1.017841 1.017841 1.017841
#> [103] 1.017841 1.017841 1.017841
#> [106] 1.017841 1.017841 1.017841
#> [109] 1.017841 1.017841 1.017841
#> [112] 1.017841 1.017841 1.017841
#> [115] 1.017841 1.017841 1.017841
#> [118] 1.017841 1.017841 1.017841
#> [121] 1.017841 1.017841 1.017841
#> [124] 1.017841 1.017841 1.017841
#> [127] 1.017841 1.017841 1.017841
#> [130] 1.017841 1.017841 1.017841
#> [133] 1.017841 1.017841 1.017841
#> [136] 1.017841 1.017841 1.017841
#> [139] 1.017841 1.017841 1.017841
#> [142] 1.017841 1.017841 1.017841
#> [145] 1.017841 1.017841 1.017841
#> [148] 1.017841 1.017841 1.017841
#> [151] 1.017841 1.017841 1.017841
#> [154] 1.017841 1.017841 1.017841
#> [157] 1.017841 1.017841 1.017841
#> [160] 1.017841 1.017841 1.017841
#> [163] 1.017841 1.017841 1.017841
#> [166] 1.017841 1.017841 1.017841
#> [169] 1.017841 1.017841 1.017841
#> [172] 1.017841 1.017841 1.017841
#> [175] 1.017841 1.017841 1.017841
#> [178] 1.017841 1.017841 1.017841
#> [181] 1.017841 1.017841 1.017841
#> [184] 1.017841 1.017841 1.017841
#> [187] 1.017841 1.017841 1.017841
#> [190] 1.017841 1.017841 1.017841
#> [193] 1.017841 1.017841 1.017841
#> [196] 1.017841 1.017841 1.017841
#> [199] 1.017841 1.017841
```

Simulate a list of 100 datasets.

```r
set.seed(4619)
norm.200.m100.sd1.vs.m200.sd1.list <- lapply(1:100, protdatasim)
metadata <- attr(norm.200.m100.sd1.vs.m200.sd1.list[[1]],"metadata")
```

It is the list of dataset shipped with package.

```r
save(norm.200.m100.sd1.vs.m200.sd1.list, file="norm.200.m100.sd1.vs.m200.sd1.list.RData", compress = "xz")
```

100 datasets with parallel comuting support. Quite long to run even with parallel computing support.

```r
library(foreach)
doParallel::registerDoParallel(cores=NULL)
requireNamespace("foreach",quietly = TRUE)
```

## AMPUTATION

```r
MV1pct.NA.data <- foreach::foreach(iforeach =  norm.200.m100.sd1.vs.m200.sd1.list,
                          .errorhandling = 'stop', .verbose = T) %dopar% 
  MVgen(dataset = iforeach[,-1], prop_NA = 0.01)
```

## IMPUTATION

```r
MV1pct.impMLE <- foreach::foreach(iforeach =  MV1pct.NA.data,
                         .errorhandling = 'stop', .verbose = F) %dopar% 
  multi.impute(data = iforeach, conditions = metadata$Condition, 
               method = "MLE", parallel  = F)
```

## ESTIMATION

```r
MV1pct.impMLE.VarRubin.Mat <- lapply(1:length(MV1pct.impMLE), function(index){
  print(paste(Sys.time(), "Dataset", index, "out of", length(MV1pct.impMLE)))
  rubin2.all(data = MV1pct.impMLE[[index]], metacond = metadata$Condition) 
})
```

## PROJECTION

```r
MV1pct.impMLE.VarRubin.S2 <- lapply(1:length(MV1pct.impMLE.VarRubin.Mat), function(id.dataset){
  print(paste("Dataset", id.dataset, "out of",length(MV1pct.impMLE.VarRubin.Mat), Sys.time()))
  as.numeric(lapply(MV1pct.impMLE.VarRubin.Mat[[id.dataset]], function(aaa){
    DesMat = DAPAR::make.design(metadata)
    return(max(diag(aaa)%*%t(DesMat)%*%DesMat))
  }))
})
```

## MODERATED T-TEST

```r
MV1pct.impMLE.mi4limma.res <- foreach(iforeach =  1:100,  .errorhandling = 'stop', .verbose = T) %dopar%
  mi4limma(qData = apply(MV1pct.impMLE[[iforeach]],1:2,mean), 
                 sTab = metadata, 
                 VarRubin = sqrt(MV1pct.impMLE.VarRubin.S2[[iforeach]]))

MV1pct.impMLE.dapar.res <- foreach(iforeach =  1:100,  .errorhandling = 'stop', .verbose = T) %dopar%
  limmaCompleteTest.mod(qData = apply(MV1pct.impMLE[[iforeach]],1:2,mean), 
                        sTab = metadata)
```

