\name{nipals}
\encoding{latin1}
\alias{nipals}

\title{Non-linear Iterative Partial Least Squares (NIPALS) algorithm}

\description{
This function performs NIPALS algorithm, i.e. the singular-value decomposition (SVD)
of a data table that can contain missing values.
}

\usage{
nipals(X, ncomp = 1, reconst = FALSE, max.iter = 500, tol = 1e-09)
}

\arguments{
  \item{X}{real matrix or data frame whose SVD decomposition is to be computed.
    It can contain missing values.}
  \item{ncomp}{integer, the number of components to keep. If missing \code{ncomp=ncol(X)}.}
  \item{reconst}{logical that specify if \code{nipals} must perform the 
	reconstitution of the data using the \code{ncomp} components.}
  \item{max.iter}{integer, the maximum number of iterations.}
  \item{tol}{a positive real, the tolerance used in the iterative algorithm.}
}

\details{
The NIPALS algorithm (Non-linear Iterative Partial Least Squares) 
has been developed by H. Wold at first for PCA and later-on for PLS. 
It is the most commonly used method for calculating the principal 
components of a data set. It gives more numerically accurate results 
when compared with the SVD of the covariance matrix, but is slower to calculate.

This algorithm allows to realize SVD with missing data, without having to 
delete the rows with missing data or to estimate the missing data.
}

\value{
The returned value is a list with components:
  \item{eig}{vector containing the pseudosingular values of \code{X}, of length \code{ncomp}.}
  \item{t}{matrix whose columns contain the left singular vectors of \code{X}.}
  \item{p}{matrix whose columns contain the right singular vectors of \code{X}.}
  \item{rec}{matrix obtained by the reconstitution of the data using the \code{ncomp} components.}
}

\references{
Tenenhaus, M. (1998). \emph{La rgression PLS: thorie et pratique}. Paris: Editions Technic.

Wold H. (1966). Estimation of principal components and related models by iterative least squares. 
In: Krishnaiah, P. R. (editors), \emph{Multivariate Analysis}. Academic Press, N.Y., 391-420.

Wold H. (1975). Path models with latent variables: The NIPALS approach. 
In: Blalock H. M. et al. (editors). \emph{Quantitative Sociology: International perspectives 
on mathematical and statistical model building}. Academic Press, N.Y., 307-357.
}

\author{Sbastien Djean and Ignacio Gonzlez.}

\seealso{\code{\link{svd}}, \code{\link{princomp}}, \code{\link{prcomp}}, \code{\link{eigen}}}

\examples{
## Hilbert matrix
hilbert <- function(n) { i <- 1:n; 1 / outer(i - 1, i, "+") }
X.na <- X <- hilbert(9)[, 1:6]

## Hilbert matrix with missing data
idx.na <- matrix(sample(c(0, 1, 1, 1, 1), 36, replace = TRUE), ncol = 6)
X.na[idx.na == 0] <- NA
X.rec <- nipals(X.na, reconst = TRUE)$rec
round(X, 2)
round(X.rec, 2)
}

\keyword{algebra}
\keyword{multivariate}