% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_contrasts.R
\name{estimate_contrasts}
\alias{estimate_contrasts}
\title{Estimate Marginal Contrasts}
\usage{
estimate_contrasts(
  model,
  levels = NULL,
  fixed = NULL,
  modulate = NULL,
  transform = "none",
  ci = 0.95,
  adjust = "holm",
  ...
)
}
\arguments{
\item{model}{A statistical model.}

\item{levels}{A character vector or formula specifying the names of the
predicting factors over which to estimate means or contrasts.}

\item{fixed}{A character vector indicating the names of the predictors to be
"fixed" (i.e., maintained), so that the estimation is made at these values.}

\item{modulate}{A character vector indicating the names of a numeric variable
along which the means or the contrasts will be estimated. Other arguments
from \code{\link{visualisation_matrix}}, such as \code{length} to adjust the
number of data points.}

\item{transform}{Is passed to the \code{type} argument in \code{emmeans::emmeans()}. See \href{https://CRAN.R-project.org/package=emmeans/vignettes/transformations.html}{this vignette}. Can be \code{"none"} (default for contrasts),
\code{"response"} (default for means), \code{"mu"}, \code{"unlink"},
\code{"log"}. \code{"none"} will leave the values on scale of the linear
predictors. \code{"response"} will transform them on scale of the response
variable. Thus for a logistic model, \code{"none"} will give estimations
expressed in log-odds (probabilities on logit scale) and \code{"response"}
in terms of probabilities.}

\item{ci}{Uncertainty Interval (CI) level. Default to 95\% (\code{0.95}).}

\item{adjust}{The p-values adjustment method for frequentist multiple comparisons. Can be
one of "holm" (default), "tukey", "hochberg", "hommel", "bonferroni", "BH",
"BY", "fdr" or "none". See the p-value adjustment section in the
\code{emmeans::test} documentation.}

\item{...}{Other arguments passed for instance to \code{\link{visualisation_matrix}}.}
}
\value{
A data frame of estimated contrasts.
}
\description{
Run a contrast analysis by estimating the differences between each level of a factor. See also other
related functions such as \code{\link{estimate_means}} and \code{\link{estimate_slopes}}.
}
\examples{
library(modelbased)

# Basic usage
model <- lm(Sepal.Width ~ Species, data = iris)
estimate_contrasts(model)

# Dealing with interactions
model <- lm(Sepal.Width ~ Species * Petal.Width, data = iris)
estimate_contrasts(model)
estimate_contrasts(model, fixed = "Petal.Width")
estimate_contrasts(model, modulate = "Petal.Width", length = 4)
estimate_contrasts(model, levels = "Petal.Width", length = 4)

# Standardized differences
estimated <- estimate_contrasts(lm(Sepal.Width ~ Species, data = iris))
effectsize::standardize(estimated)

# Other models (mixed, Bayesian, ...)
if (require("lme4")) {
  data <- iris
  data$Petal.Length_factor <- ifelse(data$Petal.Length < 4.2, "A", "B")

  model <- lmer(Sepal.Width ~ Species + (1 | Petal.Length_factor), data = data)
  estimate_contrasts(model)
}

data <- mtcars
data$cyl <- as.factor(data$cyl)
data$am <- as.factor(data$am)
\dontrun{
if (require("rstanarm")) {
  model <- stan_glm(mpg ~ cyl * am, data = data, refresh = 0)
  estimate_contrasts(model)
  estimate_contrasts(model, fixed = "am")

  model <- stan_glm(mpg ~ cyl * wt, data = data, refresh = 0)
  estimate_contrasts(model)
  estimate_contrasts(model, fixed = "wt")
  estimate_contrasts(model, modulate = "wt", length = 4)
  estimate_contrasts(model, levels = "wt", length = 4)

  model <- stan_glm(Sepal.Width ~ Species + Petal.Width + Petal.Length, data = iris, refresh = 0)
  estimate_contrasts(model, fixed = "Petal.Width", modulate = "Petal.Length", test = "bf")
}

if (require("brms")) {
  model <- brm(mpg ~ cyl * am, data = data, refresh = 0)
  estimate_contrasts(model)
}
}

}
