#' Read mesh data from ply files saved by AVIZO
#'
#' A function that reads Stanford ply files as saved by the 3D data
#' visualization software Avizo
#' 
#' @param file An ASCII PLY file generated by Avizo
#' @param ShowSpecimen Logical indicating whether or not the
#' mesh should be displayed
#' @param addNormals Logical indicating whether or not normals of
#' mesh vertices shold be calculated and appended to object
#' 
#' @details If ShowSpecimen is True, a gray shade3d of the mesh is generated in
#' a new rgl window for previewing the specimen. When saving to the ply file type,
#' Avizo inserts additional property parameters into the file heading that
#' sometimes describe various components of the mesh. These additional properties
#' cause the read.ply function native to the geomorph package to fail. This function
#' properly reads ply files generated by Avizo (like read.ply) and can be stored as an 
#' object accepted as input in the other molaR functions. Ply files generated through
#' other software (such as MeshLab) can be read using read.ply. 
#' 
#' @export
#' read.AVIZO.ply()

read.AVIZO.ply <- function (file, ShowSpecimen = TRUE, addNormals = TRUE) 
{
    plyfile <- scan(file = file, what = "char", sep = "\n", strip.white = TRUE, 
        quiet = TRUE)
    is.ply <- grep("ply", plyfile)
    if ((length(is.ply) == 0)) 
        stop("File is not a PLY file")
    format <- unlist(strsplit(grep(c("format "), plyfile, value = TRUE), 
        " "))
    if (format[2] != "ascii") 
        stop("PLY file is not ASCII format: ", "format = ", format[2:length(format)])
    poly <- NULL
    material <- NULL
    xline <- unlist(strsplit(grep(c("vertex "), plyfile, value = TRUE), 
        " "))
    npoints <- as.numeric(xline[grep(c("vertex"), xline) + 1])
    yline <- unlist(strsplit(grep(c("element face"), plyfile, 
        value = TRUE), " "))
    npoly <- as.numeric(yline[grep(c("face"), yline) + 1])
    headerend <- grep(c("end_header"), plyfile)
    ncolpts <- (length(grep(c("property"), plyfile)) - 7)
    cols <- grep(c("property"), plyfile, value = TRUE)
    x <- grep(c(" x"), cols)
    y <- grep(c(" y"), cols)
    z <- grep(c(" z"), cols)
    points <- as.matrix(as.numeric(unlist(strsplit(plyfile[(headerend + 
        1):(headerend + npoints)], " "))))
    dim(points) <- c(ncolpts, npoints)
    xpts <- points[x, ]
    ypts <- points[y, ]
    zpts <- points[z, ]
    vertices <- rbind(xpts, ypts, zpts, 1)
    if (yline[3] == 0) 
        print("Object has zero faces")
    if (yline[3] != 0) {
        poly <- as.matrix(as.numeric(unlist(strsplit(plyfile[(headerend + 
            npoints + 1):(headerend + npoints + npoly)], " "))))
        dim(poly) <- c((poly[1] + 2), npoly)
        poly <- poly[-1, ]
        poly <- poly[-4, ]
        poly = poly + 1
    }
    colinfo <- grep("property uchar red", plyfile)
    if (length(colinfo) != 0) {
        color <- rgb(points[4, ], points[5, ], points[6, ], maxColorValue = 255)
        material$color <- matrix(color[poly], dim(poly))
    }
    mesh <- list(vb = vertices, it = poly, primitivetype = "triangle", 
        material = material)
    class(mesh) <- c("mesh3d", "shape3d")
    if (addNormals == TRUE) {
        mesh <- addNormals(mesh)
    }
    if (ShowSpecimen == TRUE) {
        clear3d()
        if (length(poly) == 0) {
            dot3d(mesh)
        }
        if (length(material) != 0) {
            shade3d(mesh)
        }
        shade3d(mesh, color = "gray")
    }
    return(mesh)
}

