% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/animate_stats.R
\name{animate_stats}
\alias{animate_stats}
\title{Animate movement data statistics}
\usage{
animate_stats(data_ani, out_dir, conv_dir = "convert", layer = "basemap",
  layer_dt = "basemap", layer_int = FALSE, layer_type = "",
  val_limits = NA, paths_col = "auto", paths_mode = "true_data",
  stats_type = "", stats_gg = "", stats_digits = 1, stats_tframe = 5,
  stats_lay = 0, stats_title = "", frames_nmax = 0,
  frames_interval = 0.04, frames_nres = 1, frames_width = 600,
  frames_height = 600, out_name = "final_gif", log_level = 1,
  log_logical = FALSE)
}
\arguments{
\item{data_ani}{list or \code{moveStack} class object. Needs to contain one or several \code{move} class objects (one for each individual path to be displayed) containing point coordinates, timestamps and projection.}

\item{out_dir}{character. Output directory for the GIF file creation.}

\item{conv_dir}{character. Command or directory to call the ImageMagick convert tool (default to be \code{convert}). You can use \code{conv_dir = get_imconvert()} to search for the right command/tool directory and/or get the required software.}

\item{layer}{raster, list or character. Single raster object or list of raster objects to be used as (dynamically changing) basemap layer. Default is \code{"basemap"} to download a static basemap layer. Use a rasterBrick class object and set layer_type to "\code{RGB}" to compute a RGB basemap.}

\item{layer_dt}{POSIXct or list. Single POSIXct date/time stamp or list of POSIXct date/time stamps corresponding to the acquisition dates of the \code{layer} raster objects.}

\item{layer_int}{logical. Whether to interpolate the basemap layer objects over time, if several are provided (\code{TRUE}), or to display them one after another depending on the animation time frame that is displayed (\code{FALSE}). Default is \code{FALSE}.}

\item{layer_type}{charachter. Layer type. Can be either "\code{RGB}" (if layer is a rasterBrick class onejct), "\code{gradient}" or "\code{discrete}". Default is "\code{RGB}". Ignored, if \code{layer = "basemap"}.}

\item{val_limits}{numeric vector. Fixed minimum and maximum limit values of the basemap value range (gradient layer type). Default is NA for data-depending minimum and maximum values. Ignored, if \code{layer_type} is "discrete" or "RGB".}

\item{paths_col}{character vector. Colours of the individual animation paths. If set to "auto", a predfined colour set will be used. If single colour, all paths will be displayed by the same colour. If more individuals then colours, the colours are repeated.}

\item{paths_mode}{character vector. Mode to be used for dealing with time information when displaying multiple individual paths. See \code{\link{animate_move}} for details. Default is "true_data".}

\item{stats_type}{character. Defines which standard plot design should be used. Select either "line" or "bar". Ignored, if \code{stats_gg} is used.}

\item{stats_gg}{character. Enables usage of \code{ggplot2} syntax for plot design. If set, \code{stats_type} is ignored. See \code{details} for information on the statistic data structure to be used by the user defined plot function.}

\item{stats_digits}{numeric. Defines how detailed the statistic plot should be as number of decimals. Values with more decimals are rounded. Default is 1 for one decimal.}

\item{stats_tframe}{numeric. Defines the temporal range of the periodic stats plot. Default is 5 meaning that five time frames back from the displayed frame are evaluated.}

\item{stats_lay}{matrix. Optional layout. Based on the \code{\link{grid.arrange}} syntax.}

\item{stats_title}{character vector. Optional plot titles. Two character strings within a vector.}

\item{frames_nmax}{numeric. Number of maximum frames. If set, the animation will be stopped, after the specified number of frames is reached. Default is 0 (displaying all frames).}

\item{frames_interval}{numeric. Duration, each frame is displayed (in seconds). Default is .04.}

\item{frames_nres}{numeric. Interval of which frames of all frames should be used (nth elements). Default is 1 (every frame is used). If set to 2, only every second frame is used.}

\item{frames_width}{numeric. Number of pixels of frame width. Default is 600.}

\item{frames_height}{numeric. Number of pixels of frame height. Defualt is 600.}

\item{out_name}{character. Name of the output file. Default is "final_gif".}

\item{log_level}{numeric. Level of console output given by the function. There are three log levels. If set to 3, no messages will be displayed except erros that caused an abortion of the process. If set to 2, warnings and errors will be displayed. If set to 1, a log showing the process activity, wanrnings ans errors will be displayed.}

\item{log_logical}{logical. For large processing schemes. If TRUE, the function returns TRUE when finished processing succesfully.}

\item{...}{optional arguments.}
}
\value{
None or logical (see \code{log_logical}. The output GIF file is written to the ouput directory.
}
\description{
\code{animate_stats} animates statistic plot from movement data provided as \code{move} class objects or a list of them and basemap data provided as \code{raster}. It extracts basemap values of pixels that are part of the movement paths and visualizes frequencies per value. The function creates an animated GIF file and saves it into the output directory. See also \code{\link{animate_move}}.
}
\details{
\code{animate_stats} is a wrapper function of \code{\link{animate_move}} to create single statistic plots without spatial plotting. For statistic plot animations sidy-by-side with spatial plot animations, use \code{\link{animate_move}} (see \code{stats_create} argument). The function can handle all arguments taken by \code{animate_stats} as well.
Use \code{stats_gg} to provide an own ggplot2 plot design as shown in the examples. The statistics are stored for both plots (periodic and accumulated) withn the variable \code{pdat} (list of two, indexed by k ranging from 1 to 2 for each plot). Both \code{pdat} lists contain the stats elements framewise for each time step. For this, see the \code{stats_gg} example. The variable \code{cols} (list of two, one per plot) contains the defined colour values and namings.
}
\examples{
#Load move and moveVis packages
library(move)
library(moveVis)

#Get the sample data from the moveVis package
data("move_data")
move_data$dt <- as.POSIXct(strptime(move_data$dt, "\%Y-\%m-\%d \%H:\%M:\%S", tz = "UTC"))

#Differentiate data per individual
indi_levels <- levels(move_data$individual)
indi_levels_n <- length(indi_levels)
for(i in 1:indi_levels_n){
  if(i == 1){
    indi_subset <- list(subset(move_data, individual == indi_levels[i]))
  }else{
    indi_subset <- c(indi_subset,list(subset(move_data,
                               individual == indi_levels[i])))
  }
}
indi_names <- paste(indi_levels, collapse = ", ")

#Create move class object
for(i in 1:length(indi_subset)){
  if(i == 1){
     data_ani <- list(move(x=indi_subset[[i]]$lon,y=indi_subset[[i]]$lat,
                                 time=indi_subset[[i]]$dt,
                                 proj=CRS("+proj=longlat +ellps=WGS84"),
                                 animal=indi_levels[i]))
  }else{
     data_ani[i] <- list(move(x=indi_subset[[i]]$lon,y=indi_subset[[i]]$lat,
                                 time=indi_subset[[i]]$dt,
                                 proj=CRS("+proj=longlat +ellps=WGS84"),
                                 animal=indi_levels[i]))}
}

#Load basemap data (MODIS NDVI)
data("basemap_data")
layer = basemap_data[[1]]
layer_dt = basemap_data[[2]]

#Find command or directory to convert tool of ImageMagick
conv_dir <- get_imconvert()

#Specify output directory
out_dir <- "/out/test"

#Call animate_stats()
animate_stats(data_ani, out_dir, conv_dir = conv_dir,
              layer=layer, layer_dt = layer_dt, layer_type = "gradient",
              stats_digits = 1, stats_type = "bar", out_name = "final_gif",
              log_level = 1,frames_nmax = 60)
              
#Define your own ggplot2 plot design
stats_gg <- 'ggplot(data = pdat[[k]][[i]], aes_(x = ~val, y = ~value, colour = ~variable)) + 
             geom_smooth() + geom_point() + theme_bw() + theme(aspect.ratio=1) +
             scale_y_continuous(expand = c(0,0),limits = c(0,stats_max[k])) +
             scale_x_continuous(expand = c(0,0)) + 
             scale_color_manual(name="",values = cols[[k]]) +
             labs(x = "Basemap Value", y="Frequency",
                  title=stats_title[[k]], label=c("123","456")) +
             theme(plot.title = element_text(hjust = 0.5),
                   plot.subtitle = element_text(hjust = 0.5))'
                  
#Call animate_stats() with stats_gg
animate_stats(data_ani, out_dir, conv_dir = conv_dir,
              layer=layer, layer_dt = layer_dt, layer_type = "gradient",
              stats_digits = 1, stats_gg = stats_gg, out_name = "final_gif",
              log_level = 1,frames_nmax = 60)

}
\seealso{
\code{\link{get_imconvert}}
}
\author{
Jakob Schwalb-Willmann
}
