#'  Detection functions
#' 
#'  Various functions used to specify key and adjustment functions for detection functions
#' 
#' 	Multi-covariate detection functions (MCDS) are represented by a function \eqn{g(x,w,\theta)} where x is distance,
#'	z is a set of covariates and \eqn{\theta} is the parameter vector.  The functions are defined such that \eqn{g(0,w,\theta)=1} and
#'	the covariates modify the scale \eqn{(x/\sigma)} where a log link is used to relate \eqn{\sigma} to the covariates,
#'	\eqn{\sigma=exp(\theta*w)}. A CDS function is obtained with a constant \eqn{\sigma} which is equivalent to an intercept 
#'	design matrix, z.
#'	\code{detfct} will call either a gamma, half-normal, hazard-rate or uniform function only returning the probability of detection at that distance.
#'	In addition to the simple model above, we may specify adjustment terms to fit the data better. These adjustments are either Cosine, Hermite and simple polynomials. 
#'	These are specified as arguments to \code{detfct}, as detailed below.   
#' 
#'		\code{detfct} meta function which calls the others and assembles the final result using either key(x)[1+series(x)] or (key(x)[1+series(x)])/(key(0)[1+series(0)]) (depending on the value of standardize)
#'		\code{keyfct.hn, keyfct.hz, keyfct.gamma} calculates half-normal or hazard-rate key function values 
#'		\code{adjfct.cos, adjfct.poly, adjfct.herm} calculates adjustment term values 
#'		\code{scalevalue} for either detection function it computes the scale with the log link using the parameters and the covariate design matrix
#'		\code{fx,fr} non-normalized probability density for line transects and point counts respectively
#' 
#' @aliases detfct adjfct.cos adjfct.herm hermite.poly adjfct.poly keyfct.hn keyfct.hz 
#'  keyfct.gamma apex.gamma scalevalue g0 fx fr 
#' @usage 	detfct(distance, ddfobj, select=NULL, index=NULL, width=NULL, standardize = TRUE, stdint=FALSE)
#' 
#'	adjfct.cos(distance, scaling = 1, adj.order, adj.parm = NULL)
#' 
#'	adjfct.poly(distance, scaling = 1, adj.order, adj.parm = NULL)
#' 
#'	adjfct.herm(distance, scaling = 1, adj.order, adj.parm = NULL)
#' 
#'	scalevalue(key.scale, z)
#' 
#'  keyfct.hn(distance, key.scale)
#' 
#'	keyfct.hz(distance, key.scale, key.shape)
#' 
#'	keyfct.gamma(distance, key.scale, key.shape)
#' 
#'	apex.gamma(key.scale, key.shape)
#' 
#'	fx(distance,ddfobj,select=NULL,index=NULL,width=NULL,standardize=TRUE,stdint=FALSE)
#' 
#'	fr(distance,ddfobj,select=NULL,index=NULL,width=NULL,standardize=TRUE,stdint=FALSE)
#' 
#' @param distance  vector of distances 
#' @param ddfobj distance sampling object (see \code{\link{create.ddfobj}}) 
#' @param z design matrix for scale function
#' @param select logical vector for selection of data values
#' @param index specific data row index
#' @param key.scale vector of scale values
#' @param key.shape vector of shape values
#' @param adj.order vector of adjustment orders
#' @param adj.parm vector of adjustment parameters
#' @param width truncation width
#' @param standardize logical used to decide whether to divide through by the function evaluated at 0 
#' @param scaling the scaling for the adjustment terms 
#' @param stdint logical used to decide whether integral is standardized
#' @return 
#'	For \code{detfct}, the value is a vector of detection probabilities for the input set of x and z.
#'	For \code{keyfct.hn, keyfct.hz}, vector of detection probability for that key function at x.
#'	For \code{adjfct.cos, adjfct.poly, adjfct.herm}, vector of the value of the adjustment term at x.
#'	For \code{scalevalue}, the value is a vector of the computed scales for the design matrix z. 
#'	For \code{apex.gamma}, the value is the distance at which the gamma peaks
#' @author Jeff Laake David Miller
#' @seealso  \code{\link{mcds}},  \code{\link{cds}}
#' @references 	Marques and Buckland 2004
#'	Laake and Borchers 2004. in Buckland et al 2004.
#'	Becker, E. F. and P. X. Quang. 2009. A gamma-shaped detection function for line transect surveys with mark-recapture and covariate data. Journal of Agricultural Biological and Environmental Statistics 14:207-223.
fx <- function(distance,ddfobj,select=NULL,index=NULL,width=NULL,standardize=TRUE,stdint=FALSE)
	return(detfct(distance,ddfobj,select,index,width,standardize,stdint)/width)
fr <- function(distance,ddfobj,select=NULL,index=NULL,width=NULL,standardize=TRUE,stdint=FALSE)
	return(detfct(distance,ddfobj,select,index,width,standardize,stdint)*2*distance/width^2)
detfct <-
function(distance,ddfobj,select=NULL,index=NULL,width=NULL,standardize=TRUE,stdint=FALSE)
{
#
# detfct - calls key and adjustment terms to create the detection function
#
# Arguments:
#	distance		- vector of distances 
#	ddfobj			- distance sampling object
#	select		    - selection of data values
#   index           - index of specific row
#	width			- truncation width
# 	standardize		- standardizes detection function so g(0)=1
# 	stdint			- standardizes the scale to 1 for integral computation
#
# Value	
#   the value of the detection function at the specified point(s)
#
# Uses : getpar, scalevalue, keyfct.hn, keyfct.hz, adjfct.poly, adjfct.herm, adjfct.cos
#
  if(is.null(select))
  {
	  if(is.null(index))
	  {
		  scale.dm=ddfobj$scale$dm
		  shape.dm=ddfobj$shape$dm
	  }
	  else
	  {
		  scale.dm=ddfobj$scale$dm[index,,drop=FALSE]
		  shape.dm=ddfobj$shape$dm[index,,drop=FALSE]
	  }
  } else
  {
	  if(is.null(index))
	  {
		  scale.dm=ddfobj$scale$dm[select,,drop=FALSE]
		  shape.dm=ddfobj$shape$dm[select,,drop=FALSE]
	  }
	  else
	  {
		  scale.dm=ddfobj$scale$dm[select,,drop=FALSE][index,,drop=FALSE]
		  shape.dm=ddfobj$shape$dm[select,,drop=FALSE][index,,drop=FALSE]
	  }
  }	  
  key=ddfobj$type
# calculate the key scale 
  if(stdint)
  {
	  if(is.null(index))
	    key.scale=rep(1,nrow(scale.dm))
      else
		key.scale=1  
  }else
  {
	if(!is.null(ddfobj$scale))
	  key.scale <- scalevalue(ddfobj$scale$parameters,scale.dm)
  }
# calculate the key shape
  if(!is.null(ddfobj$shape)) 
	  key.shape <- scalevalue(ddfobj$shape$parameters,shape.dm)
  if(key=="gamma")
  {	  
	  key.shape=key.shape+1
	  key.shape[key.shape==1]=key.shape[key.shape==1]+0.000001
  } 
# 19-Jan-06 jll; added proper standardize code to get std integral.
#  I left standardize code below in case it is needed for adjustment fcts
#
# Decide on keyfct.* and run.
  if(key == "hn"){
	key.vals <- keyfct.hn(distance, key.scale)
  }else if(key == "hr"){
	key.vals <- keyfct.hz(distance, key.scale, key.shape)
  }else if(key == "unif"){
	key.vals <- rep(1/width,length(distance))
  }else if(key == "gamma"){
	key.vals <- keyfct.gamma(distance, key.scale, key.shape)
  }

  # If we are using adjustment terms.
  if(!is.null(ddfobj$adjustment)){
    adj.series=ddfobj$adjustment$series
	 adj.scale=ddfobj$adjustment$scale
	 adj.order=ddfobj$adjustment$order
	 adj.parm=ddfobj$adjustment$parameters
		
    # Find out if we are scaling by width or by key scale
    if(adj.scale == "width")
      scaling <- width
    else
      scaling <- key.scale

    ## Decide on adjustment term and run.
    # If we have simple polynomials
    if(adj.series == "poly"){
      adj.vals <- adjfct.poly(distance,scaling,adj.order,adj.parm)

    # Hermite polynomials
    }else if(adj.series == "herm"){
      adj.vals <- adjfct.herm(distance,scaling,adj.order,adj.parm)

    # Cosine series
    }else if(adj.series == "cos"){
      adj.vals <- adjfct.cos(distance,scaling,adj.order,adj.parm)
    }

# If we have adjustment terms then we need to standardize the detection
# function. So find the values for the key and adjustment terms at 0   

# dlm 25-Aug-05  This causes a division by zero error in the optimization
#		 so lets only do it when we need to, it cancels in the
#		 likelihood anyway.
    if(standardize == TRUE){
      if(key == "hn"){
        key.val.0 <- keyfct.hn(rep(0,length(distance)), key.scale)
      }else if(key == "hr"){
        key.val.0 <- keyfct.hz(rep(0,length(distance)), key.scale, key.shape)
      }else if(key == "gamma"){
        key.val.0 <- keyfct.gamma(rep(0,length(distance)), key.scale, key.shape)
      }else if(key == "unif"){
        key.val.0 <- rep(1,length(distance))
      }

      if(adj.series == "poly"){
        adj.val.0 <- adjfct.poly(rep(0,length(distance)),scaling,adj.order,adj.parm)
      }else if(adj.series == "herm"){
        adj.val.0 <- adjfct.herm(rep(0,length(distance)),scaling,adj.order,adj.parm)
      }else if(adj.series == "cos"){
        adj.val.0 <- adjfct.cos(rep(0,length(distance)),scaling,adj.order,adj.parm)
      }

# Now return the standardized value of the detection function
      return((key.vals*(1+adj.vals))/(key.val.0*(1+adj.val.0)))

    }else
      return(key.vals*(1+adj.vals))

  }else{
# If we have no adjustment terms then just return the key value.
    return(key.vals)
  }
}

