#' Run enrichment analysis procedure
#' 
#' This is a generic function that chooses an enrichment analysis procedure
#' based on the model class and runs the analysis.
#' @param model An S4 object which represents one of mulea's tests 
#' (ORA or GSEA). See details
#' for more information.
#' @details The function requires the definition of a model. Models currently
#' implemented in mulea include Gene Set Enrichment Analysis (GSEA) and
#' Over-Representation Analysis (ORA). These models must be defined through
#' their specific functions which are provided in this package. 
#' @seealso \code{\link{gsea}}, \code{\link{ora}}
#' @export
#' @examples
#' library(mulea)
#' 
#' # loading and filtering the example ontology from a GMT file
#' tf_gmt <- read_gmt(file = system.file( package="mulea", "extdata", 
#'     "Transcription_factor_RegulonDB_Escherichia_coli_GeneSymbol.gmt"))
#' tf_gmt_filtered <- filter_ontology(gmt = tf_gmt, min_nr_of_elements = 3, 
#'     max_nr_of_elements = 400)
#' 
#' # loading the example data
#' sign_genes <- readLines(system.file(package = "mulea", "extdata", 
#'     "target_set.txt"))
#' background_genes <- readLines(system.file(package="mulea", "extdata", "
#'     background_set.txt"))
#'
#' # creating the ORA model
#' ora_model <- ora(gmt = tf_gmt_filtered, 
#'     # the test set variable
#'     element_names = sign_genes, 
#'     # the background set variable
#'     background_element_names = background_genes, 
#'     # the p-value adjustment method
#'     p_value_adjustment_method = "eFDR", 
#'     # the number of permutations
#'     number_of_permutations = 10000,
#'     # the number of processor threads to use
#'     nthreads = 2)
#' # running the ORA
#' ora_results <- run_test(ora_model)
#' 
#' @return Results in form of `data.frame`. Structure of `data.frame` depends on
#'   object processed by this generic method.
#'   In the case of `run_test` was used with the model generated 
#'   by the `ora` function the returned 
#'   `data.frame` contains the following columns:
#'   
#'   1. 'ontology_id': Identifiers of the ontology elements.
#'   2. 'ontology_name': Names of the ontology elements.
#'   3. 'nr_common_with_tested_elements': Number of common elements between the 
#'       ontology element and the vector defined by the element_names parameter 
#'       of the `ora` function.
#'   4. 'nr_common_with_background_elements': Number of common elements between 
#'       the ontology element and the vector defined by the 
#'       background_element_names parameter of the `ora` function.
#'   5. 'p_value': The raw *p*-value of the overrepresentation analysis.
#'   6. The adjusted *p*-value. 
#'      The column named based on the 
#'      p_value_adjustment_method parameter of the 
#'      `ora` function, *e.g.* 'eFDR'
#'      
#'   In the case of `run_test` was used with the model 
#'      generated by the `gsea` function the returned 
#'      `data.frame` contains the following columns:
#'   
#'   1. 'ontology_id': Identifiers of the ontology elements.
#'   2. 'ontology_name': Names of the ontology elements.
#'   3. 'nr_common_with_tested_elements': Number of common elements between the 
#'       ontology element and the vector defined by the element_names parameter 
#'       of the `gsea` function.
#'   4. 'p_value': The raw *p*-value of the gene set enrichment analysis.
#'   5. 'adjusted_p_value': The adjusted *p*-value.    
#' @importFrom methods new
setGeneric("run_test", function(model)
    standardGeneric("run_test"))
