% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/censored_likelihood_XS.R
\encoding{UTF8}
\name{censoredLikelihoodXS}
\alias{censoredLikelihoodXS}
\title{Censored log-likelihood function of the extremal Student model}
\usage{
censoredLikelihoodXS(
  obs,
  loc,
  corrFun,
  nu,
  u,
  p = 499L,
  vec = NULL,
  nCores = 1L,
  cl = NULL,
  likelihood = "mgp",
  ntot = NULL,
  std = FALSE,
  ...
)
}
\arguments{
\item{obs}{List of vectors for which at least one component exceeds a high threshold.}

\item{loc}{Matrix of coordinates as given by \code{expand.grid()}.}

\item{corrFun}{correlation function taking a vector of coordinates as input.}

\item{nu}{degrees of freedom of the Student process}

\item{u}{Vector of thresholds under which to censor components.}

\item{p}{Number of samples used for quasi-Monte Carlo estimation. Must be a prime number.}

\item{vec}{Generating vector for the quasi-Monte Carlo procedure. For a given \code{p} and dimensionality,
can be computed using \code{genVecQMC}.}

\item{nCores}{Number of cores used for the computation}

\item{cl}{Cluster instance as created by \code{makeCluster} of the \code{parallel} package.}

\item{likelihood}{vector of string specifying the contribution. Either \code{"mgp"} for multivariate generalized Pareto, 
\code{"poisson"} for a Poisson contribution for the observations falling below or \code{"binom"} for a binomial contribution.}

\item{ntot}{integer number of observations below and above the threshold, to be used with Poisson or binomial likelihood}

\item{std}{logical; if \code{std = TRUE}, consider \code{obs/u} for scalar u and exceedances over 1 rather than \code{obs} \eqn{>} \code{u} for potentially vector \code{u}. This affects the value of the log-likelihood function. Default to \code{FALSE}.}

\item{...}{Additional arguments passed to Cpp routine.}
}
\value{
Negative censored log-likelihood function for the set of observations \code{obs} and correlation function \code{corrFun}, with \code{attributes}  \code{exponentMeasure} for all of the \code{likelihood} type selected, in the order \code{"mgp"}, \code{"poisson"}, \code{"binom"}..
}
\description{
Compute the peaks-over-threshold censored negative log-likelihood function for the extremal Student model.
}
\details{
The function computes the censored log-likelihood function based on the representation
developed by Ribatet (2013); see also Thibaud and Opitz (2015). Margins must have been
standardized, for instance to unit Frechet.
}
\examples{
#Define correlation function
corrFun <- function(h, alpha = 1, lambda = 1){
   exp(-norm(h, type = "2")^alpha/lambda)
}

#Define locations
loc <- expand.grid(1:4, 1:4)

#Compute generating vector
p <- 499L
latticeRule <- genVecQMC(p, (nrow(loc) - 1))
primeP <- latticeRule$primeP
vec <- latticeRule$genVec

#Simulate data
Sigma <- exp(-as.matrix(dist(loc))^0.8)
obs <- rExtremalStudentParetoProcess(n = 1000, nu = 5, Sigma = Sigma)
obs <- split(obs, row(obs))

#Evaluate risk functional
maxima <- sapply(obs, max)
thresh <- quantile(maxima, 0.9)

#Select exceedances
exceedances <- obs[maxima > thresh]

#Compute log-likelihood function
eval <- censoredLikelihoodXS(exceedances, loc, corrFun, nu = 5, u = thresh, primeP, vec)

}
\references{
Thibaud, E. and T. Opitz (2015). Efficient inference and simulation for elliptical Pareto processes. Biometrika, 102(4), 855-870.

Ribatet, M. (2013). Spatial extremes: max-stable processes at work. JSFS, 154(2), 156-177.
}
\author{
Leo Belzile
}
