% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/constructors.R
\name{edgeset.constructors}
\alias{edgeset.constructors}
\alias{network.bipartite}
\alias{network.adjacency}
\alias{network.edgelist}
\alias{network.incidence}
\title{Edgeset Constructors for Network Objects}
\usage{
network.bipartite(x, g, ignore.eval = TRUE, names.eval = NULL, ...)

network.adjacency(x, g, ignore.eval = TRUE, names.eval = NULL, ...)

network.edgelist(x, g, ignore.eval = TRUE, names.eval = NULL, ...)

network.incidence(x, g, ignore.eval = TRUE, names.eval = NULL, ...)
}
\arguments{
\item{x}{a matrix containing edge information}

\item{g}{an object of class \code{network}}

\item{ignore.eval}{logical; ignore edge value information in x?}

\item{names.eval}{a name for the edge attribute under which to store edge
values, if any}

\item{\dots}{possible additional arguments (such as \code{edge.check})}
}
\value{
Invisibly, an object of class \code{network}; these functions modify
  their argument in place.
}
\description{
These functions convert relational data in matrix form to 
  network edge sets.
}
\details{
Each of the above functions takes a \code{network} and a matrix
  as input, and modifies the supplied \code{network} object by adding the
  appropriate edges.  \code{network.adjacency} takes \code{x} to be an 
  adjacency matrix; \code{network.edgelist} takes \code{x} to be an edgelist
  matrix; and \code{network.incidence} takes \code{x} to be an incidence
  matrix.  \code{network.bipartite} takes \code{x} to be a two-mode 
  adjacency matrix where rows and columns reflect each respective mode 
  (conventionally, actors and events); If \code{ignore.eval==FALSE}, 
  (non-zero) edge values are stored as edgewise attributes with name 
  \code{names.eval}.  The \code{edge.check} argument can be added via 
  \code{\dots} and will be passed to \code{\link{add.edges}}.

Edgelist matrices to be used with \code{network.edgelist} should have one 
  row per edge, with the first two columns indicating the sender and 
  receiver of each edge (respectively).  Edge values may be provided in 
  additional columns. The edge attributes will be created with names 
  corresponding to the column names unless alternate names are provided via 
  \code{names.eval}. The vertices specified in the first two columns, which
  can be characters, are added to the network in default sort order. The 
  edges are added in the order specified by the edgelist matrix.
  
Incidence matrices should contain one row per vertex, with one column per 
  edge. A non-zero entry in the matrix means that the edge with the id 
  corresponding to the column index will have an incident vertex with an
  id corresponding to the row index. In the directed case, negative cell
  values are taken to indicate tail vertices, while positive values 
  indicate head vertices. 

Results similar to \code{network.adjacency} can also be obtained by means 
  of extraction/replacement operators.  See the associated man page for 
  details.
}
\examples{
#Create an arbitrary adjacency matrix
m<-matrix(rbinom(25,1,0.5),5,5)
diag(m)<-0

g<-network.initialize(5)    #Initialize the network
network.adjacency(m,g)      #Import the edge data

#Do the same thing, using replacement operators
g<-network.initialize(5)
g[,]<-m

# load edges from a data.frame via network.edgelist
edata <-data.frame(
  tails=c(1,2,3),
  heads=c(2,3,1),
  love=c('yes','no','maybe'),
  hate=c(3,-5,2),
  stringsAsFactors=FALSE
  )

g<-network.edgelist(edata,network.initialize(4),ignore.eval=FALSE)
as.sociomatrix(g,attrname='hate')
g\%e\%'love'

# load edges from an incidence matrix
inci<-matrix(c(1,1,0,0, 0,1,1,0, 1,0,1,0),ncol=3,byrow=FALSE)
inci
g<-network.incidence(inci,network.initialize(4,directed=FALSE))
as.matrix(g)

# load in biparite dataframe with weights
bipMat<-data.frame(
        event1=c(1,2,1,0),
        event2=c(0,0,3,0),
        event3=c(1,1,0,4),
        row.names=c("a","b","c","d"))
net<-network(bipMat,matrix.type='bipartite',ignore.eval=FALSE,names.eval='pies')
as.matrix(net,attername='pies')



}
\references{
Butts, C. T.  (2008).  \dQuote{network: a Package for Managing 
  Relational Data in R.}  \emph{Journal of Statistical Software}, 24(2).  
  \url{http://www.jstatsoft.org/v24/i02/}
}
\seealso{
\code{\link{loading.attributes}}, \code{\link{network}}, 
  \code{\link{network.initialize}}, \code{\link{add.edges}}, 
  \code{\link{network.extraction}}
}
\author{
Carter T. Butts \email{buttsc@uci.edu} and David Hunter 
  \email{dhunter@stat.psu.edu}
}
\keyword{classes}
\keyword{graphs}
