% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_nhdplus.R
\name{plot_nhdplus}
\alias{plot_nhdplus}
\title{Plot NHDPlus}
\usage{
plot_nhdplus(
  outlets = NULL,
  bbox = NULL,
  streamorder = NULL,
  nhdplus_data = NULL,
  gpkg = NULL,
  plot_config = NULL,
  add = FALSE,
  actually_plot = TRUE,
  overwrite = TRUE,
  ...
)
}
\arguments{
\item{outlets}{list of nldi outlets. Other inputs are coerced into nldi outlets, see details.}

\item{bbox}{object of class bbox with a defined crs. See examples.}

\item{streamorder}{integer only streams of order greater than or equal will be returned}

\item{nhdplus_data}{geopackage containing source nhdplus data (omit to download)}

\item{gpkg}{path and file with .gpkg ending. If omitted, no file is written.}

\item{plot_config}{list containing plot configuration, see details.}

\item{add}{boolean should this plot be added to an already built map.}

\item{actually_plot}{boolean actually draw the plot? Use to get data subset only.}

\item{overwrite}{passed on the \link{subset_nhdplus}.}

\item{...}{parameters passed on to rosm.}
}
\value{
plot data is returned invisibly.
}
\description{
Given a list of outlets, get their basin boundaries and network and return a plot.
}
\details{
plot_nhdplus supports several input specifications. An unexported function "as_outlet"
is used to convert the outlet formats as described below.
\enumerate{
  \item if outlets is omitted, the bbox input is required and all nhdplus data
  in the bounding box is plotted.
  \item If outlets is a list of integers, it is assumed to be NHDPlus IDs (comids)
  and all upstream tributaries are plotted.
  \item if outlets is an integer vector, it is assumed to be all NHDPlus IDs (comids)
  that should be plotted. Allows custom filtering.
  \item If outlets is a character vector, it is assumed to be NWIS site ids.
  \item if outlets is a list containing only characters, it is assumed to be a list
  of nldi features and all upstream tributaries are plotted.
  \item if outlets is a data.frame with point geometry, a point in polygon match
  is performed and upstream with tributaries from the identified catchments is plotted.
}

The \code{plot_config} parameter is a list with names "basin", "flowline" and "outlets".
The following shows the defaults that can be altered.
\enumerate{
  \item basin \code{list(lwd = 1, col = NA, border = "black")}
  \item flowline \code{list(lwd = 1, col = "blue")}
  \item outlets \preformatted{
   list(default = list(col = "black", border = NA, pch = 19, cex = 1),
        nwissite = list(col = "grey40", border = NA, pch = 17, cex = 1),
        huc12pp = list(col = "white", border = "black", pch = 22, cex = 1),
        wqp = list(col = "red", border = NA, pch = 20, cex = 1))
        }
}
}
\examples{
\donttest{
rosm::set_default_cachedir(tempfile())

plot_nhdplus("05428500")

plot_nhdplus("05428500", streamorder = 2)

plot_nhdplus(list(13293970, 13293750))

sample_data <- system.file("extdata/sample_natseamless.gpkg", package = "nhdplusTools")
plot_nhdplus(list(13293970, 13293750), streamorder = 3, nhdplus_data = sample_data)

plot_nhdplus(list(list("comid", "13293970"),
                  list("nwissite", "USGS-05428500"),
                  list("huc12pp", "070900020603"),
                  list("huc12pp", "070900020602")),
             streamorder = 2,
             nhdplus_data = sample_data)


plot_nhdplus(sf::st_as_sf(data.frame(x = -89.36083,
                                     y = 43.08944),
                          coords = c("x", "y"), crs = 4326),
             streamorder = 2,
             nhdplus_data = sample_data)

plot_nhdplus(list(list("comid", "13293970"),
                  list("nwissite", "USGS-05428500"),
                  list("huc12pp", "070900020603"),
                  list("huc12pp", "070900020602")),
             streamorder = 2,
             nhdplus_data = sample_data,
             plot_config = list(basin = list(lwd = 2),
                                outlets = list(huc12pp = list(cex = 1.5),
                                               comid = list(col = "green"))))

bbox <- sf::st_bbox(c(xmin = -89.43, ymin = 43, xmax = -89.28, ymax = 43.1),
                    crs = "+proj=longlat +datum=WGS84 +no_defs")

fline <- sf::read_sf(sample_data, "NHDFlowline_Network")
comids <- nhdplusTools::get_UT(fline, 13293970)

plot_nhdplus(comids)

#' # With Local Data
plot_nhdplus(bbox = bbox, nhdplus_data = sample_data)

# With downloaded data
plot_nhdplus(bbox = bbox, streamorder = 3)

# Can also plot on top of the previous!
plot_nhdplus(bbox = bbox, nhdplus_data = sample_data,
             plot_config = list(flowline = list(lwd = 0.5)))
plot_nhdplus(comids, nhdplus_data = sample_data, streamorder = 3, add = TRUE,
             plot_config = list(flowline = list(col = "darkblue")))
}
}
