% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/overlap.R
\name{overlap}
\alias{overlap}
\title{Monte Carlo calculation of niche region overlap metrics.}
\usage{
overlap(
  niche.par,
  nreps,
  nprob,
  alpha = 0.95,
  species.names,
  norm.redraw = TRUE
)
}
\arguments{
\item{niche.par}{A list with \code{nspecies = length(niche.par)}, each element of which in turn is a list with elements \code{mu} and \code{Sigma}.  See Details.}

\item{nreps}{The number of overlap metric calculations for each species.  Defaults to the smallest number of parameter samples supplied by \code{niche.par}.  See 'Details'.}

\item{nprob}{The number of normal draws for each Monte Carlo overlap metric calculation.  See 'Details'.}

\item{alpha}{Scalar or vector of niche region sizes for calculating the niche overlap metric. Defaults to 0.95.}

\item{species.names}{Names of the species. Defaults to \code{names(niche.par)}.}

\item{norm.redraw}{Logical. If \code{FALSE}, the same \code{nprob*nspecies} iid \eqn{N(0,1)} draws are used for each calculation of the overlap metric. This increases the Monte Carlo error, but the procedure is about 1.5x faster.  Defaults to \code{TRUE}.}
}
\value{
Returns an array of size \code{c(nspecies, nspecies, nreps, nlevels)}, where \code{nlevels} is the number of alpha levels at which to calculate the overlap metric.  For each of the last two dimensions of the output array, the first two dimensions form an \code{nspecies} by \code{nspecies} matrix giving each pairwise calculation of overlap metric between two species for given \eqn{\Theta_A}, \eqn{\Theta_B}, and \code{alpha}. In each of these matrices, Species \eqn{A} is along the rows of this matrix and Species \eqn{B} is along the columns.
}
\description{
Calculates the distribution of a niche region overlap metric for each pairwise species combination and user-specified niche region sizes.
}
\details{
The overlap metric is the probability that a randomly drawn individual from species \eqn{A} will be found within the niche region of species \eqn{B} (for a given niche region size, e.g., \code{alpha = .95}).  It is a single number which is a function of the parameters for each species, \eqn{\Theta_A = (\mu_A, \Sigma_A)} and \eqn{\Theta_B = (\mu_B, \Sigma_B)}.  This number is difficult to calculate directly, but easy to approximate stochastically by generating \code{nprob} draws from the distribution of species \eqn{A} and counting the fraction of them which fall in the niche region of species \eqn{B}.

Typically the true values of \eqn{\Theta_A} and \eqn{\Theta_B} are unknown and must be estimated from the data. Thus, the overlap metric is calculated for \code{nreps} combinations of samples from \eqn{p(\Theta_A | X)} and \eqn{p(\Theta_B | X)} which are supplied in \code{niche.par}.

See Swanson et al. (2015) for a detailed description of niche overlap and its calculation.
}
\examples{
# fish data
data(fish)

# generate parameter draws from the "default" posteriors of each fish
nsamples <- 500
system.time({
  fish.par <- tapply(1:nrow(fish), fish$species,
                     function(ii) niw.post(nsamples = nsamples, X = fish[ii,2:4]))
})

# overlap calculation. use nsamples = nprob = 1e4 for more accurate results.
system.time({
  over <- overlap(fish.par, nreps = nsamples, nprob = nsamples,
                  alpha = c(.95, .99))
})

# posterior expectations of overlap metrics
over.mean <- apply(over*100, c(1:2, 4), mean)
round(over.mean)

# posterior 95\% credible intervals of overlap metrics
over.cred <- apply(over*100, c(1:2, 4), quantile,
                   prob = c(.025, .975), na.rm = TRUE)
round(over.cred[,,,1]) # display alpha = .95 niche region
}
\references{
Swanson, H.K., Lysy, M., Stasko, A.D., Power, M., Johnson, J.D., and Reist, J.D. "A new probabilistic method for quantifying n-dimensional ecological niches and niche overlap." \emph{Ecology: Statistical Reports} 96:2 (2015): 318-324. \doi{10.1890/14-0235.1}.
}
\seealso{
\code{\link[=overlap.plot]{overlap.plot()}}, \code{\link[=niw.post]{niw.post()}}, \code{\link[=niiw.post]{niiw.post()}}.
}
