% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv.nnGarrote.R
\name{cv.nnGarrote}
\alias{cv.nnGarrote}
\title{Non-negative Garrote Estimator - Cross-Validation}
\usage{
cv.nnGarrote(
  x,
  y,
  intercept = TRUE,
  initial.model = c("LS", "glmnet")[1],
  lambda.nng = NULL,
  lambda.initial = NULL,
  alpha = 0,
  nfolds = 5,
  verbose = TRUE
)
}
\arguments{
\item{x}{Design matrix.}

\item{y}{Response vector.}

\item{intercept}{Boolean variable to determine if there is intercept (default is TRUE) or not.}

\item{initial.model}{Model used for the groups. Must be one of "LS" (default) or "glmnet".}

\item{lambda.nng}{Shinkage parameter for the non-negative garrote. If NULL(default), it will be computed based on data.}

\item{lambda.initial}{The shinkrage parameter for the "glmnet" regularization.}

\item{alpha}{Elastic net mixing parameter for initial estimate. Should be between 0 (default) and 1.}

\item{nfolds}{Number of folds for the cross-validation procedure.}

\item{verbose}{Boolean variable to determine if console output for cross-validation progress is printed (default is TRUE).}
}
\value{
An object of class cv.nnGarrote
}
\description{
\code{cv.nnGarrote} computes the non-negative garrote estimator with cross-validation.
}
\examples{
\donttest{
# Setting the parameters
p <- 500
n <- 100
n.test <- 5000
sparsity <- 0.15
rho <- 0.5
SNR <- 3
set.seed(0)
# Generating the coefficient
p.active <- floor(p*sparsity)
a <- 4*log(n)/sqrt(n)
neg.prob <- 0.2
nonzero.betas <- (-1)^(rbinom(p.active, 1, neg.prob))*(a + abs(rnorm(p.active)))
true.beta <- c(nonzero.betas, rep(0, p-p.active))
# Two groups correlation structure
Sigma.rho <- matrix(0, p, p)
Sigma.rho[1:p.active, 1:p.active] <- rho
diag(Sigma.rho) <- 1
sigma.epsilon <- as.numeric(sqrt((t(true.beta) \%*\% Sigma.rho \%*\% true.beta)/SNR))

# Simulate some data
library(mvnfast)
x.train <- mvnfast::rmvn(n, mu=rep(0,p), sigma=Sigma.rho)
y.train <- 1 + x.train \%*\% true.beta + rnorm(n=n, mean=0, sd=sigma.epsilon)
x.test <- mvnfast::rmvn(n.test, mu=rep(0,p), sigma=Sigma.rho)
y.test <- 1 + x.test \%*\% true.beta + rnorm(n.test, sd=sigma.epsilon)

# Applying the NNG with Ridge as an initial estimator
nng.out <- cv.nnGarrote(x.train, y.train, intercept=TRUE,
                        initial.model=c("LS", "glmnet")[2],
                        lambda.nng=NULL, lambda.initial=NULL, alpha=0,
                        nfolds=5)
nng.predictions <- predict(nng.out, newx=x.test)
mean((nng.predictions-y.test)^2)/sigma.epsilon^2
coef(nng.out)
}

}
\seealso{
\code{\link{coef.cv.nnGarrote}}, \code{\link{predict.cv.nnGarrote}}
}
\author{
Anthony-Alexander Christidis, \email{anthony.christidis@stat.ubc.ca}
}
