% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svarmod.sb.R
\name{fitsvar.sb.iso}
\alias{fitsvar.sb.iso}
\alias{fitsvar}
\title{Fit an isotropic Shapiro-Botha variogram model}
\usage{
fitsvar.sb.iso(esv, dk = 4 * ncol(esv$data$x), nx = NULL,
  rmax = esv$grid$max, min.contrib = 10, method = c("cressie",
  "equal", "npairs", "linear"), iter = 10,
  tol = sqrt(.Machine$double.eps))
}
\arguments{
\item{esv}{pilot semivariogram estimate, a \code{\link{np.svar}}-\code{\link{class}} 
(or \code{\link{svar.bin}}) object. Typically an output of the function
\code{\link{np.svariso}}.}

\item{dk}{dimension of the kappa function (\code{dk == 0} corresponds to a model 
valid in any dimension; if \code{dk > 0}, it should be greater than 
or equal to the dimension of the spatial process \code{ncol(esv$data$x)}).}

\item{nx}{number of discretization nodes. Defaults to \code{min(nesv - 1, 50)},
where \code{nesv} is the number of semivariogram estimates.}

\item{rmax}{maximum lag considered in the discretization
(range of the fitted variogram on output).}

\item{min.contrib}{minimum number of (equivalent) contributing pairs 
(pilot estimates with a lower number are ignored, with a warning).}

\item{method}{string indicating the WLS fitting method to be used
(e.g. \code{method = "cressie"}). See "Details" below.}

\item{iter}{maximum number of interations of the WLS algorithm (used only
if \code{method == "cressie"}).}

\item{tol}{absolute convergence tolerance (used only
if \code{method == "cressie"}).}
}
\value{
Returns the fitted variogram model, an object of \code{\link{class}} \code{fitsvar}.
A \code{\link{svarmod}} object
with additional components \code{esv} (pilot semivariogram estimate) and \code{fit} containing:
\item{u}{vector of lags/distances.}
\item{sv}{vector of pilot semivariogram estimates.}
\item{fitted.sv}{vector of fitted semivariances.}
\item{w}{vector of (least squares) weights.}
\item{wls}{value of the objective function.}
\item{method}{string indicating the WLS fitting method used.}
\item{iter}{number of WLS iterations (if \code{method == "cressie"}).}
}
\description{
Fits a `nonparametric' isotropic Shapiro-Botha variogram model by WLS through
quadratic programming.
Following Gorsich and Genton (2004), the nodes are selected as the scaled 
roots of Bessel functions (see \code{\link{disc.sb}}).
}
\details{
The fit is done using a (possibly iterated) weighted least squares criterion, minimizing: 
\deqn{WLS(\theta) = \sum_i w_i[(\hat{\gamma}(h_i)) -	\gamma(\theta; h_i)]^2.}
The different options for the argument \code{method} define the WLS algorithm used:
\describe{
 \item{\code{"cressie"}}{The default method. The procedure is
 iterative, with \eqn{w_i = 1} (OLS) used for the first step
 and with the weights recalculated at each iteration,
 following Cressie (1985), until convergence: \deqn{w_i =
 N(h_i)/\gamma(\hat{\theta}; h_i)^2,} where \eqn{N(h_i)}
 is the (equivalent) number of contributing pairs in the
 estimation at lag \eqn{h_i}.}
 \item{\code{"equal"}}{Ordinary least squares: \eqn{w_i = 1}.}
 \item{\code{"npairs"}}{\eqn{w_i = N(h_i).}} 
 \item{\code{"linear"}}{\eqn{w_i = N(h_i)/h_i^2} 
 (default fitting method in \pkg{gstat} package).} 
} 
Function \code{\link[quadprog]{solve.QP}} of \pkg{quadprog} package is used
to solve a strictly convex quadratic program. To avoid problems, the Choleski decomposition
of the matrix corresponding to the original problem is computed using \code{\link{chol}} with \code{pivot = TRUE}.
If this matrix is only positive semi-definite (non-strictly convex QP),
the number of discretization nodes will be less than \code{nx}.
}
\examples{
# Trend estimation
lp <- locpol(aquifer[,1:2], aquifer$head, nbin = c(41,41),
             h = diag(100, 2), hat.bin = TRUE)
                               # 'np.svariso.corr()' requires a 'lp$locpol$hat' component

# Variogram estimation
esvar <- np.svariso.corr(lp, maxlag = 150, nlags = 60, h = 60, plot = FALSE)

# Variogram fitting
svm2 <- fitsvar.sb.iso(esvar)  # dk = 2
svm3 <- fitsvar.sb.iso(esvar, dk = 0) # To avoid negative covariances...
svm4 <- fitsvar.sb.iso(esvar, dk = 10) # To improve fit...

plot(svm4, main = "Nonparametric bias-corrected semivariogram and fitted models", legend = FALSE)
plot(svm3, add = TRUE)
plot(svm2, add = TRUE, lty = 3)
legend("bottomright", legend = c("NP estimates", "fitted model (dk = 10)", "dk = 0", "dk = 2"),
            lty = c(NA, 1, 1, 3), pch = c(1, NA, NA, NA), lwd = c(1, 2, 1, 1))
}
\references{
Ball, J.S. (2000) Automatic computation of zeros of Bessel functions and other
  special functions. \emph{SIAM Journal on Scientific Computing}, \bold{21}, 
  1458-1464.

Cressie, N. (1985) Fitting variogram models by weighted least squares.
  \emph{Mathematical Geology}, \bold{17}, 563-586. 

Cressie, N. (1993) \emph{Statistics for Spatial Data}. New York. Wiley.

Fernandez Casal R., Gonzalez Manteiga W. and  Febrero Bande M. (2003) 
Flexible Spatio-Temporal Stationary Variogram Models, 
\emph{Statistics and Computing}, \bold{13}, 127-136.

Gorsich, D.J. and Genton, M.G. (2004) On the discretization of nonparametric 
  covariogram estimators. \emph{Statistics and Computing}, \bold{14}, 99-108.

Shapiro, A. and Botha, J.D. (1991) Variogram fitting with a general class of 
  conditionally non-negative definite functions. \emph{Computational Statistics 
  and Data Analysis}, \bold{11}, 87-96.
}
\seealso{
\code{\link{svarmod.sb.iso}}, \code{\link{disc.sb}}, \code{\link{plot.fitsvar}}.
}
