% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/caliper.R
\docType{methods}
\name{caliper}
\alias{caliper}
\alias{caliper,InfinitySparseMatrix-method}
\alias{caliper,matrix-method}
\alias{caliper,optmatch.dlist-method}
\title{Prepare matching distances suitable for matching within calipers.}
\usage{
caliper(x, width, exclude = c(), compare = `<=`, values = FALSE)

\S4method{caliper}{InfinitySparseMatrix}(x, width, exclude = c(),
  compare = `<=`, values = FALSE)

\S4method{caliper}{matrix}(x, width, exclude = c(), compare = `<=`,
  values = FALSE)

\S4method{caliper}{optmatch.dlist}(x, width, exclude = c(), compare = `<=`,
  values = FALSE)
}
\arguments{
\item{x}{A distance specification created with \code{\link{match_on}} or similar.}

\item{width}{The width of the caliper: how wide of a margin to
allow in matches. Be careful in setting the width. Vector valued
arguments will be recycled for each of the finite entries in
\code{x} (and no order is guaranteed for \code{x} for some types of
distance objects).}

\item{exclude}{(Optional) A character vector of observations (corresponding to row and column names) to exclude from the caliper.}

\item{compare}{A function that decides that whether two
observations are with the caliper. The default is
\code{`<=`}. \code{`<`} is a common alternative.}

\item{values}{Should the returned object be made of all zeros
(\code{values = FALSE}, the default) or should the object include
the values of the original object (\code{values = TRUE})?}
}
\value{
A matrix like object that is suitable to be given
as \code{distance} argument to \code{\link{fullmatch}} or
\code{\link{pairmatch}}. The caliper will be only zeros and \code{Inf} values,
indicating a possible match or no possible match, respectively.

You can combine the results of \code{caliper} with other distances using the
\code{`+`} operator. See the examples for usage.
}
\description{
Encodes calipers, or maximum allowable distances within which to
match. The result of a call to \code{caliper} is itself a distance specification between
treated and control units that can be used with 
\code{pairmatch()} or \code{fullmatch()}. Calipers can also be combined with
other distance specifications for richer matching problems.
}
\details{
\code{caliper} is a generic function with methods for any of the allowed distance
specifications: user created matrices, the results of \code{\link{match_on}}, the results
of \code{\link{exactMatch}}, or combinations (using \code{`+`}) of these objects.

\code{width} provides the size of the caliper, the allowable distance for
matching. If the distance between a treated and control pair is less than or equal
to this distance, it is allowed kept; otherwise, the pair is discarded from future
matching. The default comparison of ``equal or less than can'' be changed to any other
comparison function using the \code{comparison} argument.

It is important to understand that \code{width} argument is defined on the
scale of these distances. For univariate distances such as propensity scores,
it is common to specificy calipers in standard deviations. If a caliper of
this nature is desired, you must either find the standard deviation directly
or use the \code{\link{match_on}} function with its \code{caliper} argument.
Since \code{match_on} has access to the underlying univariate scores, for
example for the GLM method, it can determine the caliper width in standard
deviations.

If you wish to exclude specific units from the caliper requirements, pass the names of 
these units in the \code{exclude} argument. These units will be allowed to match any other
unit.
}
\examples{
data(nuclearplants)


### Caliper of 100 MWe on plant capacity
caliper(match_on(pr~cap, data=nuclearplants, method="euclidean"), width=100)

### Caliper of 1/2 a pooled SD of plant capacity
caliper(match_on(pr~cap, data=nuclearplants), width=.5)

### Caliper  of .2 pooled SDs in the propensity score
ppty <- glm(pr ~ . - (pr + cost), family = binomial(), data = nuclearplants)
ppty.dist <- match_on(ppty)

pptycaliper <- caliper(ppty.dist, width = .2)

### caliper on the Mahalanobis distance
caliper(match_on(pr ~ t1 + t2, data = nuclearplants), width = 3)

### Combining a Mahalanobis distance matching with a caliper
### of 1 pooled SD in the propensity score:
mhd.pptyc <- caliper(ppty.dist, width = 1) +
          match_on(pr ~ t1 + t2, data = nuclearplants)
pairmatch(mhd.pptyc, data = nuclearplants)

### Excluding observations from caliper requirements:
caliper(match_on(pr ~ t1 + t2, data = nuclearplants), width = 3, exclude = c("A", "f"))

### Returning values directly (equal up to the the attributes)
all(abs((caliper(ppty.dist, 1) + ppty.dist) -
        caliper(ppty.dist, 1, values = TRUE)) < .Machine$Double.eps)
}
\author{
Mark M. Fredrickson and Ben B. Hansen
}
\references{
P.~R. Rosenbaum and D.~B. Rubin (1985),
\sQuote{Constructing a control group using multivariate matched sampling
  methods that incorporate the propensity score},
 \emph{The American Statistician}, \bold{39} 33--38.
}
\seealso{
\code{\link{exactMatch}}, \code{\link{match_on}}, \code{\link{fullmatch}}, \code{\link{pairmatch}}
}
\keyword{nonparametric}

