% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/draw.sim.R
\name{draw.sim}
\alias{draw.sim}
\title{Draw a sim object}
\usage{
draw.sim(sim, fossils = NULL, sortBy = "TS", lwdLin = 4, showLabel = TRUE, ...)
}
\arguments{
\item{sim}{A \code{sim} object, containing extinction times, speciation 
times, parent, and status information for each species in the simulation. 
See \code{?sim}.}

\item{fossils}{A \code{data.frame} containing the fossil occurrences of 
each lineage, e.g. as returned by the \code{sample.clade} function. The
format of this argument will define the way fossils are drawn (see below).}

\item{sortBy}{A single character or integer vector indicating how lineages 
should be sorted in the plot. If it is a string (see example 3), it 
indicates which element in the \code{sim} object that should be used to sort
lineages in the plot. If it is a vector of integers, it directly specifies
the order in which lineages should be drawn, from the bottom (i.e. the
first integer) to the upper side (#th integer, with # = number of lineages
in \code{sim}) of the figure. Default value of this parameter is "TS", so by
default species will be sorted by order of origination in the simulation.}

\item{lwdLin}{The relative thickness/size of all elements (i.e., lines and 
points in the plot. Default value is 4 (i.e. equal to \code{lwd = 4} for 
the black horizontal lines).}

\item{showLabel}{A \code{logical} on whether to draw species labels (i.e. 
species 1 being t1, species 2 t2 etc.). Default is \code{TRUE}.}

\item{...}{Further arguments to be passed to \code{plot}}
}
\value{
A plot of the simulation in the graphics window. If the 
\code{fossils} data.frame is supplied, its format will dictate how fossil
occurrences will be plotted. If \code{fossils} has a \code{SampT} column
(i.e. the occurrence times are exact), fossil occurrences are assigned as 
dots. If \code{fossils} has columns \code{MaxT} and \code{MinT} (i.e. the 
early and late stage bounds associated with each occurrence), fossil 
occurrences are represented as slightly jittered, semitransparent bars 
indicating the early and late bounds of each fossil occurrence.
}
\description{
Draws species longevities for a paleobuddy simulation (a \code{sim} object -
see \code{?sim}) in the graphics window. Allows for the assignment of 
speciation and sampling events, and further customization.
}
\examples{

###
# we start drawing a simple simulation

# maximum simulation time
tMax <- 10

# set seed
set.seed(1)

# run a simulation
sim <- bd.sim(n0 = 1, lambda = 0.6, mu = 0.55, tMax = tMax, 
              nFinal = c(10,20)) 
              
# draw it
draw.sim(sim)

###
# we can add fossils to the drawing

# maximum simulation time
tMax <- 10

# set seed
set.seed(1)

# run a simulation
sim <- bd.sim(n0 = 1, lambda = 0.6, mu = 0.55, tMax = tMax, 
              nFinal = c(10,20)) 

# set seed
set.seed(1)

# simulate data resulting from a fossilization process
# with exact occurrence times
fossils <- sample.clade(sim = sim, rho = 4, tMax = tMax, returnTrue = TRUE)

# draw it
draw.sim(sim, fossils = fossils)

# we can order the vertical drawing of species based on
# any element of sim
draw.sim(sim, fossils = fossils, sortBy = "PAR")
# here we cluster lineages with their daughters by
# sorting them by the "PAR" list of the sim object

draw.sim(sim, fossils = fossils, sortBy = "TE")
# here we sort lineages by their extinction times

###
# try with fossil ranges

# maximum simulation time
tMax <- 10

# set seed
set.seed(1)

# run birth-death simulation
sim <- bd.sim(n0 = 1, lambda = 0.6, mu = 0.55, tMax = tMax, 
              nFinal = c(10,20)) 

# simulate data resulting from a fossilization process
# with fossil occurrence time ranges

# set seed
set.seed(20)

# create time bins randomly
bins <- c(tMax, 0, runif(n = rpois(1, lambda = 6), min = 0, max = tMax))

# set seed
set.seed(1)

# simulate fossil sampling
fossils <- sample.clade(sim = sim, rho = 2, tMax = tMax, 
                        returnTrue = FALSE, bins = bins)

# draw it, sorting lineages by their parent
draw.sim(sim, fossils = fossils, sortBy = "PAR")

# adding the bounds of the simulated bins
abline(v = bins, lty = 2, col = "blue", lwd = 0.5)

###
# we can control how to sort displayed species exactly

# maximum simulation time
tMax <- 10

# set seed
set.seed(1)

# run birth-death simulations
sim <- bd.sim(n0 = 1, lambda = 0.6, mu = 0.55, tMax = tMax, 
              nFinal = c(10,20)) 

# set seed
set.seed(1)  

# simulate fossil sampling
fossils <- sample.clade(sim = sim, rho = 4, tMax = tMax, returnTrue = TRUE)

# draw it with random sorting (in pratice this could be a trait
# value, for instance)
draw.sim(sim, fossils = fossils, sortBy = sample(1:length(sim$TS)))

}
\author{
Matheus Januario
}
