% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/perCapitaRates.R
\name{perCapitaRates}
\alias{perCapitaRates}
\title{perCapitaRates}
\usage{
perCapitaRates(timeList, plot = TRUE, logRates = FALSE,
  drop.extant = FALSE, isExtant = NULL, jitter = TRUE,
  legendPosition = "topleft")
}
\arguments{
\item{timeList}{A list composed of two matrices, giving interval start and end 
dates and taxon first and last occurrences within those intervals. See details.}

\item{plot}{If \code{TRUE}, the per-capita origination and extinctions rates are 
plotted for each interval. Rates which cannot be calculated for an interval
will not be plotted, thus appearing as a gap in the plotted graph. The
author takes no responsibility for the aesthetics of this plot.}

\item{logRates}{If \code{TRUE}, rates plotted on log scale.}

\item{drop.extant}{Drops all extant taxa from a dataset before
calculating per-capita origination and extinction rates.}

\item{isExtant}{A vector of \code{TRUE} and \code{FALSE} values, same length as the
number of taxa in the second matrix of timeList, where \code{TRUE} values indicate
taxa that are alive in the modern day (and thus are boundary crossers which
leave the most recent interval). By default, this argument is NULL and instead
which taxa are extant is inferred based on which taxa occur in an interval
with start and end times both equal to zero. See details.}

\item{jitter}{If \code{TRUE} (default) the extinction rate will be plotted slightly
ahead of the origination rate on the time axis, so the two can be differentiated.}

\item{legendPosition}{The position of a legend indicating which line is
origination rate and which is extinction rate on the resulting plot. This
is given as the possible positions for argument 'x' of the function 
\code{\link{legend}}, and by default is \code{"topleft"}, which will be generally
useful if origination and extinction rates are initially low. If 
\code{legendPosition = NA}, then a legend will not be plotted.}
}
\value{
This function will invisibly return a ten column matrix,
where the number of rows is equal to the number of intervals. The
first two columns are interval start and end times and the third
column is interval length. The fourth through eighth column is the
four fundamental classes of taxa from Foote (2001): Nbt, NbL, NFt,
NFL and their sum, N. The final two columns are the per-capita
rates estimated for each interval in units per lineage time-units;
the ninth column is the origination rate ('pRate') and the tenth
column is the extinction rate ('qRate').
}
\description{
Calculates and plots per-capita origination and extinction rates from
sequential discrete-time taxon ranges, following Foote (2000).
}
\details{
This function calculates the per-capita rates of taxonomic origination
and extinction from paleontological range data, as described by Foote 
(2000). These values are the instantaneous rate of either type of event
occurring per lineage time-units. Although Foote (2001) also presents a
number of alternative rates collected from the prior literature such as
the 'Van Valen' rate metrics, these are not implemented here, but could be
estimated using the matrix invisibly output by this function (See Foote,
2000, for the relevant equations for calculating these).

The timeList object should be a list composed of two matrices, the first
matrix giving by-interval start and end times (in absolute time), the second
matrix giving the by-taxon first and last appearances in the intervals
defined in the first matrix, numbered as the rows. Absolute time should be
decreasing, while the intervals should be numbered so that the number
increases with time. Taxa alive in the modern should be either (a) listed 
in isExtant or (b) listed as last occurring in a time interval that 
begins at time 0 and ends at time 0. See the documentation for the time-scaling 
function \code{\link{bin_timePaleoPhy}} and the simulation function 
\code{\link{binTimeData}} for more information on formatting.

Unlike some functions in paleotree, such as the diversity curve functions,
intervals must be both sequential and non-overlapping. The diversity curve
functions deal with such issues by assuming taxa occur from the base of the
interval they are first found in until the end of the last interval they
are occur in. This inflation of boundary crossers could badly bias estimates
of per-capita diversification rates.
}
\examples{

#with the retiolinae dataset
data(retiolitinae)
perCapitaRates(retioRanges)

#Simulate some fossil ranges with simFossilRecord
set.seed(444)
record <- simFossilRecord(p = 0.1, q = 0.1, nruns = 1,
	nTotalTaxa = c(80,100), nExtant = 0)
taxa <- fossilRecord2fossilTaxa(record)
#simulate a fossil record with imperfect sampling with sampleRanges()
rangesCont <- sampleRanges(taxa,r = 0.5)
#Now let's use binTimeData() to bin in intervals of 5 time units
rangesDisc <- binTimeData(rangesCont,int.length = 5)
#and get the per-capita rates
perCapitaRates(rangesDisc)
#on a log scale
perCapitaRates(rangesDisc,logRates = TRUE)

#get mean and median per-capita rates
res <- perCapitaRates(rangesDisc,plot = FALSE)
apply(res[,c("pRate","qRate")],2,mean,na.rm = TRUE)
apply(res[,c("pRate","qRate")],2,median,na.rm = TRUE)

#with modern taxa
set.seed(444)
record <- simFossilRecord(p = 0.1, q = 0.1, nruns = 1,
	nExtant = c(10,50))
taxa <- fossilRecord2fossilTaxa(record)
#simulate a fossil record with imperfect sampling with sampleRanges()
rangesCont <- sampleRanges(taxa,r = 0.5,,modern.samp.prob = 1)
#Now let's use binTimeData() to bin in intervals of 5 time units
rangesDisc <- binTimeData(rangesCont,int.length = 5)
#and now get per-capita rates
perCapitaRates(rangesDisc)

}
\references{
Foote, M. 2000 Origination and extinction components of taxonomic diversity:
general problems. Pp. 74--102. In D. H. Erwin, and S. L. Wing, eds. Deep
Time: Paleobiology's Perspective. The Paleontological Society, Lawrence,
Kansas.
}
\seealso{
\code{\link{DiversityCurves}}, \code{\link{SamplingConv}}
}
