% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cluster.R
\name{pdSpecClust1D}
\alias{pdSpecClust1D}
\title{Intrinsic 1D wavelet-based clustering of multivariate spectra.}
\usage{
pdSpecClust1D(P, K, jmax, metric = "Riemannian", m = 2, d.jmax = 0.1,
  eps = c(1e-04, 1e-04), tau = 0.5, max.iter = 50, return.centers = F,
  ...)
}
\arguments{
\item{P}{a (\eqn{d,d,n,S})-dimensional array of \eqn{n}-dimensional sequences of HPD matrices for \code{S}
different subjects, with \eqn{n} a dyadic number.}

\item{K}{the number of clusters, should be a integer larger than 1.}

\item{jmax}{an upper bound on the maximum wavelet scale to be considered in the clustering procedure. If
\code{jmax} is not specified, it is set equal to the largest (i.e. finest) possible wavelet scale.}

\item{metric}{the metric that the space of HPD matrices is equipped with. The default choice is \code{"Riemannian"},
but this can also be one of: \code{"logEuclidean"}, \code{"Cholesky"}, \code{"rootEuclidean"} or
\code{"Euclidean"}. The intrinsic distance measures in the clustering algorithm fundamentally rely on the chosen metric.}

\item{m}{the fuzziness parameter for both the fuzzy c-medoids and the weighted fuzzy c-means algorithm. \code{m}
should be larger or equal to \eqn{1}. If \eqn{m = 1} the cluster assignments are no longer fuzzy (i.e. the procedure
performs hard clustering).}

\item{d.jmax}{a proportion that is used to determine the maximum wavelet scale to be considered in the clustering
procedure. A larger value \code{d.jmax} leads to less wavelet coefficients being taken into account, and therefore
lower computational effort in the procedure. If \code{d.jmax} is not specified, by default \code{d.jmax = 0.1}.}

\item{eps}{an optional vector with two elements determining the stopping criterion. The fuzzy c-medoids algorithm
(i.e. first clustering step) terminates if the (integrated) intrinsic distance between cluster centers is smaller than
\code{eps[1]}. The weighted fuzzy c-means (i.e. second clustering step) terminates if the (integrated) distance between
cluster centers is smaller than \code{eps[2]}. If \code{eps} is not specified, by default \code{eps = c(1E-4, 1E-4)}.}

\item{tau}{an optional argument tuning the weight given to the cluster assignments obtained in the first step of
the clustering algorithm. If \code{tau} is not specified, by default \code{tau = 0.5}.}

\item{max.iter}{an optional argument tuning the maximum number of iterations in both the first and second step of the
clustering algorithm, defaults to \code{max.iter = 50}.}

\item{return.centers}{should the cluster centers transformed back the space of HPD matrices also be returned?
Defaults to \code{return.centers = F}.}

\item{...}{additional arguments for internal use.}
}
\value{
The function returns a list with 6 components:
\describe{
  \item{cl.prob }{an (\eqn{S,K})-dimensional matrix, where the value at position (\eqn{s,k}) in the
  matrix corresponds to the probabilistic cluster membership assignment of subject \eqn{s} with respect
  to cluster \eqn{k}.}
  \item{cl.centers.D }{a list of \code{K} wavelet coefficient pyramids, where each pyramid of wavelet
  coefficients is associated to a cluster center.}
  \item{cl.centers.M0 }{a list \code{K} arrays of coarse-scale midpoints at scale \code{j = 0}, where each
  array is associated to a cluster center.}
  \item{cl.centers.f }{if \code{return.centers = T} returns a list of \code{K} \eqn{(d,d,n)}-dimensional arrays,
  where each array corresponds to a discretized curve of HPD matrices associated to a cluster center. If
  \code{return.centers = F}, \code{cl.centers.f} returns \code{NULL}.}
  \item{cl.jmax }{the maximum wavelet scale taken into account in the clustering procedure determined by
  the input arguments \code{jmax} and \code{d.jmax}.}
  \item{iters }{the number of iterations in respectively the first and second step of the clustering procedure.}
}
}
\description{
\code{pdSpecClust1D} performs clustering of multivariate spectral matrices via a two-step fuzzy
clustering algorithm in the intrinsic manifold wavelet domain of curves in the space of HPD matrices
equipped with a metric, e.g. the Riemannian metric, specified by the user.
}
\details{
The input array \code{P} contains initial noisy HPD spectral estimates of the
(\eqn{d,d})-dimensional spectral matrices at \eqn{n} different frequencies for \eqn{S}
different subjects, where \eqn{n} is a dyadic number. The initial spectral estimates can
be e.g. the tapered HPD periodograms given as output by \code{\link{pdPgram}}. \cr
For each subject \eqn{s}, thresholded wavelet coefficients in the intrinsic manifold wavelet domain are
calculated by \code{\link{pdSpecEst1D}}.\cr
The maximum wavelet scale taken into account in the clustering procedure is determined by the arguments
\code{jmax} and \code{d.jmax}. The maximum scale is set to the minimum of \code{jmax} and the wavelet
scale \eqn{j} for which the proportion of nonzero thresholded wavelet coefficients (averaged
across subjects) is smaller than \code{d.jmax}.\cr
The \eqn{S} subjects are assigned to \eqn{K} different clusters in a probabilistic fashion according to a
two-step procedure:
\enumerate{
\item In the first step, an intrinsic fuzzy c-medoids algorithm, with fuzziness parameter \eqn{m} is applied to the
\eqn{S} coarsest midpoints at scale \code{j = 0} in the subject-specific midpoints pyramids. Note that the intrinsic
c-medoids algorithm crucially relies on the metric that the space of HPD matrices gets equipped with.
\item In the second step, a weighted fuzzy c-means algorithm based on the Euclidean
distance function, also with fuzziness parameter \eqn{m}, is applied to the nonzero thresholded wavelet
coefficients of the \eqn{S} different subjects. The tuning parameter \code{tau} controls the weight given
to the cluster assignments obtained in the first step of the clustering algorithm.
}
If \code{return.centers = T}, the function also returns the \code{K} HPD spectral curves corresponding to
the cluster centers based on the given metric by applying the intrinsic inverse 1D wavelet transform (
\code{\link{InvWavTransf1D}}) to the cluster centers in the wavelet domain.
}
\examples{
## ARMA(1,1) process: Example 11.4.1 in (Brockwell and Davis, 1991)

Phi1 <- array(c(0.5, 0, 0, 0.6, rep(0, 4)), dim = c(2, 2, 2))
Phi2 <- array(c(0.7, 0, 0, 0.4, rep(0, 4)), dim = c(2, 2, 2))
Theta <- array(c(0.5, -0.7, 0.6, 0.8, rep(0, 4)), dim = c(2, 2, 2))
Sigma <- matrix(c(1, 0.71, 0.71, 2), nrow = 2)

## Generate periodogram data for 10 subjects
pgram <- function(Phi) pdPgram(rARMA(2^9, 2, Phi, Theta, Sigma)$X)$P
P <- array(c(replicate(5, pgram(Phi1)), replicate(5, pgram(Phi2))), dim=c(2,2,2^8,10))

cl <- pdSpecClust1D(P, K = 2, metric = "logEuclidean")

}
\references{
Chau, J. and von Sachs, R. (2017). \emph{Positive definite multivariate spectral
estimation: a geometric wavelet approach}. Available at \url{http://arxiv.org/abs/1701.03314}.

Brockwell, P.J. and Davis, R.A. (1991). \emph{Time series: Theory and Methods}. New York: Springer.
}
\seealso{
\code{\link{pdSpecEst1D}}, \code{\link{WavTransf1D}}, \code{\link{pdDist}}, \code{\link{pdPgram}}
}
