% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/opt.R
\name{pedmod_opt}
\alias{pedmod_opt}
\alias{pedmod_start}
\alias{pedmod_start_loadings}
\title{Optimize the Log Marginal Likelihood}
\usage{
pedmod_opt(
  ptr,
  par,
  maxvls,
  abs_eps,
  rel_eps,
  opt_func = NULL,
  seed = 1L,
  indices = NULL,
  minvls = -1L,
  do_reorder = TRUE,
  use_aprx = FALSE,
  n_threads = 1L,
  cluster_weights = NULL,
  fix = NULL,
  standardized = FALSE,
  method = 0L,
  use_tilting = FALSE,
  vls_scales = NULL,
  ...
)

pedmod_start(
  ptr,
  data,
  maxvls = 1000L,
  abs_eps = 0,
  rel_eps = 0.01,
  seed = 1L,
  indices = NULL,
  scale_max = 9,
  minvls = 100L,
  do_reorder = TRUE,
  use_aprx = TRUE,
  n_threads = 1L,
  cluster_weights = NULL,
  standardized = FALSE,
  method = 0L,
  sc_start = NULL,
  use_tilting = FALSE,
  vls_scales = NULL
)

pedmod_start_loadings(
  ptr,
  data,
  indices = NULL,
  cluster_weights = NULL,
  sc_start_invariant = NULL
)
}
\arguments{
\item{ptr}{object from \code{\link{pedigree_ll_terms}} or
\code{\link{pedigree_ll_terms_loadings}}.}

\item{par}{starting values passed to \code{opt_func}.}

\item{maxvls}{maximum number of samples in the approximation for each
marginal likelihood term.}

\item{abs_eps}{absolute convergence threshold for
\code{\link{eval_pedigree_ll}} and \code{\link{eval_pedigree_grad}}.}

\item{rel_eps}{rel_eps convergence threshold for
\code{\link{eval_pedigree_ll}} and \code{\link{eval_pedigree_grad}}.}

\item{opt_func}{function to perform minimization with arguments like
\code{\link{optim}}. BFGS is used with \code{\link{optim}} if this argument
is \code{NULL}.}

\item{seed}{seed to pass to \code{\link{set.seed}} before each gradient and
function evaluation. Use \code{NULL} if the seed should not be fixed.}

\item{indices}{zero-based vector with indices of which log marginal
likelihood terms to include. Use \code{NULL} if all indices should be
used.}

\item{minvls}{minimum number of samples for each
marginal likelihood term. Negative values provides a
default which depends on the dimension of the integration.}

\item{do_reorder}{\code{TRUE} if a heuristic variable reordering should
be used. \code{TRUE} is likely the best value.}

\item{use_aprx}{\code{TRUE} if a less precise approximation of
\code{\link{pnorm}} and \code{\link{qnorm}} should be used. This may
reduce the computation time while not affecting the result much.}

\item{n_threads}{number of threads to use.}

\item{cluster_weights}{numeric vector with weights for each cluster. Use
\code{NULL} if all clusters have weight one.}

\item{fix}{integer vector with indices of \code{par} to fix. This is useful
for computing profile likelihoods. \code{NULL} yields all parameters.}

\item{standardized}{logical for whether to use the standardized or direct
parameterization. See \code{\link{standardized_to_direct}} and the vignette
at \code{vignette("pedmod", package = "pedmod")}.}

\item{method}{integer with the method to use. Zero yields randomized Korobov
lattice rules while one yields scrambled Sobol sequences.}

\item{use_tilting}{\code{TRUE} if the minimax tilting method suggested
by Botev (2017). See \doi{10.1111/rssb.12162}.}

\item{vls_scales}{can be a numeric vector with a positive scalar for each
cluster. Then \code{vls_scales[i] * minvls} and
\code{vls_scales[i] * maxvls} is used for cluster \code{i} rather than
\code{minvls} and \code{maxvls}. Set \code{vls_scales = NULL} if the latter
should be used.}

\item{...}{Arguments passed to \code{opt_func}.}

\item{data}{the \code{\link{list}} that was passed to
\code{\link{pedigree_ll_terms}} or \code{\link{pedigree_ll_terms_loadings}}.}

\item{scale_max}{the maximum value for the scale parameters. Sometimes, the
optimization method tends to find large scale parameters and get stuck.
Setting a maximum solves this.}

\item{sc_start}{starting value for the scale parameters. Use \code{NULL} if
you have no value to start with.}

\item{sc_start_invariant}{scale parameter(s) like sc_start. It is the value
that all individuals should have (i.e. not one that varies by individual).}
}
\value{
\code{pedmod_opt}: The output from the \code{opt_func} argument. Thus, if
\code{fix} is supplied then this is optimal values of only \code{par[-fix]}
with
\code{par[fix]} being fixed to the inputs. Thus, the length is only the
number of non-fixed parameters.

\code{pedmod_start}: A \code{list} with:
\itemize{
 \item par: the starting value.
 \item beta_no_rng: the fixed effects MLEs without random effects.
 \item logLik_no_rng: the log maximum likelihood without random effects.
 \item logLik_est: the likelihood at par.
}

\code{pedmod_start_loadings}: A \code{list} with:
\itemize{
 \item par: the starting value.
 \item beta_no_rng: the fixed effects MLEs without random effects.
 \item logLik_no_rng: the log maximum likelihood without random effects.
}
}
\description{
Optimizes \code{\link{eval_pedigree_ll}} and \code{\link{eval_pedigree_grad}}
using a passed optimization function.
}
\details{
\code{pedmod_start} and \code{pedmod_start_loadings}
yield starting values which can be used for
\code{pedmod_opt}. The methods are based on a heuristics.
}
\examples{
\donttest{
# we simulate outcomes with an additive genetic effect. The kinship matrix is
# the same for all families and given by
K <- matrix(c(
  0.5  , 0    , 0.25 , 0   , 0.25 , 0   , 0.125 , 0.125 , 0.125 , 0.125 ,
  0    , 0.5  , 0.25 , 0   , 0.25 , 0   , 0.125 , 0.125 , 0.125 , 0.125 ,
  0.25 , 0.25 , 0.5  , 0   , 0.25 , 0   , 0.25  , 0.25  , 0.125 , 0.125 ,
  0    , 0    , 0    , 0.5 , 0    , 0   , 0.25  , 0.25  , 0     , 0     ,
  0.25 , 0.25 , 0.25 , 0   , 0.5  , 0   , 0.125 , 0.125 , 0.25  , 0.25  ,
  0    , 0    , 0    , 0   , 0    , 0.5 , 0     , 0     , 0.25  , 0.25  ,
  0.125, 0.125, 0.25 , 0.25, 0.125, 0   , 0.5   , 0.25  , 0.0625, 0.0625,
  0.125, 0.125, 0.25 , 0.25, 0.125, 0   , 0.25  , 0.5   , 0.0625, 0.0625,
  0.125, 0.125, 0.125, 0   , 0.25 , 0.25, 0.0625, 0.0625, 0.5   , 0.25  ,
  0.125, 0.125, 0.125, 0   , 0.25 , 0.25, 0.0625, 0.0625, 0.25  , 0.5
), 10)

# simulates a data set.
#
# Args:
#   n_fams: number of families.
#   beta: the fixed effect coefficients.
#   sig_sq: the scale parameter.
sim_dat <- function(n_fams, beta = c(-1, 1, 2), sig_sq = 3){
  # setup before the simulations
  Cmat <- 2 * K
  n_obs <- NROW(K)
  Sig <- diag(n_obs) + sig_sq * Cmat
  Sig_chol <- chol(Sig)

  # simulate the data
  out <- replicate(
    n_fams, {
      # simulate covariates
      X <- cbind(`(Intercept)` = 1, Continuous = rnorm(n_obs),
                 Binary = runif(n_obs) > .5)

      # assign the linear predictor + noise
      eta <- drop(X \%*\% beta) + drop(rnorm(n_obs) \%*\% Sig_chol)

      # return the list in the format needed for the package
      list(y = as.numeric(eta > 0), X = X, scale_mats = list(Cmat))
    }, simplify = FALSE)

  # add attributes with the true values and return
  attributes(out) <- list(beta = beta, sig_sq = sig_sq)
  out
}

# simulate the data
set.seed(1)
dat <- sim_dat(100L)

# fit the model
ptr <- pedigree_ll_terms(dat, max_threads = 1L)
start <- pedmod_start(ptr = ptr, data = dat, n_threads = 1L)
fit <- pedmod_opt(ptr = ptr, par = start$par, n_threads = 1L, use_aprx = TRUE,
                  maxvls = 5000L, minvls = 1000L, abs_eps = 0, rel_eps = 1e-3)
fit$par # the estimate
-fit$value # the log maximum likelihood
start$logLik_no_rng # the log maximum likelihood without the random effects
}
}
\seealso{
\code{\link{pedmod_sqn}}.
}
