% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_lmms.R
\name{fit_lmms}
\alias{fit_lmms}
\title{Step 1 of PRC-LMM (estimation of the linear mixed models)}
\usage{
fit_lmms(y.names, fixefs, ranefs, long.data, surv.data, t.from.base,
  n.boots = 0, n.cores = 1, max.ymissing = 0.2, verbose = TRUE,
  seed = 123, control = list(opt = "optim", niterEM = 500, maxIter = 500))
}
\arguments{
\item{y.names}{character vector with the names of the
response variables which the LMMs have to be fitted to}

\item{fixefs}{fixed effects formula for the model, example:
\code{~ time}}

\item{ranefs}{random effects formula for the model,
specified using the representation of random effect
structures of the \code{R} package \code{nlme}}

\item{long.data}{a data frame with the longitudinal predictors,
comprehensive of a variable called \code{id} with the subject 
ids}

\item{surv.data}{a data frame with the survival data and (if 
relevant) additional baseline covariates. \code{surv.data} should at least
contain a subject id (called \code{id}), the time to event outcome  
(\code{time}), and binary event variable (\code{event})}

\item{t.from.base}{name of the variable containing time from 
baseline in \code{long.data}}

\item{n.boots}{number of bootstrap samples to be used in the
cluster bootstrap optimism correction procedure (CBOCP). If 0, no
bootstrapping is performed}

\item{n.cores}{number of cores to use to parallelize part of
the computations. If \code{ncores = 1} (default), 
no parallelization is done. Pro tip: you can use 
\code{parallel::detectCores()} to check how many 
cores are available on your computer}

\item{max.ymissing}{maximum proportion of subjects allowed to not have any
measurement of a longitudinal response variable. Default is 0.2}

\item{verbose}{if \code{TRUE} (default and recommended value), information
on the ongoing computations is printed in the console}

\item{seed}{random seed used for the bootstrap sampling. Default 
is \code{seed = 123}}

\item{control}{a list of control values to be passed to \code{lme} when fitting the
linear mixed models. By default, we set \code{opt = 'optim', niterEM = 500, maxIter = 500}. 
See \code{?nlme::lmeControl} for all possible arguments and values}
}
\value{
A list containing the following objects:
\itemize{
\item \code{call.info}: a list containing the following function
call information: \code{call}, \code{y.names}, \code{fixefs},
\code{ranefs};
\item \code{lmm.fits.orig}: a list with the LMMs fitted on the
original dataset (it should comprise as many LMMs as the elements
of \code{y.names} are);
\item \code{df.sanitized}: a sanitized version of the supplied 
\code{long.data} dataframe, without the
longitudinal measurements that are taken after the event
or after censoring;
\item \code{n.boots}: number of bootstrap samples;
\item \code{boot.ids}: a list with the ids of bootstrapped subjects 
(when \code{n.boots > 0});
\item \code{lmms.fits.boot}: a list of lists, which contains the LMMs fitted 
on each bootstrapped datasets (when \code{n.boots > 0}).
}
}
\description{
This function performs the first step for the estimation
of the PRC-LMM model proposed in Signorelli et al. (2021)
}
\examples{
# generate example data
set.seed(1234)
p = 4 # number of longitudinal predictors
simdata = simulate_prclmm_data(n = 100, p = p, p.relev = 2, 
             seed = 123, t.values = c(0, 0.2, 0.5, 1, 1.5, 2))
 
# specify options for cluster bootstrap optimism correction
# procedure and for parallel computing 
do.bootstrap = FALSE
# IMPORTANT: set do.bootstrap = TRUE to compute the optimism correction!
n.boots = ifelse(do.bootstrap, 100, 0)
more.cores = FALSE
# IMPORTANT: set more.cores = TRUE to parallelize and speed computations up!
if (!more.cores) n.cores = 1
if (more.cores) {
   # identify number of available cores on your machine
   n.cores = parallel::detectCores()
   if (is.na(n.cores)) n.cores = 8
}

# step 1 of PRC-LMM: estimate the LMMs
y.names = paste('marker', 1:p, sep = '')
step1 = fit_lmms(y.names = y.names, 
                 fixefs = ~ age, ranefs = ~ age | id, 
                 long.data = simdata$long.data, 
                 surv.data = simdata$surv.data,
                 t.from.base = t.from.base,
                 n.boots = n.boots, n.cores = n.cores)
# estimated betas and variances for the 3rd marker:
summary(step1, 'marker3', 'betas')
summary(step1, 'marker3', 'variances')
# usual T table:
summary(step1, 'marker3', 'tTable')
}
\references{
Signorelli, M. (2023). pencal: an R Package for the Dynamic 
Prediction of Survival with Many Longitudinal Predictors.
arXiv preprint: arXiv:2309.15600

Signorelli, M., Spitali, P., Al-Khalili Szigyarto, C, 
The MARK-MD Consortium, Tsonaka, R. (2021). 
Penalized regression calibration: a method for the prediction 
of survival outcomes using complex longitudinal and 
high-dimensional data. Statistics in Medicine, 40 (27), 6178-6196.
DOI: 10.1002/sim.9178
}
\seealso{
\code{\link{simulate_prclmm_data}},
\code{\link{summarize_lmms}} (step 2),
\code{\link{fit_prclmm}} (step 3),
\code{\link{performance_prc}}
}
\author{
Mirko Signorelli
}
