% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/eco.scape.R
\name{eco.scape}
\alias{eco.scape}
\title{eco.space scape simulation with a macro-ecological focus}
\usage{
eco.scape(tree, scape.size = 10, g.center = 1, wd.all = 0.2 * (scape.size
  + 1)^2, signal.center = TRUE, center.scale = 1, site.stoch.scale = 0,
  sd.center = 1, sd.range = 1, K = 100, extinction = FALSE,
  rho = NULL)
}
\arguments{
\item{tree}{\code{\link[ape:phylo]{phylo}} object; must have branch
lengths and be ultrametric}

\item{scape.size}{edge dimension of square landscape}

\item{g.center}{strength of phylogenetic signal in species range
centers. See \code{\link[ape:corBlomberg]{corBlomberg}},
1=brownian,<1=rates of evol accelerate, >1=rates decelerate.}

\item{wd.all}{niche width, larger values simulate broader range
sizes}

\item{signal.center}{simulate with phylosignal in range centers}

\item{center.scale}{adjust strength of phylogenetic attraction in
range centers independent of g.center}

\item{site.stoch.scale}{adjust strength of random variation in
species richness across sites}

\item{sd.center}{sd in \code{\link[stats:rnorm]{rnorm}}rnorm for
the range centers, increase to get more variation in center values
across species}

\item{sd.range}{sd in rnorm for the range sizes, increase to get
more variation in range sizes across gradients}

\item{K}{carrying capacity of a site in terms of maximum
individuals that can be present. Currently a constant value. Used
to scale the presence-absence matrix to include abundances.}

\item{extinction}{TRUE/FALSE can species on the tree go extinct on
the landscape? If the number of species present on the landscape
should equal the number of tips on the tree, choose FALSE. See
Details.}

\item{rho}{Grafen branch adjustment of phylogenetic tree see
\code{\link[ape:corGrafen]{corGrafen}}}
}
\value{
\item{Y}{presence/absence matrix}
\item{Yab}{abundance matrix}
\item{index}{spatial coordinates for X and Y (stacked columns)}
\item{X.joint}{full probabilities of species at sites, used to
construct Y}
\item{X1}{probabilities of species along gradient 1}
\item{X2}{probabilities of species along gradient 2}
\item{gradient1, gradient2}{environmental gradient values}
\item{nichewd}{average niche width of the assemblage}
\item{K}{carrying capacity of each cell}
\item{environ}{matrix depicting environmental values over the 2D
landscape}
\item{sppXs}{full probabilities of each species as an array
arranged in a scape.size X scape.size matr ix}
\item{V.phylo}{initial phylogenetic covariance matrix from tree,
output of vcv.phylo(tree, corr=T)}
\item{V.phylo.rho}{phylogenetic covariance matrix from tree scaled
by Grafen if rho is provided, other wise just an output of
vcv.phylo(tree, corr=T)}
\item{V.center}{scaled (by g.center) phylo covariance matrix used
in the simulations}
\item{bspp1}{species optima for gradient 1}
\item{bspp2}{pecies optima for gradient 2}
}
\description{
\code{eco.scape} is a modified version of the Helmus et al. method
implemented in \code{\link{scape}}. It produces phylogenetically
structured communities. It allows phylogenetic signals in niche
optima, but unlike \code{\link{scape}}, does not include the
ability to specify niche optima signal type (attraction/repulsion)
or phylogenetic signal in range size. Instead, the focus is on
having more control over the macroecological characteristics of the
resulting landscapes. In particular, eco.scape produces landscapes
with fixed mean range sizes, reasonable range size and abundance
distributions, and control over whether species present on a tree
must be present in the landscape.
}
\details{
Simulates a landscape with species (i.e., tree tips) distributions
dependent on a supplied phylogenetic tree. The amount and type of
structure is determened by the signal parameter
\code{g.center}. Parameters are based on an Ornstein-Uhlenbeck
model of evolution with stabilizing selection. Values of g=1
indicate no stabilizing selection and correspond to the Brownian
motion model of evolution; 01 corresponds to disruptive selection
where phylogenetic signal for the supplied tree is amplified. See
corBlomberg. Communities are simulated along two gradients where
the positions along those gradients, \code{g.center}, can exhibit
phylogenetic signal.

The function returns a landscape where the average range size is
equivalent to the wd.all parameter - in the scape function, this
parameter is not necessarily returned in the resulting
landscape. To do this, the probability of presence (th) that
returns the wd.all parameter is solved for. If there is no solution
that can produce the wd.all given, the error "Error in uniroot(f,
lower = 0, upper = max(X.), tol = 10^-200): f() values at end
points not of opposite sign" will occur. This seems to mostly arise
for extreme or unlikely parameter values (small species pools, low
carrying capacities). Try adjusting parameter values first.

The \code{extinction} parameter specifies whether all of the
species on the tree should be present in the final landscape. Some
species will have probabilities of presence less than those
required for presence. If extinctions is \code{TRUE}, these species
will not be present. If \code{FALSE}, these species will be present
in 1 site, that in which they have the highest probability of
presence.
}
\examples{
# Simulations
tree <- rcoal(64)

scape1 <- eco.scape(tree, scape.size=25, g.center=1,
    signal.center=FALSE, K=100, extinction=TRUE)
scape2 <- eco.scape(tree, scape.size=16, g.center=0.2,
    signal.center=TRUE, K=100, extinction=FALSE)
scape3 <- eco.scape(tree, scape.size=16, g.center=20,
    signal.center=TRUE, K=100, extinction=TRUE)

# Plotting distributions and landscape patterns
original_landscape <- scape1
abundmax <- original_landscape$K
PA_mat <- as.matrix(original_landscape$Y)
abund_mat <- original_landscape$Yab
site.size <- nrow(PA_mat)
species <- ncol(PA_mat)
mx <- original_landscape$gradient
env <- original_landscape$environ$env.gradient
par(mfrow=c(2,2), oma=c(0,0,2,0))
heatcol <- (colorRampPalette(c("yellow","red")))

image(matrix(env,sqrt(site.size),sqrt(site.size),byrow=TRUE),
    col=heatcol(max(env)),xaxt="n",yaxt="n",main="Env gradient")

image(matrix(rowSums(PA_mat),sqrt(site.size),sqrt(site.size),byrow=TRUE),
    col=heatcol(16),xaxt="n",yaxt="n",main="Species Richness")

hist(colSums(PA_mat),ylab="Number of species",xlab="Number of sites",
    main="Species Area Relationship",col="lightgrey")

hist(colSums(abund_mat),ylab="Number of species",xlab="Number of individuals",
    main="Species Abundance Relationship",col="lightgrey")
mtext("Env random, phy.signal=0.2, 32 species", outer=TRUE, side=3, cex=1.25)
}
\author{
Matt Helmus, Caroline Tucker, cosmetic edits by Will Pearse
}
\seealso{
\code{\link{scape}} \code{\link{sim.phy}} \code{\link{sim.meta}}
}

