% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_error.R
\name{simulate_error}
\alias{simulate_error}
\title{Simulate gas exchange data with measurement error}
\usage{
simulate_error(
  ph_out,
  chamber_pars,
  n = 1L,
  use_tealeaves = ("T_air" \%in\% colnames(ph_out))
)
}
\arguments{
\item{ph_out}{A data frame of output from \code{photo()} or \code{photosynthesis()}
with units.}

\item{chamber_pars}{A data frame with a single row of chamber parameters.
See Note below for table of required parameters.}

\item{n}{Integer. Number of replicated simulations per row of \code{ph_out}.}

\item{use_tealeaves}{Flag. The \strong{tealeaves} package uses a slightly
different equation to calculate the saturating water content of air as a
function temperature and pressure than LI-COR. If FALSE, the function uses
LI-COR's equation in the LI6800 manual. If TRUE, it uses the \strong{tealeaves}
function for internal consistency. The function attempts to guess whether
\code{ph_out} was run with \strong{tealeaves}, but this can be manually overridden by
providing a value for the argument.}
}
\value{
A data frame with \code{n * nrow(ph_out)} rows. It contains all the
original output in \code{ph_out} as well as a column \code{.rep} indicating replicate
number from 1 to \code{n}. Other new columns are assumed or measured chamber
parameters and 'measured' values estimated from synthetic data with
measurement error:\tabular{lll}{
   column name \tab assumed or derived? \tab description \cr
   \code{flow} \tab assumed \tab chamber flow rate \cr
   \code{leaf_area} \tab assumed \tab leaf area in chamber \cr
   \code{sigma_CO2_r} \tab assumed \tab standard deviation of measurement error in CO2_r \cr
   \code{sigma_CO2_s} \tab assumed \tab standard deviation of measurement error in CO2_s \cr
   \code{sigma_H2O_r} \tab assumed \tab standard deviation of measurement error in H2O_r \cr
   \code{sigma_H2O_s} \tab assumed \tab standard deviation of measurement error in H2O_s \cr
   \code{c_0} \tab derived \tab CO\eqn{_2} concentration before entering chamber [\eqn{\mu}mol / mol] \cr
   \code{w_i} \tab derived \tab Water vapor concentration within leaf [mmol / mol] \cr
   \code{w_a} \tab derived \tab Water vapor concentration in chamber [mmol / mol] \cr
   \code{w_0} \tab derived \tab Water vapor concentration before entering chamber [mmol / mol] \cr
   \code{g_tw} \tab derived \tab Leaf conductance to water vapor [mol/m\eqn{^2}/s] \cr
   \code{E_area} \tab derived \tab Evaporation rate per area [mmol/m\eqn{^2}/s] \cr
   \code{E} \tab derived \tab Total evaporation rate [mmol/s] \cr
   \code{CO2_r} \tab derived \tab CO\eqn{_2} concentration before entering chamber with measurement error [\eqn{\mu}mol / mol] \cr
   \code{CO2_s} \tab derived \tab CO\eqn{_2} concentration in chamber with measurement error [\eqn{\mu}mol / mol] \cr
   \code{H2O_s} \tab derived \tab Water vapor concentration in chamber with measurement error [mmol / mol] \cr
   \code{H2O_r} \tab derived \tab Water vapor concentration before entering chamber with measurement error [mmol / mol] \cr
   \code{E_meas} \tab derived \tab Total evaporation rate (measured) [mmol/s] \cr
   \code{E_area_meas} \tab derived \tab Evaporation rate per area (measured) [mmol/m\eqn{^2}/s] \cr
   \code{g_tw_meas} \tab derived \tab Leaf conductance to water vapor (measured) [mol/m\eqn{^2}/s] \cr
   \code{g_sc_meas} \tab derived \tab Stomatal conductance to CO\eqn{_2} (measured) [mol/m\eqn{^2}/s] \cr
   \code{g_tc_meas} \tab derived \tab Leaf conductance to CO\eqn{_2} (measured) [mol/m\eqn{^2}/s] \cr
   \code{A_meas} \tab derived \tab Net photosynthetic CO\eqn{_2} assimilation (measured) [\eqn{\mu}mol/m\eqn{^2}/s] \cr
   \code{C_i} \tab derived \tab Intercellular CO\eqn{_2} concentration (measured) [\eqn{\mu}mol/mol] \cr
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}
}
\note{
The required parameters for the \code{chamber_pars} argument are:
\itemize{
\item \code{flow} [\eqn{\mu}mol / s]: chamber flow rate
\item \code{leaf_area} [cm ^ 2]: leaf area in chamber
\item \code{sigma_CO2_s} [\eqn{\mu}mol / mol]: standard deviation of sample [CO\eqn{_2}] measurement error
\item \code{sigma_CO2_r} [\eqn{\mu}mol / mol]: standard deviation of reference [CO\eqn{_2}]
\item \code{sigma_H2O_s} [mmol / mol]: standard deviation of sample [H\eqn{_2}O] measurement error
\item \code{sigma_H2O_r} [mmol / mol]: standard deviation of sample [H\eqn{_2}O] measurement error
}

Units for \code{flow} and \code{leaf_area} should be provided; units are implied for sigma's but not necessary to specify because \code{rnorm()} drop units.

To evaluate the accuracy and precision of parameter estimation methods, it
may be useful to simulate data with realistic measurement error. This
function takes output from from \code{photo()} or \code{photosynthesis()} models, adds
measurement error in CO\eqn{_2} and H\eqn{_2}O concentrations, and calculates
parameter estimates with synthetic data. Currently, the function assumes a
simplified 1-dimensional CO\eqn{_2} and H\eqn{_2}O conductance model: zero
cuticular conductance, infinite boundary layer conductance, and infinite
airspace conductance. Other assumptions include:
\itemize{
\item chamber flow rate, leaf area, leaf temperature, and air pressure are known
without error
\item measurement error is normally distributed mean 0 and standard deviation
specified in \code{chamber_pars}
}

This function was designed with the LI-COR LI6800 instrument in mind, but in
principle applies to any open path gas exchange system.
}
\examples{
library(photosynthesis)

# Use photosynthesis() to simulate 'real' values
# `replace = ...` sets parameters to meet assumptions of `simulate_error()`
lp = make_leafpar(replace = list(
  g_sc = set_units(0.1, mol/m^2/s),
  g_uc = set_units(0, mol/m^2/s),
  k_mc = set_units(0, 1),
  k_sc = set_units(0, 1),
  k_uc = set_units(0, 1)
  ),
  use_tealeaves = FALSE)
   
 ep = make_enviropar(replace = list(
   wind = set_units(Inf, m/s)
 ), use_tealeaves = FALSE) 
 bp = make_bakepar()
 cs = make_constants(use_tealeaves = FALSE)
 
 chamber_pars = data.frame(
   flow = set_units(600, umol / s),
   leaf_area = set_units(6, cm ^ 2),
   sigma_CO2_s = 0.1,
   sigma_CO2_r = 0.1,
   sigma_H2O_s = 0.1,
   sigma_H2O_r = 0.1
 )
   
ph = photosynthesis(lp, ep, bp, cs, use_tealeaves = FALSE, quiet = TRUE) |>
  simulate_error(chamber_pars, n = 1L)
  
}
