% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/board_url.R
\name{board_url}
\alias{board_url}
\title{Use a vector of URLs as a board}
\usage{
board_url(
  urls,
  cache = NULL,
  use_cache_on_failure = is_interactive(),
  headers = NULL
)
}
\arguments{
\item{urls}{Identify available pins being served at a URL or set of URLs (see details):
\itemize{
\item Unnamed string: URL to a \link[=write_board_manifest]{manifest file}.
\item Named character vector: URLs to specific pins (does not support versioning).
\item Named list: URLs to pin version directories (supports versioning).
}}

\item{cache}{Cache path. Every board requires a local cache to avoid
downloading files multiple times. The default stores in a standard
cache location for your operating system, but you can override if needed.}

\item{use_cache_on_failure}{If the pin fails to download, is it ok to
use the last cached version? Defaults to \code{is_interactive()} so you'll
be robust to poor internet connectivity when exploring interactively,
but you'll get clear errors when the code is deployed.}

\item{headers}{Named character vector for additional HTTP headers (such as for
authentication). See \code{\link[=connect_auth_headers]{connect_auth_headers()}} for Posit Connect support.}
}
\description{
\code{board_url()} lets you build up a board from individual urls or a \link[=write_board_manifest]{manifest file}.

\code{board_url()} is read only.
}
\details{
The way \code{board_url()} works depends on the type of the \code{urls} argument:
\itemize{
\item Unnamed character scalar, i.e. \strong{a single URL} to a
\link[=write_board_manifest]{manifest file}: If the URL ends in a \code{/},
\code{board_url()} will look for a \verb{_pins.yaml} manifest. If the manifest
file parses to a named list, versioning is supported. If it parses to a
named character vector, the board will not support versioning.
\item \strong{Named character vector of URLs}: If the URLs end in a \code{/},
\code{board_url()} will look for a \code{data.txt} that provides metadata for the
associated pin. The easiest way to generate this file is to upload a pin
version directory created by \code{\link[=board_folder]{board_folder()}}. Versioning is not supported.
\item \strong{Named list}, where the values are character vectors of URLs and each
element of the vector refers to a version of the particular pin: If a
URL ends in a \code{/}, \code{board_url()} will look for a \code{data.txt} that
provides metadata. Versioning is supported.
}

Using a vector of URLs can be useful because \code{\link[=pin_download]{pin_download()}} and
\code{\link[=pin_read]{pin_read()}} will be cached; they'll only re-download the data if it's
changed from the last time you downloaded it (using the tools of
\href{https://developer.mozilla.org/en-US/docs/Web/HTTP/Caching}{HTTP caching}).
You'll also be protected from the vagaries of the internet; if a fresh
download fails, you'll get the previously cached result with a warning.

Using a \link[=write_board_manifest]{manifest file} can be useful because you
can serve a board of pins and allow collaborators to access the board
straight from a URL, without worrying about board-level storage details.
Some examples are provided in \code{vignette("using-board-url")}.
}
\section{Authentication for \code{board_url()}}{
The \code{headers} argument allows you to pass authentication details or other
HTTP headers to the board, such as for a Posit Connect vanity URL that is
not public (see \code{\link[=board_connect_url]{board_connect_url()}}) or a private GitHub repo.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gh_pat_auth <- c(
  Authorization = paste("token", "github_pat_XXXX")
)
board <- board_url(
  "https://raw.githubusercontent.com/username/repo/main/path/to/pins",
  headers = gh_pat_auth
)

board \%>\% pin_list()
}\if{html}{\out{</div>}}
}

\examples{
\dontshow{if (!pins:::is_cran_check()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
github_raw <- function(x) paste0("https://raw.githubusercontent.com/", x)

## with a named vector of URLs to specific pins:
b1 <- board_url(c(
  files = github_raw("rstudio/pins-r/main/tests/testthat/pin-files/"),
  rds = github_raw("rstudio/pins-r/main/tests/testthat/pin-rds/"),
  raw = github_raw("rstudio/pins-r/main/tests/testthat/pin-files/first.txt")
))

b1 \%>\% pin_read("rds")
b1 \%>\% pin_browse("rds", local = TRUE)

b1 \%>\% pin_download("files")
b1 \%>\% pin_download("raw")

## with a manifest file:
b2 <- board_url(github_raw("rstudio/pins-r/main/tests/testthat/pin-board/"))
b2 \%>\% pin_list()
b2 \%>\% pin_versions("y")
\dontshow{\}) # examplesIf}
}
\seealso{
Other boards: 
\code{\link{board_connect_url}()},
\code{\link{board_connect}()},
\code{\link{board_folder}()}
}
\concept{boards}
