context("tidy_labels_levels")


mtcars2 <- mtcars
Hmisc::label(mtcars2$mpg) <- "Gas Mileage"
Hmisc::label(mtcars2$qsec) <- "Quarter Mile Time"
Hmisc::label(mtcars2$am) <- "Transmission"
Hmisc::label(mtcars2$wt) <- "Weight"
Hmisc::label(mtcars2$gear) <- "Gears"

#* Basic Output for a model with no interactions
#* Note: numeric_level has no impact as there are no
#*       interactions involving numeric variables.

fit <- lm(mpg ~ qsec + factor(am) + wt + factor(gear), data = mtcars2)

test_that("output for a model with no interactions",
{
  expect_that(tidy_levels_labels(fit, 
                                 descriptors = c("term", "term_plain", "label", 
                                                 "level", "level_detail"),
                                 numeric_level = "term"),
              not(throws_error()))
})
 
#* Assign factors ahead of the model. This allows 
#* the user to determine the levels that display.
#* Compare the output for 'am' with the output for 'gear'

mtcars2$am <- factor(mtcars2$am, 0:1, c("Automatic", "Manual"))
Hmisc::label(mtcars2$am) <- "Transmission" # Label was lost in variable conversion
fit <- lm(mpg ~ qsec + am + wt + factor(gear), data = mtcars2)

test_that("factor labels from data are preserved",
{
  expect_equal(tidy_levels_labels(fit, 
                                 descriptors = c("term", "term_plain", "label", 
                                                 "level", "level_detail"),
                                 numeric_level = "term")$level[3],
               "Manual")
})
 
 
#* Include an interaction between a factor and numeric.

fit <- lm(mpg ~ qsec + am * wt + factor(gear), data = mtcars2)


test_that("use an interaction",
{
  expect_that(tidy_levels_labels(fit, 
                                 descriptors = c("term", "term_plain", "label", 
                                                 "level", "level_detail"),
                                 numeric_level = "term"),
              not(throws_error()))
})
 
#* Now observe how 'level' and 'level_detail' change 
#* in the interaction terms as we choose different 
#* values for 'numeric_level'

test_that("use an interaction with term_plain",
{
  expect_that(tidy_levels_labels(fit, 
                                 descriptors = c("term", "term_plain", "label", 
                                                 "level", "level_detail"),
                                 numeric_level = "term_plain"),
              not(throws_error()))
})
   
test_that("use an interaction with label",
{
  expect_that(tidy_levels_labels(fit, 
                                 descriptors = c("term", "term_plain", "label", 
                                                 "level", "level_detail"),
                                 numeric_level = "label"),
              not(throws_error()))
})

test_that("tidy_labels_levels with bad descriptor list",
{
  expect_that(tidy_levels_labels(fit,
                                 descriptors = c("term_boring"),
                                 numeric_level = "term"),
              throws_error())
})

test_that("tidy_labels_levels with bad numeric_level",
{
  expect_that(tidy_levels_labels(fit,
                                 descriptors = c("term"),
                                 numeric_level = "term_boring"),
              throws_error())
})