#' Highlight tokens in fulltext.
#' 
#' Highlight tokens by term or cpos in a html document generated by
#' read/html.
#' 
#' @param .Object either a \code{"partition"} or \code{"character"} object
#' @param html character vector with a website
#' @param highlight a \code{"list"} of character vectors, the names need to provide the colors
#' @param tooltips a \code{"list"} of character vectors, all names need to be included in lists of the highlight-list
#' @param cqp logical, whether to use CQP syntax
#' @param ... further parameters
#' @name highlight
#' @rdname highlight
setGeneric("highlight", function(.Object, ...) standardGeneric("highlight"))


#' @rdname highlight
setMethod("highlight", "partition", function(.Object, html, highlight=list(), cqp=is.cqp, tooltips=NULL){
  for (color in names(highlight)){
    # tooltipTags <- .makeTooltipTags(color, tooltips)
    for (x in highlight[[color]]){
      hitCpos <- cpos(.Object, query=x, cqp=cqp)
      if (!is.null(hitCpos)){
        for (i in c(1:nrow(hitCpos))){
          for (j in c(hitCpos[i,1]:hitCpos[i,2])){
            html <- gsub(
              paste('<span id="', j, '">(.*?)</span>', sep=""),
              paste(
                paste('<span id="', j, '"><span style="background-color:', color, '">', sep=""),
                # tooltipTags[["start"]], "\\1", tooltipTags[["end"]], '</span></span>', sep=""
              ),
              html
            )
          }
        }
      }
    }
  }
  html
})

#' @rdname highlight
setMethod("highlight", "character", function(.Object, highlight = list(), tooltips = NULL){
  if (requireNamespace("xml2", quietly = TRUE)){
    doc <- xml2::read_html(.Object)
    
    for (color in names(highlight)){
      
      # tooltipTags <- .makeTooltipTags(color, tooltips)
      
      lapply(
        highlight[[color]],
        function(x){
          if (is.numeric(x)){
            node <- xml2::xml_find_first(doc, xpath = sprintf('//span[@id="%d"]', x))
            xml2::xml_set_attr(node, attr = "style", value = sprintf("background-color:%s", color))
            if (color %in% names(tooltips)){
              xml2::xml_add_child(.x = node, .value = node, copy = TRUE)
              xml2::xml_text(node) <- ""
              xml2::xml_name(node) <- "a"
              xml2::xml_attrs(node) <- c(href = "#", class = "tooltip", "data-tooltip" = "this is a tooltip")
            }
          } else {
            nodes <- xml2::xml_find_all(doc, xpath = sprintf('//span[@token="%s"]', x))
            lapply(
              nodes,
              function(node){
                xml2::xml_set_attr(node, attr = "style", value = sprintf("background-color:%s", color))
              }
            )
          }
          NULL
        }
      )
    }
    
  } else {
      stop("package 'xml2' needs to be installed for highlighting using cpos/ids")
    }
  as.character(doc)
})

#' @rdname highlight
setMethod("highlight", "html", function(.Object, highlight = list(), tooltips = NULL){
  htmltools::HTML(
    highlight(as.character(.Object), highlight = highlight, tooltips = tooltips)
  )
})
