% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/skeleton_spec.R
\docType{methods}
\name{skeleton specification}
\alias{skeleton specification}
\alias{vectorfield}
\alias{map}
\title{The deterministic skeleton of a model}
\usage{
vectorfield(f)

map(f, delta.t = 1)
}
\arguments{
\item{f}{procedure for evaluating the deterministic skeleton
This can be a C snippet, an \R function, or the name of a native routine in a dynamically linked library.}

\item{delta.t}{positive numerical value; the size of the discrete time step corresponding to an application of the map}
}
\description{
Specification of the deterministic skeleton.
}
\details{
The skeleton is a dynamical system that expresses the central tendency of the unobserved Markov state process.
As such, it is not uniquely defined, but can be both interesting in itself and useful in practice.
In \pkg{pomp}, the skeleton is used by \code{\link{trajectory}} and \code{\link{traj_objfun}}.

If the state process is a discrete-time stochastic process, then the skeleton is a discrete-time map.
To specify it, provide \preformatted{
  skeleton = map(f, delta.t)}
to \code{pomp}, where \code{f} implements the map and \code{delta.t} is the size of the timestep covered at one map iteration.

If the state process is a continuous-time stochastic process, then the skeleton is a vectorfield (i.e., a system of ordinary differential equations).
To specify it, supply \preformatted{
  skeleton = vectorfield(f)}
to \code{pomp}, where \code{f} implements the vectorfield, i.e., the right-hand-size of the differential equations.

In either case, \code{f} can be furnished either as a C snippet (the preferred choice), or an \R function.
\link[=Csnippet]{General rules for writing C snippets can be found here}.
In writing a \code{skeleton} C snippet, be aware that:
\enumerate{
  \item For each state variable, there is a corresponding component of the deterministic skeleton.
  The goal of such a snippet is to compute all the components.
  \item When the skeleton is a map, the component corresponding to state variable \code{x} is named \code{Dx} and is the new value of \code{x} after one iteration of the map.
  \item When the skeleton is a vectorfield, the component corresponding to state variable \code{x} is named \code{Dx} and is the value of \eqn{dx/dt}.
  \item As with the other C snippets, all states, parameters and covariates, as well as the current time, \code{t}, will be defined in the context within which the snippet is executed.
  \item \bold{NB:} When the skeleton is a map, the duration of the timestep will \bold{not} be defined in the context within which the snippet is executed.
  When the skeleton is a vectorfield, of course, no timestep is defined.
  In this regard, C snippets for the skeleton and rprocess components differ.
}
The tutorials on the \href{https://kingaa.github.io/pomp/}{package website} give some examples.

If \code{f} is an \R function, its arguments should be taken from among the state variables, parameters, covariates, and time.
It must also take the argument \sQuote{\code{...}}.
As with the other basic components, \code{f} may take additional arguments, provided these are passed along with it in the call to \code{pomp}.
The function \code{f} must return a numeric vector of the same length as the number of state variables, which contains the value of the map or vectorfield at the required point and time.
}
\section{Default behavior}{

The default \code{skeleton} is undefined.
It will yield missing values (\code{NA}) for all state variables.
}

\section{Note for Windows users}{


Some Windows users report problems when using C snippets in parallel computations.
These appear to arise when the temporary files created during the C snippet compilation process are not handled properly by the operating system.
To circumvent this problem, use the \code{\link[=pomp]{cdir}} and \code{\link[=pomp]{cfile}} options to cause the C snippets to be written to a file of your choice, thus avoiding the use of temporary files altogether.
}

\examples{
\donttest{
  ## Starting with an existing pomp object,
  ## e.g., the continuous-time Verhulst-Pearl model,

  verhulst() -> po
  
  ## we add or change the deterministic skeleton
  ## using the 'skeleton' argument in any 'pomp'
  ## elementary or estimation function
  ## (or in the 'pomp' constructor itself).
  ## Here, we pass the skeleton specification
  ## to 'trajectory' as an R function.
  ## Since this is a continuous-time POMP, the
  ## skeleton is a vectorfield.

  po |>
    trajectory(
      skeleton=vectorfield(
        function(r, K, n, ...) {
          c(n=r*n*(1-n/K))
        }
      ),
      format="data.frame"
    ) -> traj

  ## We can also pass it as a C snippet:

  po |>
    traj_objfun(
      skeleton=vectorfield(Csnippet("Dn=r*n*(1-n/K);")),
      paramnames=c("r","K"),
      statenames="n"
    ) -> ofun

  ofun()

  ## For a discrete-time POMP, the deterministic skeleton
  ## is a map.  For example,

  gompertz() -> po

  po |>
    traj_objfun(
      skeleton=map(
        Csnippet("
          double dt = 1.0;
          double s = exp(-r*dt);
          DX = pow(K,(1-s))*pow(X,s);"
        ), delta.t=1
      ),
      paramnames=c("r","K"),
      statenames=c("X")
    ) -> ofun

  ofun()

}
}
\seealso{
\code{\link{skeleton}}

More on implementing POMP models: 
\code{\link{Csnippet}},
\code{\link{accumulator variables}},
\code{\link{basic components}},
\code{\link{betabinomial}},
\code{\link{covariates}},
\code{\link{distributions}},
\code{\link{dmeasure specification}},
\code{\link{dprocess specification}},
\code{\link{emeasure specification}},
\code{\link{parameter transformations}},
\code{\link{pomp-package}},
\code{\link{pomp}},
\code{\link{prior specification}},
\code{\link{rinit specification}},
\code{\link{rmeasure specification}},
\code{\link{rprocess specification}},
\code{\link{transformations}},
\code{\link{userdata}},
\code{\link{vmeasure specification}}

More on methods for deterministic process models: 
\code{\link{flow}()},
\code{\link{skeleton}()},
\code{\link{trajectory matching}},
\code{\link{trajectory}()}
}
\concept{deterministic methods}
\concept{implementation information}
