#' @title
#' Constructing a component-wise poset of binary vectors.
#'
#' @description
#' Constructs a component-wise poset, starting from a collection of binary variables.
#'
#' @param variables A vector of character strings (the names of the input binary variables).
#'
#' @return
#' An object of S4 class `BinarVariablePOSet` (subclass of `POSet`).
#'
#' @details
#' Given \eqn{k} input binary variables, the function produces a poset \eqn{(V,\leq_{cmp})}, where \eqn{V} is the set of \eqn{2^k} binary vectors built from the variables, and \eqn{\leq_{cmp}} is the component-wise order.
#'
#' @examples
#' vrbs <- c("var1", "var2", "var3")
#' binPoset <-  BinaryVariablePOSet(variables = vrbs)
#'
#' @name BinaryVariablePOSet
#' @export BinaryVariablePOSet
BinaryVariablePOSet <- function(variables) {
  if (!is.character(variables))
    stop("variables must be character array")
  if (length(unique(variables)) != length(variables))
    stop("variables contains duplicated values")
  tryCatch({
    ptr <- .Call("_BuildBinaryVariablePOSet", variables)
    result <- methods::new("BinaryVariablePOSet", ptr = ptr)
    return (result)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
