#' @title Fuzzy Separation matrix computation with Product t-norm and Probabilistic-sum t-conorm
#'
#' @description Starting from a poset dominance matrix, computes fuzzy Separation matrices by using Product t-norm and Probabilistic-sum t-conorm
#'
#' @param dom square matrix representing the dominance degrees between pairs of poset elements. Columns and rows names of `dom`
#' are interpreted as the labels of the poset elements. `dom` can be computed by using functions such as `BLSDominance`,
#' `BubleyByesMRP` and `ExactMRP`.
#'
#' @param type type of fuzzy separation to be computed. Possible choices are:
#' "symmetric", "asymmetricLower", "asymmetricUpper", "vertical", "horizontal".
#' For details on the definition of symmetric, asymmetric, vertical and horizontal separations see Fattore et al. (2024).
#'
#' @param ... additional types of fuzzy separations to be computed. Possible choices are:
#' "symmetric", "asymmetricLower", "asymmetricUpper", "vertical", "horizontal".
#'
#' @return list of required fuzzy separation matrices.
#'
#' @references Fattore, M., De Capitani, L., Avellone, A., and Suardi, A. (2024).
#' A fuzzy posetic toolbox for multi-criteria evaluation on ordinal data systems.
#' Annals of Operations Research, https://doi.org/10.1007/s10479-024-06352-3.
#'
#' @examples
#' el <- c("a", "b", "c", "d")
#'
#' dom_list <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = el, dom = dom_list)
#'
#' BLS <- BLSDominance(pos)
#'
#' FSep <- FuzzySeparationProbabilistic(BLS, type="symmetric", "asymmetricLower", "vertical")
#'
#' @name FuzzySeparationProbabilistic
#' @export FuzzySeparationProbabilistic
FuzzySeparationProbabilistic <- function(dom, type, ...) {
  SeparationTypes <- c("symmetric", "asymmetricLower", "asymmetricUpper", "vertical", "horizontal")
  SeparationTypesC <- c("symmetric", "asymmetricLower", "asymmetricUpper")
  if (!is.matrix(dom)) {
    stop("dom must be a square matrix")
  }
  if (ncol(dom) != nrow(dom)) {
    stop("dom must be a square matrix")
  }
  if (!(type %in% SeparationTypes)) {
    stop("type TO_DO")
  }
  parameter_list = list(type)
  for(t in list(...)) {
    if (!(t %in% SeparationTypes)) {
      stop(".... TO_DO")
    }
    parameter_list[[length(parameter_list)+1]] = t
  }

  functions_list <- parameter_list
  if (SeparationTypes[4] %in% parameter_list) {
    functions_list[[length(functions_list)+1]] = SeparationTypes[2]
    functions_list[[length(functions_list)+1]] = SeparationTypes[3]
  }
  if (SeparationTypes[5] %in% parameter_list) {
    functions_list[[length(functions_list)+1]] = SeparationTypes[1]
    functions_list[[length(functions_list)+1]] = SeparationTypes[2]
    functions_list[[length(functions_list)+1]] = SeparationTypes[3]
  }
  functions_list <- intersect(functions_list, SeparationTypesC)

  norm <- "product"
  conorm <- ""

  tryCatch({
    result <- .Call("_FuzzySeparation", dom, norm, conorm, functions_list)
    if ("vertical" %in% parameter_list) {
      vertical <- abs(result[["asymmetricLower"]] - result[["asymmetricUpper"]])
      result[["vertical"]] = vertical
    }
    if ("horizontal" %in% parameter_list) {
      horizzotal <- result[["symmetric"]] - (abs(result[["asymmetricLower"]] - result[["asymmetricUpper"]]))
      result[["horizontal"]] = horizzotal
    }
    if ("symmetric" %in% parameter_list) {
      horizzotal <- result[["symmetric"]] - (abs(result[["asymmetricLower"]] - result[["asymmetricUpper"]]))
      result[["horizontal"]] = horizzotal
    }
    result <- result[unlist(parameter_list)]
    return (result)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
