% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LEGet.R
\name{LEGet}
\alias{LEGet}
\title{Generates linear extensions of a given poset, by using a linear extension generator}
\usage{
LEGet(
  generator,
  from_start = TRUE,
  n = NULL,
  error = NULL,
  output_every_sec = NULL
)
}
\arguments{
\item{generator}{The linear extension generator built by function \code{\link[=LEGenerator]{LEGenerator()}} or \code{\link[=LEBubleyDyer]{LEBubleyDyer()}}.
If \code{LEGenerator} is used, \code{n} linear extensions of the poset are generated,
according to a deterministic generation order, consistent with the algorithm given in Habib M, Medina R, Nourine L and Steiner G (2001).
If \code{LEBubleyDyer} is used, the proper number of linear extensions is randomly
sampled by using the Bubley and Dyer (1999) procedure, based on parameter \code{n} and \code{error}.}

\item{from_start}{Logical value indicating whether the linear extensions generator should be reset or not.
If \code{from_start=FALSE}, linear extensions generation starts from the last linear extension generated
in the previous \code{LEGet} call.
If \code{from_start=TRUE}, previous \code{LEGet} calls do not impact on the new linear extensions generation.

In  more details, when \code{generator} is created via \code{\link[=LEGenerator]{LEGenerator()}}, linear extensions are built after a deterministic
rule that fixes the generation order. In this case, if \code{from_start=TRUE} the generation starts from the first
linear extension in the generation order; if \code{from_start=FALSE}, the first linear extension generated is the successor,
in the prefixed generation order, to the last one generated by the previous \code{LEGet} call. This allows
the user to generate the set of all linear extensions of a poset in subsequent \code{LEGet} calls, so as to keep control
on computational
times and memory space.

When \code{generator} is created via \code{\link[=LEBubleyDyer]{LEBubleyDyer()}}, linear extensions are generated according
to the Bubley-Dyer sampling procedure that, starting from a randomly generated linear extension,
produces a sequence of linear orders in which the \eqn{k}-th one is obtained by a proper
random modification of the \eqn{(k-1)}-th one.
In this case, if \code{from_start=TRUE}, the first linear extension is chosen at random; if \code{from_start=FALSE},
it is obtained by randomly modifying the last one, generated in the previous \code{LEGet} call.}

\item{n}{number of linear extensions to be generated.}

\item{error}{A real number in the interval \eqn{(0,1)} representing the "distance" from uniformity
in the sampling distribution of linear extensions.
This parameter is used only when \code{generator} is of class  \code{BubleyDyerGenerator}. It determines the number
of linear extensions to be genrated, in order to achieve the desired "distance" from uniformity,
in the sampling distribution of linear extensions.
According to Bubley and Dyer (1999), if \verb{error=}\eqn{\epsilon} and \eqn{E} is the
number of elements in the poset, then the number \eqn{n_\epsilon} of linear
extensions to be sampled is given by

\eqn{n_\epsilon=E^4(\ln(E))^2+E^3\ln(E)\ln(\epsilon^{-1})}.

If both arguments \code{n} and \code{error} are specified by the user, the number of linear extensions actually generated
is \code{n}.}

\item{output_every_sec}{integer specifying a time interval (in seconds).
By specifying this argument, during the execution of \code{LEGet}, the number of linear extensions
actually generated is printed on the R-Console, every \code{output_every_sec}seconds.}
}
\value{
A matrix whose columns reports the generated linear extensions
}
\description{
Generates the linear extensions of a poset, by using a linear extension generator
built by functions \code{LEGenerator} and \code{LEBubleyDyer}.
}
\examples{
el <- c("a", "b", "c", "d", "e")

dom <- matrix(c(
  "a", "b",
  "c", "b",
  "c", "d"
), ncol = 2, byrow = TRUE)

pos <- POSet(elements = el, dom = dom)

LEgen <- LEGenerator(pos)
LEmatrix <- LEGet(LEgen)

LEgen <- LEGenerator(pos)
LEmatrix_first <- LEGet(LEgen, n=10)
LEmatrix_second <- LEGet(LEgen, from_start=FALSE)
LEmatrix <- cbind(LEmatrix_first,LEmatrix_second)

#Randomly generate n=30 linear extensions from the poset
LEgen <- LEBubleyDyer(pos)
LEmatrix <- LEGet(LEgen, n=30)

#Randomly generate n=60 linear extensions from the poset in two steps
LEgen <- LEBubleyDyer(pos)
LEmatrix_first <- LEGet(LEgen, n=30)
LEmatrix_second <- LEGet(LEgen, n=30, from_start=FALSE)
LEmatrix <- cbind(LEmatrix_first,LEmatrix_second)

#Generates linear extensions from the poset with precision=1
LEgen <- LEBubleyDyer(pos)
LEmatrix <- LEGet(LEgen, error=0.002)

}
\references{
Bubley, R., Dyer, M. (1999). Faster random generation of linear extensions.
Discrete Mathematics, 201, 81-88. https://doi.org/10.1016/S0012-365X(98)00333-1

Habib M, Medina R, Nourine L and Steiner G (2001). Efficient algorithms on distributive lattices.
Discrete Applied Mathematics, 110, 169-187. https://doi.org/10.1016/S0166-218X(00)00258-4.
}
