% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dosing_optim.R
\name{poso_time_cmin}
\alias{poso_time_cmin}
\title{Predict time to a selected trough concentration}
\usage{
poso_time_cmin(
  dat = NULL,
  prior_model = NULL,
  tdm = FALSE,
  target_cmin,
  dose = NULL,
  endpoint = "Cc",
  estim_method = "map",
  nocb = FALSE,
  p = NULL,
  greater_than = TRUE,
  from = 0.2,
  last_time = 72,
  add_dose = NULL,
  interdose_interval = NULL,
  duration = 0,
  indiv_param = NULL
)
}
\arguments{
\item{dat}{Dataframe. An individual subject dataset following the
structure of NONMEM/rxode2 event records.}

\item{prior_model}{A \code{posologyr} prior population pharmacokinetics
model, a list of six objects.}

\item{tdm}{A boolean. If \code{TRUE}: computes the predicted time to reach the
target trough concentration (Cmin) following the last event from \code{dat},
and using Maximum A Posteriori estimation. If \code{FALSE} : performs the
estimation for a simulated scenario defined by the remaining parameters.}

\item{target_cmin}{Numeric. Target trough concentration (Cmin).}

\item{dose}{Numeric. Dose administered.}

\item{endpoint}{Character. The endpoint of the prior model to be optimised
for. The default is "Cc", which is the central concentration.}

\item{estim_method}{A character string. An estimation method to be used for
the individual parameters. The default method "map" is the Maximum A
Posteriori estimation, the method "prior" simulates from the prior
population model, and "sir" uses the Sequential Importance Resampling
algorithm to estimate the a posteriori distribution of the individual
parameters. This argument is ignored if \code{indiv_param} is provided.}

\item{nocb}{A boolean. For time-varying covariates: the next observation
carried backward (nocb) interpolation style, similar to NONMEM.  If
\code{FALSE}, the last observation carried forward (locf) style will be used.
Defaults to \code{FALSE}.}

\item{p}{Numeric. The proportion of the distribution of cmin to consider for
the estimation. Mandatory for \code{estim_method=sir}.}

\item{greater_than}{A boolean. If \code{TRUE}: targets a time leading to a
proportion \code{p} of the cmins to be greater than \code{target_cmin}.
Respectively, lower if \code{FALSE}.}

\item{from}{Numeric. Starting time for the simulation of the
individual time-concentration profile. The default value is
0.2}

\item{last_time}{Numeric. Ending time for the simulation of the
individual time-concentration profile. The default value is
72.}

\item{add_dose}{Numeric. Additional doses administered at
inter-dose interval after the first dose. Optional.}

\item{interdose_interval}{Numeric. Time for the inter-dose interval
for multiple dose regimen. Must be provided when add_dose is used.}

\item{duration}{Numeric. Duration of infusion, for zero-order
administrations.}

\item{indiv_param}{Optional. A set of individual parameters : THETA,
estimates of ETA, and covariates.}
}
\value{
A numeric time to the selected trough concentration, from the
time of administration.
}
\description{
Predicts the time needed to reach a selected trough concentration
(Cmin) given a population pharmacokinetic model, a set of individual
parameters, a dose, and a target Cmin.
}
\examples{
rxode2::setRxThreads(1) # limit the number of threads

# model
mod_run001 <- list(
ppk_model = rxode2::rxode({
  centr(0) = 0;
  depot(0) = 0;

  TVCl = THETA_Cl;
  TVVc = THETA_Vc;
  TVKa = THETA_Ka;

  Cl = TVCl*exp(ETA_Cl);
  Vc = TVVc*exp(ETA_Vc);
  Ka = TVKa*exp(ETA_Ka);

  K20 = Cl/Vc;
  Cc = centr/Vc;

  d/dt(depot) = -Ka*depot;
  d/dt(centr) = Ka*depot - K20*centr;
  d/dt(AUC) = Cc;
}),
error_model = function(f,sigma) {
  dv <- cbind(f,1)
  g <- diag(dv\%*\%sigma\%*\%t(dv))
  return(sqrt(g))
},
theta = c(THETA_Cl=4.0, THETA_Vc=70.0, THETA_Ka=1.0),
omega = lotri::lotri({ETA_Cl + ETA_Vc + ETA_Ka ~
    c(0.2,
      0, 0.2,
      0, 0, 0.2)}),
sigma = lotri::lotri({prop + add ~ c(0.05,0.0,0.00)}))
# df_patient01: event table for Patient01, following a 30 minutes intravenous
# infusion
df_patient01 <- data.frame(ID=1,
                        TIME=c(0.0,1.0,14.0),
                        DV=c(NA,25.0,5.5),
                        AMT=c(2000,0,0),
                        EVID=c(1,0,0),
                        DUR=c(0.5,NA,NA))
# predict the time needed to reach a concentration of 2.5 mg/l
# after the administration of a 2500 mg dose over a 30 minutes
# infusion
poso_time_cmin(dat=df_patient01,prior_model=mod_run001,
dose=2500,duration=0.5,from=0.5,target_cmin=2.5)

}
