% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pre.R
\name{pre}
\alias{pre}
\title{Derive a prediction rule ensemble}
\usage{
pre(
  formula,
  data,
  family = gaussian,
  use.grad = TRUE,
  weights,
  type = "both",
  sampfrac = 0.5,
  maxdepth = 3L,
  learnrate = 0.01,
  mtry = Inf,
  ntrees = 500,
  confirmatory = NULL,
  removecomplements = TRUE,
  removeduplicates = TRUE,
  winsfrac = 0.025,
  normalize = TRUE,
  standardize = FALSE,
  ordinal = TRUE,
  nfolds = 10L,
  tree.control,
  tree.unbiased = TRUE,
  verbose = FALSE,
  par.init = FALSE,
  par.final = FALSE,
  sparse = FALSE,
  ...
)
}
\arguments{
\item{formula}{a symbolic description of the model to be fit of the form 
\code{y ~ x1 + x2 + ...+ xn}. Response (left-hand side of the formula) 
should be of class numeric (for \code{family = "gaussian"} or
\code{"mgaussian"}), integer (for \code{family = "poisson"}), factor (for 
\code{family = "binomial"} or \code{"multinomial"}). See Examples below. 
Note that the minus sign (\code{-}) may not be used in the formula to omit
the intercept or variables in \code{data}, and neither should  \code{+ 0} 
be used to omit the intercept. To omit the intercept from the final ensemble, 
add \code{intercept = FALSE} to the call (although omitting the intercept from
the final ensemble will only very rarely be appropriate). To omit variables 
from the final ensemble, make sure they are excluded from \code{data}.}

\item{data}{\code{data.frame} containing the variables in the model. Response 
must be of class \code{factor} for classification, \code{numeric} for (count) 
regression, \code{Surv} for survival regression. Input variables must be of 
class numeric, factor or ordered factor. Otherwise, \code{pre} will attempt
to recode.}

\item{family}{specifies a glm family object. Can be a character string (i.e., 
\code{"gaussian"}, \code{"binomial"}, \code{"poisson"}, \code{"multinomial"}, 
\code{"cox"} or \code{"mgaussian"}), or a corresponding family object 
(e.g., \code{gaussian}, \code{binomial} or \code{poisson}, see 
\code{\link[stats]{family}}). Specification of argument \code{family} is 
strongly advised but not required. If \code{family} is not specified, 
Otherwise, the program will try to make an informed guess, based on the 
class of the response variable specified in \code{formula}. als see Examples 
below.}

\item{use.grad}{logical. Should gradient boosting with regression trees be
employed when \code{learnrate > 0}? If \code{TRUE}, use trees fitted by 
\code{\link[partykit]{ctree}} or \code{\link[rpart]{rpart}} as in Friedman 
(2001), but without the line search. If \code{use.grad = FALSE}, 
\code{\link[partykit]{glmtree}} instead of \code{\link[partykit]{ctree}} 
will be employed for rule induction, yielding longer computation times, 
higher complexity, but possibly higher predictive accuracy. See Details for 
supported combinations of \code{family}, \code{use.grad} and \code{learnrate}.}

\item{weights}{optional vector of observation weights to be used for 
deriving the ensemble.}

\item{type}{character. Specifies type of base learners to include in the 
ensemble. Defaults to \code{"both"} (initial ensemble will include both rules 
and linear functions). Other option are \code{"rules"} (prediction 
rules only) or \code{"linear"} (linear functions only).}

\item{sampfrac}{numeric value \eqn{> 0} and \eqn{\le 1}. Specifies
the fraction of randomly selected training observations used to produce each 
tree. Values \eqn{< 1} will result in sampling without replacement (i.e., 
subsampling), a value of 1 will result in sampling with replacement 
(i.e., bootstrap sampling). Alternatively, a sampling function may be supplied, 
which should take arguments \code{n} (sample size) and \code{weights}.}

\item{maxdepth}{positive integer. Maximum number of conditions in rules. 
If \code{length(maxdepth) == 1}, it specifies the maximum depth of 
of each tree grown. If \code{length(maxdepth) == ntrees}, it specifies the
maximum depth of every consecutive tree grown. Alternatively, a random
sampling function may be supplied, which takes argument \code{ntrees} and 
returns integer values. See also \code{\link{maxdepth_sampler}}.}

\item{learnrate}{numeric value \eqn{> 0}. Learning rate or boosting parameter.}

\item{mtry}{positive integer. Number of randomly selected predictor variables for 
creating each split in each tree. Ignored when \code{tree.unbiased=FALSE}.}

\item{ntrees}{positive integer value. Number of trees to generate for the 
initial ensemble.}

\item{confirmatory}{character vector. Specifies one or more confirmatory terms 
to be included in the final ensemble. Linear terms can be specified as the 
name of a predictor variable included in \code{data}, rules can be specified
as, for example, \code{"x1 > 6 & x2 <= 8"}, where x1 and x2 should be names
of variables in \code{data}. Terms thus specified will be included in the
final ensemble, as their coefficients will not be penalized in the estimation.}

\item{removecomplements}{logical. Remove rules from the ensemble which are
identical to (1 - an earlier rule)?}

\item{removeduplicates}{logical. Remove rules from the ensemble which are 
identical to an earlier rule?}

\item{winsfrac}{numeric value \eqn{> 0} and \eqn{\le 0.5}. Quantiles of data 
distribution to be used for 
winsorizing linear terms. If set to 0, no winsorizing is performed. Note 
that ordinal variables are included as linear terms in estimating the
regression model and will also be winsorized.}

\item{normalize}{logical. Normalize linear variables before estimating the 
regression model? Normalizing gives linear terms the same a priori influence 
as a typical rule, by dividing the (winsorized) linear term by 2.5 times its 
SD.}

\item{standardize}{logical. Should rules and linear terms be standardized to
have SD equal to 1 before estimating the regression model? This will also 
standardize the dummified factors, users are advised to use the default 
\code{standardize = FALSE}.}

\item{ordinal}{logical. Should ordinal variables (i.e., ordered factors) be
treated as continuous for generating rules? \code{TRUE} (the default)
generally yields simpler rules, shorter computation times and better 
generalizability of the final ensemble.}

\item{nfolds}{positive integer. Number of cross-validation folds to be used for 
selecting the optimal value of the penalty parameter \eqn{\lambda} in selecting
the final ensemble.}

\item{tree.control}{list with control parameters to be passed to the tree 
fitting function, generated using \code{\link[partykit]{ctree_control}},
\code{\link[partykit]{mob_control}} (if \code{use.grad = FALSE}), or 
\code{\link[rpart]{rpart.control}} (if \code{tree.unbiased = FALSE}).}

\item{tree.unbiased}{logical. Should an unbiased tree generation algorithm 
be employed for rule generation? Defaults to \code{TRUE}, if set to 
\code{FALSE}, rules will be generated employing the CART algorithm
(which suffers from biased variable selection) as implemented in 
\code{\link[rpart]{rpart}}. See details below for possible combinations 
with \code{family}, \code{use.grad} and \code{learnrate}.}

\item{verbose}{logical. Should progress be printed to the command line?}

\item{par.init}{logical. Should parallel \code{foreach} be used to generate 
initial ensemble? Only used when \code{learnrate == 0}. Note: Must register 
parallel beforehand, such as doMC or others. Furthermore, setting 
\code{par.init = TRUE} will likely only increase computation time for smaller 
datasets.}

\item{par.final}{logical. Should parallel \code{foreach} be used to perform cross 
validation for selecting the final ensemble? Must register parallel beforehand, 
such as doMC or others.}

\item{sparse}{logical. Should sparse design matrices be used? Likely improves
computation times for large datasets.}

\item{...}{Additional arguments to be passed to
\code{\link[glmnet]{cv.glmnet}}.}
}
\value{
An object of class \code{pre}. It contains the initial ensemble of 
rules and/or linear terms and a range of possible final ensembles. 
By default, the final ensemble employed by all other
methods and functions in package \code{pre} is selected using the 'minimum
cross validated error plus 1 standard error' criterion. All functions and 
methods for objects of class \code{pre} take a \code{penalty.parameter.val} 
argument, which can be used to select a different criterion.
}
\description{
\code{pre} derives a sparse ensemble of rules and/or linear functions for 
prediction of a continuous, binary, count, multinomial, multivariate 
continuous or survival response.
}
\details{
Note: obervations with missing values will be removed prior to 
analysis (and a warning issued).

In some cases, duplicated variable names may appear in the model.
For example, the first variable is a factor named 'V1' and there are also
variables named 'V10' and/or 'V11' and/or 'V12' (etc). Then for 
for the binary factor V1, dummy contrast variables will be created, named 
'V10', 'V11', 'V12' (etc). As should be clear from this example, this yields 
duplicated variable names, which may yield problems, for example in the 
calculation of predictions and importances, later on. This can be prevented 
by renaming factor variables with numbers in their name, prior to analysis.

The table below provides an overview of combinations of response 
variable types, \code{use.grad}, \code{tree.unbiased} and
\code{learnrate} settings that are supported, and the tree induction 
algorithm that will be employed as a result:

\tabular{lccccc}{
\strong{use.grad} \tab \strong{tree.unbiased} \tab \strong{learnrate} \tab \strong{family} \tab \strong{tree alg.} \tab \strong{Response variable format} \cr
\cr
TRUE	\tab TRUE	\tab 0 \tab gaussian	  \tab ctree\tab Single, numeric (non-integer) \cr
TRUE	\tab TRUE	\tab 0 \tab mgaussian	  \tab ctree\tab Multiple, numeric (non-integer) \cr
TRUE	\tab TRUE	\tab 0 \tab binomial	  \tab ctree\tab Single, factor with 2 levels \cr
TRUE	\tab TRUE	\tab 0 \tab multinomial	\tab ctree\tab Single, factor with >2 levels \cr
TRUE	\tab TRUE	\tab 0 \tab poisson	    \tab ctree\tab Single, integer \cr
TRUE \tab TRUE \tab 0 \tab cox         \tab ctree\tab Object of class 'Surv' \cr
\cr
TRUE	\tab TRUE	\tab >0 \tab 	gaussian	  \tab ctree \tab Single, numeric (non-integer) \cr
TRUE	\tab TRUE	\tab >0	\tab mgaussian	  \tab ctree \tab Multiple, numeric (non-integer) \cr
TRUE	\tab TRUE	\tab >0	\tab binomial	  \tab ctree  \tab Single, factor with 2 levels \cr
TRUE	\tab TRUE	\tab >0	\tab multinomial	\tab ctree \tab Single, factor with >2 levels \cr
TRUE	\tab TRUE	\tab >0	\tab poisson	    \tab ctree  \tab Single, integer \cr
TRUE \tab TRUE \tab >0 \tab cox         \tab ctree\tab Object of class 'Surv' \cr
\cr
FALSE \tab TRUE \tab 0 \tab gaussian	  \tab glmtree \tab Single, numeric (non-integer) \cr
FALSE \tab TRUE \tab 0 \tab binomial	  \tab glmtree \tab Single, factor with 2 levels \cr
FALSE \tab TRUE \tab 0 \tab poisson	    \tab glmtree \tab Single, integer \cr
\cr
FALSE \tab TRUE \tab >0 \tab gaussian	  \tab glmtree \tab Single, numeric (non-integer) \cr
FALSE \tab TRUE \tab >0 \tab binomial	  \tab glmtree \tab Single, factor with 2 levels \cr
FALSE \tab TRUE \tab >0 \tab poisson	    \tab glmtree \tab Single, integer \cr
\cr
TRUE	\tab FALSE \tab 0 \tab gaussian	  \tab rpart \tab Single, numeric (non-integer) \cr
TRUE	\tab FALSE \tab 0 \tab binomial	  \tab rpart \tab Single, factor with 2 levels \cr
TRUE	\tab FALSE \tab 0 \tab multinomial	\tab rpart \tab Single, factor with >2 levels \cr
TRUE	\tab FALSE \tab 0 \tab poisson	    \tab rpart \tab Single, integer \cr
TRUE \tab FALSE \tab 0 \tab cox         \tab rpart\tab Object of class 'Surv' \cr
\cr
TRUE \tab FALSE	\tab >0 \tab gaussian	  \tab rpart \tab Single, numeric (non-integer) \cr
TRUE \tab FALSE	\tab >0 \tab binomial	  \tab rpart \tab Single, factor with 2 levels \cr
TRUE \tab FALSE	\tab >0 \tab poisson	  \tab rpart \tab Single, integer \cr
TRUE \tab FALSE \tab >0 \tab cox         \tab rpart \tab Object of class 'Surv'
}

If an error along the lines of 'factor ... has new levels ...' is encountered, 
consult \code{?rare_level_sampler} for explanation and solutions.
}
\note{
Parts of the code for deriving rules from the nodes of trees was copied 
with permission from an internal function of the \code{partykit} package, written
by Achim Zeileis and Torsten Hothorn.
}
\examples{
\donttest{## Fit pre to a continuous response:
airq <- airquality[complete.cases(airquality), ]
set.seed(42)
airq.ens <- pre(Ozone ~ ., data = airq)
airq.ens

## Fit pre to a binary response:
airq2 <- airquality[complete.cases(airquality), ]
airq2$Ozone <- factor(airq2$Ozone > median(airq2$Ozone))
set.seed(42)
airq.ens2 <- pre(Ozone ~ ., data = airq2, family = "binomial")
airq.ens2

## Fit pre to a multivariate continuous response:
airq3 <- airquality[complete.cases(airquality), ] 
set.seed(42)
airq.ens3 <- pre(Ozone + Wind ~ ., data = airq3, family = "mgaussian")
airq.ens3

## Fit pre to a multinomial response:
set.seed(42)
iris.ens <- pre(Species ~ ., data = iris, family = "multinomial")
iris.ens

## Fit pre to a survival response:
library("survival")
lung <- lung[complete.cases(lung), ]
set.seed(42)
lung.ens <- pre(Surv(time, status) ~ ., data = lung, family = "cox")
lung.ens

## Fit pre to a count response:
## Generate random data (partly based on Dobson (1990) Page 93: Randomized 
## Controlled Trial):
counts <- rep(as.integer(c(18, 17, 15, 20, 10, 20, 25, 13, 12)), times = 10)
outcome <- rep(gl(3, 1, 9), times = 10)
treatment <- rep(gl(3, 3), times = 10)
noise1 <- 1:90
set.seed(1)
noise2 <- rnorm(90)
countdata <- data.frame(treatment, outcome, counts, noise1, noise2)
set.seed(42)
count.ens <- pre(counts ~ ., data = countdata, family = "poisson")
count.ens}
}
\references{
Fokkema, M. (2020). Fitting prediction rule ensembles with R 
package pre. \emph{Journal of Statistical Software, 92}(12), 1-30.
\doi{10.18637/jss.v092.i12}

Fokkema, M. & Strobl, C. (2020). Fitting prediction rule ensembles to psychological 
research data: An introduction and tutorial. \emph{Psychological Methods 25}(5), 
636-652. \doi{10.1037/met0000256}, \url{https://arxiv.org/abs/1907.05302}

Friedman, J. H. (2001). Greedy function approximation: a gradient boosting 
machine. \emph{The Annals of Applied Statistics, 29}(5), 1189-1232.

Friedman, J. H., & Popescu, B. E. (2008). Predictive learning via rule 
ensembles. \emph{The Annals of Applied Statistics, 2}(3), 916-954, \doi{10.1214/07-AOAS148}.

Hothorn, T., & Zeileis, A. (2015). partykit: A modular toolkit for recursive 
partytioning in R. \emph{Journal of Machine Learning Research, 16}, 3905-3909.
}
\seealso{
\code{\link{print.pre}}, \code{\link{plot.pre}}, 
\code{\link{coef.pre}}, \code{\link{importance}}, \code{\link{predict.pre}}, 
\code{\link{interact}}, \code{\link{cvpre}}
}
