% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/etc_utils_autoplot.R, R/g_autoplot.R
\name{autoplot}
\alias{autoplot}
\alias{autoplot.mmcurves}
\alias{autoplot.mmpoints}
\alias{autoplot.mscurves}
\alias{autoplot.mspoints}
\alias{autoplot.smcurves}
\alias{autoplot.smpoints}
\alias{autoplot.sscurves}
\alias{autoplot.sspoints}
\title{Plot performance evaluation measures with ggplot2}
\usage{
\method{autoplot}{sscurves}(object, curvetype = c("ROC", "PRC"), ...)

\method{autoplot}{mscurves}(object, curvetype = c("ROC", "PRC"), ...)

\method{autoplot}{smcurves}(object, curvetype = c("ROC", "PRC"), ...)

\method{autoplot}{mmcurves}(object, curvetype = c("ROC", "PRC"), ...)

\method{autoplot}{sspoints}(object, curvetype = c("error", "accuracy",
  "specificity", "sensitivity", "precision"), ...)

\method{autoplot}{mspoints}(object, curvetype = c("error", "accuracy",
  "specificity", "sensitivity", "precision"), ...)

\method{autoplot}{smpoints}(object, curvetype = c("error", "accuracy",
  "specificity", "sensitivity", "precision"), ...)

\method{autoplot}{mmpoints}(object, curvetype = c("error", "accuracy",
  "specificity", "sensitivity", "precision"), ...)
}
\arguments{
\item{object}{An S3 object generated by \code{\link{evalmod}}.
  The \code{autoplot} function takes one of the following S3 objects.

\enumerate{

  \item ROC and Precision-Recall curves (mode = "rocprc")

  \tabular{lllll}{
    \strong{S3 object}
    \tab \tab \strong{# of models}
    \tab \tab \strong{# of test datasets} \cr

    sscurves \tab \tab single   \tab \tab single   \cr
    mscurves \tab \tab multiple \tab \tab single   \cr
    smcurves \tab \tab single   \tab \tab multiple \cr
    mmcurves \tab \tab multiple \tab \tab multiple
  }

  \item Basic evaluation measures (mode = "basic")

  \tabular{lllll}{
    \strong{S3 object}
    \tab \tab \strong{# of models}
    \tab \tab \strong{# of test datasets} \cr

    sspoints \tab \tab single   \tab \tab single   \cr
    mspoints \tab \tab multiple \tab \tab single   \cr
    smpoints \tab \tab single   \tab \tab multiple \cr
    mmpoints \tab \tab multiple \tab \tab multiple
  }
}}

\item{curvetype}{A character vector with the following curve types
\enumerate{

  \item ROC and Precision-Recall curves (mode = "rocprc")

    \describe{
      \item{"ROC"}{ROC curve}
      \item{"PRC"}{Precision-Recall curve}
      \item{c("ROC", "PRC")}{ROC and Precision-Recall curves}
    }

  \item Basic evaluation measures (mode = "basic")

    \describe{
      \item{"error"}{Normalized threshold values vs. error rate}
      \item{"accuracy"}{Normalized threshold values vs. accuracy}
      \item{"specificity"}{Normalized threshold values vs. specificity}
      \item{"sensitivity"}{Normalized threshold values vs. sensitivity}
      \item{"precision"}{Normalized threshold values vs. precision}
      \item{c("error", "accuracy", "specificity", "sensitivity",
              "precision")}{All of the above}
    }
}}

\item{...}{All the following arguments can be specified.

\describe{
  \item{curvetype}{
    \enumerate{

      \item ROC and Precision-Recall curves (mode = "rocprc")

        \describe{
          \item{"ROC"}{ROC curve}
          \item{"PRC"}{Precision-Recall curve}
          \item{c("ROC", "PRC")}{ROC and Precision-Recall curves}
         }

      \item Basic evaluation measures (mode = "basic")

        \describe{
          \item{"error"}{Normalized threshold values vs. error rate}
          \item{"accuracy"}{Normalized threshold values vs. accuracy}
          \item{"specificity"}{Normalized threshold values vs. specificity}
          \item{"sensitivity"}{Normalized threshold values vs. sensitivity}
          \item{"precision"}{Normalized threshold values vs. precision}
          \item{c("error", "accuracy", "specificity", "sensitivity",
              "precision")}{All of the above}
        }
     }
  }
  \item{type}{
    \describe{
      \item{"l"}{lines}
      \item{"p"}{points}
      \item{"b"}{both lines and points}
    }
  }
  \item{show_cb}{
    A Boolean value to specify whether point-wise confidence
    bounds are drawn. It is effective only when \code{calc_avg} is
    set to \code{TRUE} of the \code{\link{evalmod}} function.
  }
  \item{raw_curves}{
    A Boolean value to specify whether raw curves are
    shown instead of the average curve. It is effective only
    when \code{raw_curves} is set to \code{TRUE}
    of the \code{\link{evalmod}} function.
  }
  \item{show_legend}{
    A Boolean value to specify whether the legend is shown.
  }
  \item{ret_grob}{
    A logical value to indicate whether
    \code{autoplot} returns a \code{grob} object. The \code{grob} object
    is internally generated by \code{\link[gridExtra]{arrangeGrob}}.
    The \code{\link[grid]{grid.draw}} function takes a \code{grob} object and
    generates a plot. It is effective only for a multiple-panel plot that can
    be generated, for example, when \code{curvetype} is
    \code{c("ROC", "PRC")}.
  }
}}
}
\value{
The \code{autoplot} function returns a \code{ggplot} object
  for a single-panel plot and a frame-grob object for a multiple-panel plot.
}
\description{
The \code{autoplot} function plots performance evaluation measures
  by using \code{\link[ggplot2]{ggplot2}} instead of the general R plot.
}
\examples{

## Load libraries
library(ggplot2)
library(grid)
library(gridExtra)

#############################################################################
### Single model & single test dataset
###

\dontrun{

## Load a dataset with 10 positives and 10 negatives
data(P10N10)

## Generate an sscurve object that contains ROC and Precision-Recall curves
sscurves <- evalmod(scores = P10N10$scores, labels = P10N10$labels)

## Plot both ROC and Precision-Recall curves
autoplot(sscurves)

## Get a grob object for multiple plots
pp1 <- autoplot(sscurves, ret_grob = TRUE)
plot.new()
grid.draw(pp1)

## A ROC curve
autoplot(sscurves, curvetype = "ROC")

## A Precision-Recall curve
autoplot(sscurves, curvetype = "PRC")

## Generate an sspoints object that contains basic evaluation measures
sspoints <- evalmod(mode = "basic", scores = P10N10$scores,
                    labels = P10N10$labels)

## Threshold values vs. basic evaluation measures
autoplot(sspoints)

## Threshold vs. precision
autoplot(sspoints, curvetype = "precision")

}

#############################################################################
### Multiple models & single test dataset
###

\dontrun{

## Create sample datasets with 100 positives and 100 negatives
samps <- create_sim_samples(1, 100, 100, "all")
mdat <- mmdata(samps[["scores"]], samps[["labels"]],
               modnames = samps[["modnames"]])

## Generate an mscurve object that contains ROC and Precision-Recall curves
mscurves <- evalmod(mdat)

## ROC and Precision-Recall curves
autoplot(mscurves)

## Hide the legend
autoplot(mscurves, show_legend = FALSE)

## Generate an mspoints object that contains basic evaluation measures
mspoints <- evalmod(mdat, mode = "basic")

## Threshold values vs. basic evaluation measures
autoplot(mspoints)

## Hide the legend
autoplot(mspoints, show_legend = FALSE)

}

#############################################################################
### Single model & multiple test datasets
###

\dontrun{

## Create sample datasets with 100 positives and 100 negatives
samps <- create_sim_samples(10, 100, 100, "good_er")
mdat <- mmdata(samps[["scores"]], samps[["labels"]],
               modnames = samps[["modnames"]],
               dsids = samps[["dsids"]])

## Generate an smcurve object that contains ROC and Precision-Recall curves
smcurves <- evalmod(mdat, raw_curves = TRUE)

## Average ROC and Precision-Recall curves
autoplot(smcurves)

## Hide confidence bounds
autoplot(smcurves, show_cb = FALSE)

## Raw ROC and Precision-Recall curves
autoplot(smcurves, raw_curves = TRUE)

## Generate an smpoints object that contains basic evaluation measures
smpoints <- evalmod(mdat, mode = "basic")

## Threshold values vs. average basic evaluation measures
autoplot(smpoints)

}

#############################################################################
### Multiple models & multiple test datasets
###

\dontrun{

## Create sample datasets with 100 positives and 100 negatives
samps <- create_sim_samples(10, 100, 100, "all")
mdat <- mmdata(samps[["scores"]], samps[["labels"]],
               modnames = samps[["modnames"]],
               dsids = samps[["dsids"]])

## Generate an mscurve object that contains ROC and Precision-Recall curves
mmcurves <- evalmod(mdat, raw_curves = TRUE)

## Average ROC and Precision-Recall curves
autoplot(mmcurves)

## Show confidence bounds
autoplot(mmcurves, show_cb = TRUE)

## Raw ROC and Precision-Recall curves
autoplot(mmcurves, raw_curves = TRUE)

## Generate an mmpoints object that contains basic evaluation measures
mmpoints <- evalmod(mdat, mode = "basic")

## Threshold values vs. average basic evaluation measures
autoplot(mmpoints)

}

}
\seealso{
\code{\link{evalmod}} for generating an S3 object.
  \code{\link{fortify}} for converting a curves and points object
  to a data frame.  \code{\link{plot}} for plotting the equivalent curves
  with the general R plot.
}

