% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/solve.R
\name{solve}
\alias{solve}
\alias{solve.ConservationProblem}
\title{Solve}
\usage{
\method{solve}{ConservationProblem}(a, b, ..., run_checks = TRUE, force = FALSE)
}
\arguments{
\item{a}{\code{\link[=problem]{problem()}} object.}

\item{b}{missing.}

\item{...}{arguments passed to \code{\link[=compile]{compile()}}.}

\item{run_checks}{\code{logical} flag indicating whether presolve checks
should be run prior solving the problem. These checks are performed using
the \code{\link[=presolve_check]{presolve_check()}} function. Defaults to \code{TRUE}.
Skipping these checks may reduce run time for large problems.}

\item{force}{\code{logical} flag indicating if an attempt to should be
made to solve the problem even if potential issues were detected during
the presolve checks. Defaults to \code{FALSE}.}
}
\value{
A \code{numeric}, \code{matrix}, \code{data.frame}, \code{\link[sf:sf]{sf::st_sf()}}, or
\code{\link[terra:rast]{terra::rast()}} object containing the solution to the problem.
Additionally, the returned object will have the following additional
attributes: \code{"objective"} containing the solution's objective,
\code{"runtime"} denoting the number of seconds that elapsed while solving
the problem, and \code{"status"} describing the status of the solution
(e.g., \code{"OPTIMAL"} indicates that the optimal solution was found).
}
\description{
Solve a conservation planning problem.
}
\details{
After formulating a conservation planning \code{\link[=problem]{problem()}},
it can be solved using an exact algorithm solver (see \link{solvers}
for available solvers). If no solver has been explicitly specified,
then the best available exact algorithm solver will be used by default
(see \code{\link[=add_default_solver]{add_default_solver()}}). Although these exact algorithm
solvers will often display a lot of information that isn't really that
helpful (e.g., nodes, cutting planes), they do display information
about the progress they are making on solving the problem (e.g., the
performance of the best solution found at a given point in time). If
potential issues were detected during the
presolve checks (see \code{\link[=presolve_check]{presolve_check()}})
and the problem is being forcibly solved (i.e., with \code{force = TRUE}),
then it is also worth checking for any warnings displayed by the solver
to see if these potential issues are actually causing issues
(e.g., \emph{Gurobi} can display warnings that include
\code{"Warning: Model contains large matrix coefficient range"} and
\code{"Warning: Model contains large rhs"}).
}
\section{Output format}{

This function will output solutions in a similar format to the
planning units associated with \code{a}. Specifically, it will return
solutions based on the following types of planning units.

\describe{

\item{\code{a} has \code{numeric} planning units}{The solution will be
returned as a \code{numeric} vector. Here, each element in the vector
corresponds to a different planning unit.
Note that if a portfolio is used to generate multiple solutions,
then a \code{list} of such \code{numeric} vectors will be returned.}

\item{\code{a} has \code{matrix} planning units}{The solution will be
returned as a \code{matrix} object.
Here, rows correspond to different planning units,
and columns correspond to different  management zones.
Note that if a portfolio is used to generate multiple solutions,
then a \code{list} of such \code{matrix} objects will be returned.}

\item{\code{a} has \code{\link[terra:rast]{terra::rast()}} planning units}{The solution
will be returned as a \code{\link[terra:rast]{terra::rast()}} object.
If the argument to \code{x} contains multiple zones, then the object
will have a different layer for each management zone.
Note that if a portfolio is used to generate multiple solutions,
then a \code{list} of \code{\link[terra:rast]{terra::rast()}} objects will be returned.}

\item{\code{a} has \code{\link[sf:sf]{sf::sf()}}, or \code{data.frame} planning units}{
The solution will be returned in the same data format as the planning
units.
Here, each row corresponds to a different planning unit,
and columns contain solutions.
If the argument to \code{a} contains a single zone, then the solution object
will contain columns named by solution.
Specifically, the column names containing the solution values
be will named as \code{"solution_XXX"} where \code{"XXX"} corresponds to a solution
identifier (e.g., \code{"solution_1"}).
If the argument to \code{a} contains multiple zones, then the columns
containing solutions will be named as \code{"solution_XXX_YYY"} where
\code{"XXX"} corresponds to the solution identifier and \code{"YYY"} is the name
of the management zone (e.g., \code{"solution_1_zone1"}).}

}
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(500)

# load data
sim_pu_raster <- get_sim_pu_raster()
sim_pu_polygons <- get_sim_pu_polygons()
sim_features <- get_sim_features()
sim_zones_pu_raster <- get_sim_zones_pu_raster()
sim_zones_pu_polygons <- get_sim_zones_pu_polygons()
sim_zones_features <- get_sim_zones_features()

# build minimal conservation problem with raster data
p1 <-
  problem(sim_pu_raster, sim_features) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.1) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve the problem
s1 <- solve(p1)

# print solution
print(s1)

# print attributes describing the optimization process and the solution
print(attr(s1, "objective"))
print(attr(s1, "runtime"))
print(attr(s1, "status"))

# calculate feature representation in the solution
r1 <- eval_feature_representation_summary(p1, s1)
print(r1)

# plot solution
plot(s1, main = "solution", axes = FALSE)

# build minimal conservation problem with polygon data
p2 <-
  problem(sim_pu_polygons, sim_features, cost_column = "cost") \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.1) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve the problem
s2 <- solve(p2)

# print solution
print(s2)

# calculate feature representation in the solution
r2 <- eval_feature_representation_summary(p2, s2[, "solution_1"])
print(r2)

# plot solution
plot(s2[, "solution_1"], main = "solution", axes = FALSE)

# build multi-zone conservation problem with raster data
p3 <-
  problem(sim_zones_pu_raster, sim_zones_features) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5, ncol = 3)) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve the problem
s3 <- solve(p3)

# print solution
print(s3)

# calculate feature representation in the solution
r3 <- eval_feature_representation_summary(p3, s3)
print(r3)

# plot solution
plot(category_layer(s3), main = "solution", axes = FALSE)

# build multi-zone conservation problem with polygon data
p4 <-
  problem(
    sim_zones_pu_polygons, sim_zones_features,
    cost_column = c("cost_1", "cost_2", "cost_3")
  ) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5, ncol = 3)) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve the problem
s4 <- solve(p4)

# print solution
print(s4)

# calculate feature representation in the solution
r4 <- eval_feature_representation_summary(
  p4, s4[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
)
print(r4)

# create new column representing the zone id that each planning unit
# was allocated to in the solution
s4$solution <- category_vector(
  s4[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
)
s4$solution <- factor(s4$solution)

# plot solution
plot(s4[, "solution"])
}
}
\seealso{
See \code{\link[=problem]{problem()}} to create conservation planning problems, and
\code{\link[=presolve_check]{presolve_check()}} to check problems for potential issues.
Also, see the \code{\link[=category_layer]{category_layer()}} and \code{\link[=category_vector]{category_vector()}} function to
reformat solutions that contain multiple zones.
}
