% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dist_funs.R
\name{dqfr}
\alias{dqfr}
\alias{pqfr}
\alias{dqfr_A1I1}
\alias{dqfr_broda}
\alias{dqfr_butler}
\alias{pqfr_A1B1}
\alias{pqfr_imhof}
\alias{pqfr_davies}
\alias{pqfr_butler}
\title{Probability distribution of ratio of quadratic forms}
\usage{
dqfr(
  quantile,
  A,
  B,
  p = 1,
  mu = rep.int(0, n),
  Sigma = diag(n),
  log = FALSE,
  method = c("broda", "hillier", "butler"),
  trim_values = TRUE,
  normalize_spa = FALSE,
  return_abserr_attr = FALSE,
  m = 100L,
  tol_zero = .Machine$double.eps * 100,
  tol_sing = tol_zero,
  ...
)

pqfr(
  quantile,
  A,
  B,
  p = 1,
  mu = rep.int(0, n),
  Sigma = diag(n),
  lower.tail = TRUE,
  log.p = FALSE,
  method = c("imhof", "davies", "forchini", "butler"),
  trim_values = TRUE,
  return_abserr_attr = FALSE,
  m = 100L,
  tol_zero = .Machine$double.eps * 100,
  tol_sing = tol_zero,
  ...
)

dqfr_A1I1(
  quantile,
  LA,
  m = 100L,
  check_convergence = c("relative", "strict_relative", "absolute", "none"),
  use_cpp = TRUE,
  tol_conv = .Machine$double.eps^(1/4),
  thr_margin = 100
)

dqfr_broda(
  quantile,
  A,
  B,
  mu = rep.int(0, n),
  autoscale_args = 1,
  stop_on_error = TRUE,
  use_cpp = TRUE,
  tol_zero = .Machine$double.eps * 100,
  epsabs = epsrel,
  epsrel = 1e-06,
  limit = 10000
)

dqfr_butler(
  quantile,
  A,
  B,
  mu = rep.int(0, n),
  order_spa = 2,
  stop_on_error = FALSE,
  use_cpp = TRUE,
  tol_zero = .Machine$double.eps * 100,
  epsabs = .Machine$double.eps^(1/2),
  epsrel = 0,
  maxiter = 5000
)

pqfr_A1B1(
  quantile,
  A,
  B,
  m = 100L,
  mu = rep.int(0, n),
  check_convergence = c("relative", "strict_relative", "absolute", "none"),
  stop_on_error = FALSE,
  use_cpp = TRUE,
  cpp_method = c("double", "long_double", "coef_wise"),
  nthreads = 1,
  tol_conv = .Machine$double.eps^(1/4),
  tol_zero = .Machine$double.eps * 100,
  thr_margin = 100
)

pqfr_imhof(
  quantile,
  A,
  B,
  mu = rep.int(0, n),
  autoscale_args = 1,
  stop_on_error = TRUE,
  use_cpp = TRUE,
  tol_zero = .Machine$double.eps * 100,
  epsabs = epsrel,
  epsrel = 1e-06,
  limit = 10000
)

pqfr_davies(
  quantile,
  A,
  B,
  mu = rep.int(0, n),
  autoscale_args = 1,
  tol_zero = .Machine$double.eps * 100,
  ...
)

pqfr_butler(
  quantile,
  A,
  B,
  mu = rep.int(0, n),
  order_spa = 2,
  stop_on_error = FALSE,
  use_cpp = TRUE,
  tol_zero = .Machine$double.eps * 100,
  epsabs = .Machine$double.eps^(1/2),
  epsrel = 0,
  maxiter = 5000
)
}
\arguments{
\item{quantile}{Numeric vector of quantiles \eqn{q}}

\item{A, B}{Argument matrices.  Should be square.  \code{B} should be nonnegative
definite.  Will be automatically symmetrized in \code{dqfr()} and
\code{pqfr()}.}

\item{p}{Positive exponent of the ratio, default \code{1}.  Unlike in
\code{\link{qfrm}()}, the numerator and denominator cannot have
different exponents.  When \code{p} is non-integer, \code{A} must be
nonnegative definite.  For details, see vignette
\code{vignette("qfratio_distr")}.}

\item{mu}{Mean vector \eqn{\bm{\mu}}{\mu} for \eqn{\mathbf{x}}{x}}

\item{Sigma}{Covariance matrix \eqn{\mathbf{\Sigma}}{\Sigma} for
\eqn{\mathbf{x}}{x}}

\item{log, lower.tail, log.p}{Logical; as in regular probability distribution functions.  But these are
for convenience only, and not meant for accuracy.}

\item{method}{Method to specify an internal function (see \dQuote{Details}).  In
\code{dqfr()}, options are:
\itemize{
\item{\code{"broda"}: }{default; uses \code{dqfr_broda()}, numerical
inversion of Broda & Paolella (2009)}
\item{\code{"hillier"}: }{uses \code{dqfr_A1I1()}, series expression
of Hillier (2001)}
\item{\code{"butler"}: }{uses \code{dqfr_butler()}, saddlepoint
approximation of Butler & Paolella (2007, 2008)}
}
In \code{pqfr()}, options are:
\itemize{
\item{\code{"imhof"}: }{default; uses \code{pqfr_imhof()}, numerical
inversion of Imhof (1961)}
\item{\code{"davies"}: }{uses \code{pqfr_davies()}, numerical inversion
of Davies (1973, 1980)}
\item{\code{"forchini"}: }{uses \code{pqfr_A1B1()}, series expression
of Forchini (2002, 2005)}
\item{\code{"butler"}: }{uses \code{pqfr_butler()}, saddlepoint
approximation of Butler & Paolella (2007, 2008)}
}}

\item{trim_values}{If \code{TRUE} (default), numerical values outside the mathematically
permissible support are trimmed in (see \dQuote{Details})}

\item{normalize_spa}{If \code{TRUE} and \code{method == "butler"}, result is normalized so that
the density integrates to unity (see \dQuote{Details})}

\item{return_abserr_attr}{If \code{TRUE}, absolute error of numerical evaluation is returned
as an attribute \code{"abserr"} (see \dQuote{Value})}

\item{m}{Order of polynomials at which the series expression is truncated.  \eqn{M}
in Hillier et al. (2009, 2014).}

\item{tol_zero}{Tolerance against which numerical zero is determined.  Used to determine,
e.g., whether \code{mu} is a zero vector, \code{A} or \code{B} equals
the identity matrix, etc.}

\item{tol_sing}{Tolerance against which matrix singularity and rank are determined.  The
eigenvalues smaller than this are considered zero.}

\item{...}{Additional arguments passed to internal functions}

\item{LA}{Eigenvalues of \eqn{\mathbf{A}}{A}}

\item{check_convergence}{Specifies how numerical convergence is checked for series expression (see
\code{\link{qfrm}})}

\item{use_cpp}{Logical to specify whether the calculation is done with \proglang{C++}
functions via \code{Rcpp}.  \code{TRUE} by default.}

\item{tol_conv}{Tolerance against which numerical convergence of series is checked.  Used
with \code{check_convergence}.}

\item{thr_margin}{Optional argument to adjust the threshold for scaling (see \dQuote{Scaling}
in \code{\link{d1_i}}).  Passed to internal functions (\code{\link{d1_i}},
\code{\link{d2_ij}}, \code{\link{d3_ijk}}) or their \proglang{C++} equivalents.}

\item{autoscale_args}{Numeric; if \code{> 0} (default), arguments are scaled to avoid failure in
numerical integration (see \code{vignette("qfratio_distr")}).  If
\code{<= 0}, the scaling is skipped.}

\item{stop_on_error}{If \code{TRUE}, execution is stopped upon an error (including
non-convergence) in evaluation of hypergeometric function,
numerical integration, or root finding.  If
\code{FALSE}, further execution is attempted regardless.}

\item{epsabs, epsrel, limit, maxiter}{Optional arguments used in numerical integration or root-finding
algorithm (see vignette: \code{vignette("qfratio_distr")})}

\item{order_spa}{Numeric to determine order of saddlepoint approximation.  More accurate
second-order approximation is used for any \code{order > 1} (default);
otherwise, (very slightly) faster first-order approximation is used.}

\item{cpp_method}{Method used in \proglang{C++} calculations to avoid numerical
overflow/underflow (see \dQuote{Details} in \code{\link{qfrm}})}

\item{nthreads}{Number of threads used in \proglang{OpenMP}-enabled \proglang{C++}
functions (see \dQuote{Multithreading} in \code{\link{qfrm}})}
}
\value{
\code{dqfr()} gives the density, and \code{pqfr()} gives the
distribution function or \eqn{p}-values corresponding to \code{quantile}.

When \code{return_abserr_attr = TRUE}, an absolute
error bound of numerical evaluation is returned as an attribute; this
feature is currently available with \code{dqfr(..., method = "broda")} and
\code{pqfr(..., method = "imhof")} only.  This error bound is automatically
transformed when trimming happens with \code{trim_values} (above) or when
\code{log}/\code{log.p = TRUE}.  See vignette for details
(\code{vignette("qfratio_distr")}).

The internal functions return a list containing \code{$d} or \code{$p}
(for density and lower \eqn{p}-value, respectively), and only this is passed
to the external function by default.  Other components may be inspected
for debugging purposes:
\itemize{
\item{\code{dqfr_A1I1()} and \code{pqfr_A1B1()}: }{have \code{$terms},
a vector of \eqn{0}th to \eqn{m}th order terms.}
\item{\code{pqfr_imhof()} and \code{dqfr_broda()}: }{have \code{$abserr},
absolute error of numerical integration; the one returned from
\code{CompQuadForm::\link[CompQuadForm]{imhof}()} is divided by
\code{pi}, as the integration result itself is (internally).  This is
passed to the external functions when \code{return_abserr_attr = TRUE}
(above).}
\item{\code{pqfr_davies()}: }{has the same components as
\code{CompQuadForm::\link[CompQuadForm]{davies}()} apart from \code{Qq}
which is replaced by \code{p = 1 - Qq}.}
}
}
\description{
\code{dqfr()}: Density of the (power of) ratio of quadratic forms,
\eqn{\left( \frac{ \mathbf{x^{\mathit{T}} A x} }{
                   \mathbf{x^{\mathit{T}} B x} } \right) ^ p
}{ ((x^T A x) / (x^T B x))^p }, where
\eqn{\mathbf{x} \sim N_n(\bm{\mu}, \mathbf{\Sigma})}{x ~ N_n(\mu, \Sigma)}.

\code{pqfr()}: Distribution function of the same.

\code{dqfr_A1I1()}: internal for \code{dqfr()},
exact series expression of Hillier (2001).  Only accommodates
the simple case where \eqn{\mathbf{B} = \mathbf{I}_n}{B = I_n} and
\eqn{\bm{\mu} = \mathbf{0}_n}{\mu = 0_n}.

\code{dqfr_broda()}: internal for \code{dqfr()},
exact numerical inversion algorithm of Broda & Paolella (2009).

\code{dqfr_butler()}: internal for \code{dqfr()},
saddlepoint approximation of Butler & Paolella (2007, 2008).

\code{pqfr_A1B1()}: internal for \code{pqfr()},
exact series expression of Forchini (2002, 2005).

\code{pqfr_imhof()}: internal for \code{pqfr()},
exact numerical inversion algorithm of Imhof (1961).

\code{pqfr_davies()}: internal for \code{pqfr()},
exact numerical inversion algorithm of Davies (1973, 1980).
This is \strong{experimental} and may be removed in the future.

\code{pqfr_butler()}: internal for \code{pqfr()},
saddlepoint approximation of Butler & Paolella (2007, 2008).

The user is supposed to use the exported functions \code{dqfr()} and
\code{pqfr()}, which are (pseudo-)vectorized with respect to
\code{quantile}.  The actual calculations are done by one
of the internal functions, which only accommodate a length-one
\code{quantile}.  The internal functions skip most checks on argument
structures and do not accommodate \code{Sigma}
to reduce execution time.
}
\details{
\code{dqfr_A1I1()} and \code{pqfr_A1B1()} evaluate the probability density
and (cumulative) distribution function, respectively,
as a partial sum of infinite series involving top-order zonal or
invariant polynomials (Hillier 2001; Forchini 2002, 2005).  As in other
functions of this package, these are evaluated with the recursive algorithm
\code{\link{d1_i}}.

\code{pqfr_imhof()} and \code{pqfr_davies()} evaluate the distribution
function by numerical inversion of the characteristic function based on
Imhof (1961) or Davies (1973, 1980), respectively.  The latter calls
\code{\link[CompQuadForm]{davies}()}, and the former with
\code{use_cpp = FALSE} calls \code{\link[CompQuadForm]{imhof}()},
from the package \pkg{CompQuadForm}.  Additional arguments for
\code{\link[CompQuadForm]{davies}()} can be passed via \code{...},
except for \code{sigma}, which is not applicable.

\code{dqfr_broda()} evaluates the probability density by numerical inversion
of the characteristic function using Geary's formula based on
Broda & Paolella (2009).  Parameters for numerical integration
can be controlled via the arguments \code{epsabs}, \code{epsrel}, and
\code{limit} (see vignette: \code{vignette("qfratio_distr")}).

\code{dqfr_butler()} and \code{pqfr_butler()} evaluate saddlepoint
approximations of the density and distribution function, respectively,
based on Butler & Paolella (2007, 2008).  These are fast but not exact.  They
conduct numerical root-finding for the saddlepoint by the Brent method,
parameters for which can be controlled by the arguments
\code{epsabs}, \code{epsrel}, and \code{maxiter}
(see vignette: \code{vignette("qfratio_distr")}).  The saddlepoint
approximation density does not integrate to unity, but can be normalized by
\code{dqfr(..., method = "butler", normalize_spa = TRUE)}.  Note that
this is usually slower than \code{dqfr(..., method = "broda")} for
a small number of quantiles.

The density is undefined, and the distribution function has points of
nonanalyticity, at the eigenvalues of
\eqn{\mathbf{B}^{-1} \mathbf{A}}{B^-1 A} (assuming nonsingular
\eqn{\mathbf{B}}{B}).  Around these points,
the series expressions tends to fail.  Avoid using the series expression
methods for these cases.

Algorithms based on numerical integration can yield spurious results
that are outside the mathematically permissible support; e.g.,
\eqn{[0, 1]} for \code{pqfr()}.  By default, \code{dqfr()} and \code{pqfr()}
trim those values into the permissible range with a warning; e.g.,
negative p-values are
replaced by ~\code{2.2e-14} (default \code{tol_zero}).  Turn
\code{trim_values = FALSE} to skip these trimming and warning, although
\code{pqfr_imhof()} and \code{pqfr_davies()} can still throw a warning
from \pkg{CompQuadForm} functions.  Note that, on the other hand,
all these functions try to return exact \code{0} or \code{1}
when \eqn{q} is outside the possible range of the statistic.
}
\examples{
## Some symmetric matrices and parameters
nv <- 4
A <- diag(nv:1)
B <- diag(sqrt(1:nv))
mu <- 0.2 * nv:1
Sigma <- matrix(0.5, nv, nv)
diag(Sigma) <- 1

## density and p-value for (x^T A x) / (x^T x) where x ~ N(0, I)
dqfr(1.5, A)
pqfr(1.5, A)

## P{ (x^T A x) / (x^T B x) <= 1.5} where x ~ N(mu, Sigma)
pqfr(1.5, A, B, mu = mu, Sigma = Sigma)

## These are (pseudo-)vectorized
qs <- 0:nv + 0.5
dqfr(qs, A, B, mu = mu)
pqfr(qs, A, B, mu = mu)

## Various methods for density
dqfr(qs, A, method = "broda")   # default
dqfr(qs, A, method = "hillier") # series; B, mu, Sigma not permitted
## Saddlepoint approximations (fast but inexact):
dqfr(qs, A, method = "butler")  # 2nd order by default
dqfr(qs, A, method = "butler", normalize_spa = TRUE) # normalized
dqfr(qs, A, method = "butler", normalize_spa = TRUE,
     order_spa = 1) # 1st order, normalized

## Various methods for distribution function
pqfr(qs, A, method = "imhof")    # default
pqfr(qs, A, method = "davies")   # very similar
pqfr(qs, A, method = "forchini") # series expression
pqfr(qs, A, method = "butler")   # saddlepoint approximation (2nd order)
pqfr(qs, A, method = "butler", order_spa = 1) # 1st order

## To see error bounds
dqfr(qs, A, return_abserr_attr = TRUE)
pqfr(qs, A, return_abserr_attr = TRUE)

}
\references{
Broda, S. and Paolella, M. S. (2009) Evaluating the density of ratios of
noncentral quadratic forms in normal variables.
\emph{Computational Statistics and Data Analysis}, \strong{53}, 1264--1270.
\doi{10.1016/j.csda.2008.10.035}

Butler, R. W. and Paolella, M. S. (2007) Uniform saddlepoint approximations
for ratios of quadratic forms. Technical Reports, Department of Statistical
Science, Southern Methodist University, no. \strong{351}.
[Available on \emph{arXiv} as a preprint.]
\doi{10.48550/arXiv.0803.2132}

Butler, R. W. and Paolella, M. S. (2008) Uniform saddlepoint approximations
for ratios of quadratic forms. \emph{Bernoulli}, \strong{14}, 140--154.
\doi{10.3150/07-BEJ6169}

Davis, R. B. (1973) Numerical inversion of a characteristic function.
\emph{Biometrika}, \strong{60}, 415--417.
\doi{10.1093/biomet/60.2.415}.

Davis, R. B. (1980) Algorithm AS 155: The distribution of a linear
combination of \eqn{\chi^2} random variables.
\emph{Journal of the Royal Statistical Society, Series C---Applied Statistics},
\strong{29}, 323--333.
\doi{10.2307/2346911}.

Forchini, G. (2002) The exact cumulative distribution function of
a ratio of quadratic forms in normal variables, with application to
the AR(1) model. \emph{Econometric Theory}, \strong{18}, 823--852.
\doi{10.1017/S0266466602184015}.

Forchini, G. (2005) The distribution of a ratio of quadratic forms in
noncentral normal variables.
\emph{Communications in Statistics---Theory and Methods}, \strong{34}, 999--1008.
\doi{10.1081/STA-200056855}.

Hillier, G. (2001) The density of quadratic form in a vector uniformly
distributed on the \eqn{n}-sphere.
\emph{Econometric Theory}, \strong{17}, 1--28.
\doi{10.1017/S026646660117101X}.

Imhof, J. P. (1961) Computing the distribution of quadratic forms in normal
variables.
\emph{Biometrika}, \strong{48}, 419--426.
\doi{10.1093/biomet/48.3-4.419}.
}
\seealso{
\code{\link{rqfr}}, a Monte Carlo random number generator

\code{vignette("qfratio_distr")} for theoretical details
}
