#' Read comma separated csv file from Qualtrics.
#'
#'#' \code{readSurvey}
#'Reads comma separated csvdata files generated by Qualtrics
#'software. The second line containing the variable labels is imported.
#'Repetitive introductions to matrix questions are automatically removed.
#'Variable labels are stored as attributes.
#'
#' @param file_name A csv data file.
#' @param convertStandardColumns logical, defaults to TRUE. If TRUE, then the function will convert general data columns (first name, last name, lat, lon, ip address, startdate, enddate etc.) to their proper format.
#' @param stripHTML logical, defaults to TRUE. If TRUE, then remove html tags.
#'
#' @author Adrian Brugger, Stefan Borer & Jasper Ginn
#' @importFrom utils read.csv
#' @return A data frame. Variable labels are stored as attributes. They are not printed on
#' the console but are visibile in the RStudio viewer.
#' @export
#' @examples
#' \dontrun{
#' my_data_frame <- readSurvey(my_csv_file)
#' }

readSurvey <- function(file_name,
                       convertStandardColumns = TRUE,
                       stripHTML = TRUE) {
    # check if file exists
    if(!file.exists(file_name)) {
        print("File does not exist")
        return(-1)
    }
    # import data including variable names (row 1) and variable labels (row 2)
    rawdata <- read.csv(file=file_name,
                        header = FALSE,
                        sep = ',',
                        stringsAsFactors=FALSE,
                        fileEncoding = "UTF-8-BOM",
                        skip = 3)
    header <- read.csv(file=file_name,
                       header = TRUE,
                       sep = ',',
                       stringsAsFactors=FALSE,
                       fileEncoding = "UTF-8-BOM",
                       nrows = 1)
    # Add names
    names(rawdata) <- names(header)
    # Import importids
    importids <- unname(unlist(read.csv(file = file_name,
                          header = F, sep = ',',
                          stringsAsFactors = FALSE,
                          fileEncoding = "UTF-8-BOM", skip=2, nrows=1)))
    # Turn to json
    #importids <- lapply(importids, function(x) {
    #  jsonlite::fromJSON(stringr::str_replace_all(x, "'", '"'),
    #                                flatten = TRUE)
    #})
    # If Qualtrics adds an empty column at the end, remove it
    if(grepl(",$", readLines(file_name, n = 1))) {
        header <- header[,1:(ncol(header)-1)]
        rawdata <- rawdata[,1:(ncol(rawdata)-1)]
    }
    # extract second row, remove it from df
    secondrow <- unlist(header)
    row.names(rawdata) <- NULL
    # ----------------------------------------------------
    # clean variable labels (row 2)
    # ----------------------------------------------------
    if(stripHTML) {
        # weird regex to strip HTML tags, leaving only content
        # https://www.r-bloggers.com/htmltotext-extracting-text-from-html-via-xpath/
        pattern <- "</?\\w+((\\s+\\w+(\\s*=\\s*(?:\".*?\"|'.*?'|[^'\">\\s]+))?)+\\s*|\\s*)/?>"
        secondrow <- gsub(pattern, "\\4", secondrow)
    }
    # Scale Question with subquestion:
    # If it matches one of ".?!" followed by "-", take subsequent part
    subquestions <- stringr::str_match(secondrow, ".*[:punct:]\\s*-(.*)")[,2]
    # Else if subquestion returns NA, use whole string
    subquestions[is.na(subquestions)] <- unlist(secondrow[is.na(subquestions)])
    # Remaining NAs default to 'empty string'
    subquestions[is.na(subquestions)] <- ""
    # -------------------
    # add variable labels
    # -------------------
    rawdata <- sjmisc::set_label(rawdata, unlist(subquestions))
    # Add types
    if(convertStandardColumns) {
      rawdata <- inferDataTypes(rawdata)
    }
    return(rawdata)
}
