% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scan_neighbor.R
\name{scan_neighbor}
\alias{scan_neighbor}
\title{Genome scan for neighbor effects with a QTL model}
\usage{
scan_neighbor(
  genoprobs,
  pheno,
  smap,
  scale,
  addcovar = NULL,
  addQTL = NULL,
  grouping = rep(1, nrow(smap)),
  response = c("quantitative", "binary"),
  contrasts = NULL
)
}
\arguments{
\item{genoprobs}{Conditional genotype probabilities as taken from \code{qtl::calc.genoprob()}.}

\item{pheno}{A vector of individual phenotypes.}

\item{smap}{A matrix showing a spatial map for individuals. The first and second column include spatial positions along an x-axis and y-axis, respectively.}

\item{scale}{A numeric scalar indicating the maximum spatial distance between a focal individual and neighbors to define neighbor effects.}

\item{addcovar}{An optional matrix including additional non-genetic covariates. It contains no. of individuals x no. of covariates.}

\item{addQTL}{An optional vector containing marker names that are considered covariates. Namely, this option allows composite interval mapping (Jansen 1993).}

\item{grouping}{An optional integer vector assigning each individual to a group. This argument can be used when \code{smap} contains different experimental replicates. Default setting means that all individuals are belong to a single group.}

\item{response}{An optional argument to select trait types. The \code{"quantitative"} or \code{"binary"} calls the \code{"gaussian"} or \code{"binomial"} family in \code{glm()}, respectively.}

\item{contrasts}{An optional vector composed of three TRUE/FALSE values, which represents the presence/absence of specific genotypes as c(TRUE/FALSE, TRUE/FALSE, TRUE/FALSE) = AA, AB, BB. If \code{NULL}, it is compiled from \code{genoprobs} automatically.}
}
\value{
A matrix of LOD scores for self and neighbor effects, with the chromosome numbers and positions. The row names correspond to marker names.
\itemize{
\item{\code{chr}} {Chromosome number}
\item{\code{pos}} {Marker position}
\item{\code{LOD_self}} {LOD score for self effects}
\item{\code{LOD_nei}} {LOD score for neighbor effects}
}
}
\description{
Genome scan using a QTL model for self and neighbor effects, with possible allowance for additional covariates and non-normal traits. Theoretical background is described in Sato, Takeda & Nagano (2020).
}
\details{
This function calculates LOD score after the additive and dominance deviation are estimated using \code{eff_neighbor()}.
As it adopts a stepwise testing from self to neighbor effects, \code{LOD_self} are the same as standard QTL mapping.
Note that the results return 0 LOD scores for covariate markers when using \code{addQTL} option.
}
\examples{
set.seed(1234)
test_map <- qtl::sim.map(len=rep(20,5),n.mar=3,include.x=FALSE)
test_cross <- qtl::sim.cross(test_map,n.ind=50)
test_smap <- cbind(runif(50,1,100),runif(50,1,100))
test_genoprobs <- qtl::calc.genoprob(test_cross,step=2)

test_scan <- scan_neighbor(genoprobs=test_genoprobs,
                           pheno=test_cross$pheno$phenotype,
                           smap=test_smap, scale=20
                           )
plot_nei(test_scan)
}
\references{
\itemize{
\item Jansen RC (1993) Interval mapping of multiple quantitative trait loci. Genetics 135:205-211.
\item Sato Y, Takeda K, Nagano AJ (2020) Neighbor QTL: an interval mapping method for quantitative trait loci underlying neighbor effects. bioRxiv \url{https://doi.org/10.1101/2020.05.20.089474}
}
}
\seealso{
\code{\link{eff_neighbor}}
}
\author{
Yasuhiro Sato (\email{sato.yasuhiro.36c@kyoto-u.jp})
}
