#' rSPDE linear mixed effects models
#'
#' Fitting linear mixed effects model with latent Whittle-Matern models.
#'
#' @param formula Formula object describing the relation between the response variables and the fixed effects. If the response variable is a matrix, each column of the matrix will be treated as a replicate.
#' @param loc A vector with the names of the columns in `data` that contain the observation locations, or a `matrix` or a `data.frame` containing the observation locations. If the model is of class `metric_graph`, the locations must be either a `matrix` or a `data.frame` with two columns, or a character vector with the names of the two columns. The first column being the number of the edge, and the second column being the normalized position on the edge. If the model is a 2d model, `loc` must be either a `matrix` or `data.frame` with two columns or a character vector with the name of the two columns that contain the location, the first entry corresponding to the `x` entry and the second corresponding to the `y` entry. 
#' @param data A `data.frame` containing the data to be used.
#' @param model Either an object generated by `matern.operators()` or `spde.matern.operators()`. If `NULL`, a simple linear regression will be performed. 
#' @param repl Vector indicating the replicate of each observation. If `NULL` it will assume there is only one replicate.
#' @param which_repl Which replicates to use? If `NULL` all replicates will be used.
#' @param optim_method The method to be used with `optim` function.
#' @param use_data_from_graph Logical. Only for models generated from graphs from `metric_graph` class. In this case, should the data, the locations and the replicates be obtained from the graph object?
#' @param starting_values_latent A vector containing the starting values for the latent model. If the latent model was generated by `matern.operators()`, then the starting values should be provided as a vector of the form c(tau,kappa). If the model was generated by `spde.matern.operators()`, then the starting values should be provided as a vector containing the nonstationary parameters.
#' @param start_sigma_e Starting value for the standard deviation of the measurament error.
#' @param start_nu Starting value for the smoothness parameter.
#' @param start_alpha Starting value for the smoothness parameter. Will be used if `start_nu` is not given.
#' @param nu If `NULL`, the smoothness parameter will be estimated, otherwise the smoothness parameter will be kept fixed at the provided value.
#' @param alpha If `NULL`, the smoothness parameter will be estimated, otherwise the smoothness parameter will be kept fixed at the provided value. Will be used if `nu` is not given.
# @param model_matrix logical indicating whether the model matrix should be returned as component of the returned value.
#' @param nu_upper_bound A parameter that limits the maximum value that nu can assume. 
#' @param rspde_order The order of the rational approximation to be used while fitting the model. If not given, the order from the model object will be used.
#' @param parallel logical. Indicating whether to use optimParallel or not.
#' @param n_cores Number of cores to be used if parallel is true.
#' @param optim_controls Additional controls to be passed to `optim` or `optimParallel`.
# @param improve_gradient Should a more precise estimate of the gradient be obtained? Turning on might increase the overall time. If `TRUE`, the "Richardson" method will be used. See the help of the `grad` function in `numDeriv` package for details. 
# @param gradient_args List of controls to be used for the gradient. The list can contain the arguments to be passed to the `method.args` argument in the `numDeriv::grad` function. See the help of the `grad` function in `numDeriv` package for details. 
#' @param improve_hessian Should a more precise estimate of the hessian be obtained? Turning on might increase the overall time.
#' @param hessian_args List of controls to be used if `improve_hessian` is `TRUE`. The list can contain the arguments to be passed to the `method.args` argument in the `numDeriv::hessian` function. See the help of the `hessian` function in `numDeriv` package for details. Observe that it only accepts the "Richardson" method for now, the method "complex" is not supported. 
#' @return A list containing the fitted model.
#' @rdname rspde_lme
#' @export
#' 

rspde_lme <- function(formula, loc, data, 
                model = NULL, repl = NULL,
                which_repl = NULL,
                optim_method = "L-BFGS-B", 
                use_data_from_graph = TRUE,
                starting_values_latent = NULL,
                start_sigma_e = NULL,
                start_alpha = NULL,
                alpha = NULL,
                start_nu = NULL,
                nu = NULL,
                nu_upper_bound = 4,
                rspde_order = NULL,
                # model_matrix = TRUE,
                parallel = FALSE,
                n_cores = parallel::detectCores()-1,
                optim_controls = list(),
                # improve_gradient = FALSE,
                # gradient_args = list(),
                improve_hessian = FALSE,
                hessian_args = list()) {

   null_model <- TRUE
   if(!is.null(model)){
    if(!inherits(model, c("CBrSPDEobj","rSPDEobj"))){
        stop("The model should be an object of class 'CBrSPDEobj' or 'rSPDEobj'.")
    }
    null_model <- FALSE
    model <- update(model, parameterization = "spde")
   }

   estimate_nu <- TRUE

   if(!is.null(nu)){
    estimate_nu <- FALSE
    if(!is.numeric(nu)){
        stop("nu must be numeric!")
    }
    if(length(nu)>1){
        stop("nu must have length 1")
    }
    if(nu < 0){
        stop("nu must be positive.")
    }
    alpha <- nu + model$d/2
   }

   if(!is.null(alpha)){
    estimate_nu <- FALSE
    if(!is.numeric(alpha)){
        stop("alpha must be numeric!")
    }
    if(length(alpha)>1){
        stop("alpha must have length 1")
    }
    if(alpha <= model$d/2){
        stop(paste("alpha must be greater than dim/2 = ",model$d/2))
    }
    nu <- alpha - model$d/2    
   }

    if(!is.null(rspde_order) && !is.null(model)){
        model <- update(model, user_m = rspde_order)
    } else if (!is.null(model)){
      rspde_order <- model$m
    } else{
      rspde_order <- NULL
    }

  if (is.null(formula)) {
    stop("No formula provided!")
  }

  call_rspde_lme <- match.call()

  if(null_model){
    model <- list(has_graph = FALSE,
                  stationary = FALSE)
  }

  time_data_start <- Sys.time()
  
  if (missing(data) && (!model$has_graph)) {
    data <- environment(formula)
  } else if(model$has_graph){
    if(use_data_from_graph){
      if(is.null(model$graph$data)){
        stop("The graph has no data! Either add data to the graph, or add the data manually and set 'use_data_from_graph' to FALSE.")
      }
          data <- model$graph$data
          repl <- model$graph$data[["__group"]]
          if(missing(loc)){
              # Don't do anything, we will replace loc anyway
          }
          loc <- cbind(model$graph$data[["__edge_number"]],
                  model$graph$data[["__distance_on_edge"]])
          }

  }

  y_term <- stats::terms(formula)[[2]]

  y_resp <- eval(y_term, envir = data, enclos = parent.frame())
  y_resp <- as.numeric(y_resp)

  cov_term <- stats::delete.response(terms(formula))

  X_cov <- stats::model.matrix(cov_term, data)

  if(all(dim(X_cov) == c(0,1))){
    names_temp <- colnames(X_cov)
    X_cov <- matrix(1, nrow = length(y_resp))
    colnames(X_cov) <- names_temp
  }

  if(is.null(repl)){
    repl <- rep(1, length(y_resp))
  }

  if(is.null(which_repl)){
    which_repl <- unique(repl)
  }

  which_repl <- unique(which_repl)
  if(length(setdiff(which_repl, repl))>0){
    warning("There are elements in 'which_repl' that are not in 'repl'.")
  }

  idx_repl <- (repl %in% which_repl)

  y_resp <- y_resp[idx_repl]
  if(ncol(X_cov)>0){
    X_cov <- X_cov[idx_repl, , drop = FALSE]
  }
  repl <- repl[idx_repl] 

  time_data_end <- Sys.time()

  time_data <- time_data_end - time_data_start

  if(!null_model){
    time_build_likelihood_start <- Sys.time()

    if(is.null(starting_values_latent)){
      if(!model$stationary){
          if(is.null(model$theta)){
              stop("For models given by spde.matern.operators(), theta must be non-null!")
          }
          starting_values_latent <- model$theta
      } else{
          # if(model$parameterization == "spde"){
              starting_values_latent <- log(c(model$tau, model$kappa))
          # } else if(model$parameterization == "matern"){
          #     starting_values_latent <- log(c(model$sigma, model$range))
          # }
      }
    } else{
      if(model$stationary){
          if(length(starting_values_latent)!=2){
              stop("starting_values_latent must be a vector of length 2.")
          }
          if(any(starting_values_latent<0)){
            stop("For stationary models, the values of starting_values_latent must be positive.")
          }
      } else{
          if(length(starting_values_latent)!=ncol(model$B.tau)){
              stop("starting_values_latent must be a vector of the same length as the number of the covariates for the latent model.")
          }
      }
    }

    if(estimate_nu){
      if(is.null(start_nu) && is.null(start_alpha)){
        start_values <- c(log(c(0.1*sd(y_resp), model$nu)),starting_values_latent)
      } else if(!is.null(start_nu)){
        if(!is.numeric(start_nu)){
          stop("start_nu must be numeric.")
        }
        if(length(start_nu>1)){
          stop("start_nu must have length 1.")
        }
        if(start_nu <= 0){
          stop("start_nu must be positive")
        }        
        start_values <- c(log(c(0.1*sd(y_resp), start_nu)),starting_values_latent)
      } else {
        if(!is.numeric(start_alpha)){
          stop("start_alpha must be numeric.")
        }
        if(length(start_alpha>1)){
          stop("start_alpha must have length 1.")
        }
        if(start_alpha <= model$d/2){
          stop(paste("start_alpha must be greater than dim/2 =", model$d/2))
        }
        start_values <- c(log(c(0.1*sd(y_resp), start_alpha - model$d/2)),starting_values_latent)
      }
    } else{
        start_values <- c(log(0.1*sd(y_resp)), starting_values_latent)
    }

    if(!is.null(start_sigma_e)){
        start_values[1] <- log(start_sigma_e)
    }

  if(is.data.frame(loc) || is.matrix(loc)){
    loc_df <- loc
  } else if(is.character(loc)){
    if(!model$has_graph){
        dim <- model$d
        if(length(loc) != dim){
            stop("If 'loc' is a character vector, it must have the same length as the dimension (unless model comes from a metric graph).")
        }
        if(dim == 1){
            loc_df <- matrix(data[[loc[1]]], ncol=1)
        } else if (dim == 2){
            loc_df <- cbind(as.vector(data[[loc[1]]]), 
                                    as.vector(data[[loc[2]]]))
        }
    } else{
        if(length(loc)!=2){
            stop("For a metric graph, 'loc' must have length two.")
        }
            loc_df <- cbind(as.vector(data[[loc[1]]]), 
                                    as.vector(data[[loc[2]]]))
    }
  } else{
    stop("loc must be either a matrix, a data.frame or a character vector with the names of the columns of the observation locations.")
  }

    repl_val <- unique(repl)
    A_list <- list()
    # y_list <- list()
    # X_cov_list <- list()
    # has_cov <- FALSE
    # if(ncol(X_cov) > 0){
    #   has_cov <- TRUE
    # }

    if(!is.null(model$make_A)) {
        for(j in repl_val){
            ind_tmp <- (repl %in% j)
            y_tmp <- y_resp[ind_tmp]            
            na_obs <- is.na(y_tmp)
            # y_list[[as.character(j)]] <- y_tmp[!na_obs]
            A_list[[as.character(j)]] <- model$make_A(loc_df[ind_tmp,])
            A_list[[as.character(j)]] <- A_list[[as.character(j)]][!na_obs, , drop = FALSE]
            # if(has_cov){
            #   X_cov_list[[as.character(j)]] <- X_cov[ind_tmp, , drop = FALSE]
            #   X_cov_list[[as.character(j)]] <- X_cov_list[[as.character(j)]][!na_obs, , drop = FALSE]
            # }

        if(inherits(model, "CBrSPDEobj")){
                  A_list[[as.character(j)]] <- kronecker(matrix(1, 1, model$m + 1), A_list[[as.character(j)]])
        }  
        }
    } else{
        stop("When creating the model object using matern.operators() or spde.matern.operators(), you should either supply a graph, or a mesh, or mesh_loc (this last one only works for dimension 1).")
    }

    n_coeff_nonfixed <- length(start_values)

    model_tmp <- model
    model_tmp$mesh <- NULL
    model_tmp$graph <- NULL
    model_tmp$make_A <- NULL

    if(inherits(model, "CBrSPDEobj")){
            likelihood <- function(theta){

                sigma_e <- exp(theta[1])
                n_cov <- ncol(X_cov)
                n_initial <- n_coeff_nonfixed                
                if(estimate_nu){
                    nu <- exp(theta[2])
                    if(nu %% 1 == 0){
                      nu <- nu - 1e-5
                    }
                    nu <- min(nu, 9.99)
                    gap <- 1
                } else{
                    gap <- 0
                }
                    
                if(model_tmp$stationary){
                    # if(model_tmp$parameterization == "spde"){
                        alpha <- nu + model$d/2
                        alpha <- max(1e-5 + model$d/2, alpha)                        
                        tau <- exp(theta[2+gap])
                        kappa <- exp(theta[3+gap])
                        model_tmp <- update.CBrSPDEobj(model_tmp,
                            user_alpha = alpha, user_tau = tau,
                            user_kappa = kappa, parameterization = "spde")
                    # } else if(model_tmp$parameterization == "matern"){
                    #     sigma <- exp(theta[2+gap])
                    #     range <- exp(theta[3+gap])
                    #     model_tmp <- update.CBrSPDEobj(model_tmp,
                    #         user_nu = nu,
                    #         user_sigma = sigma, user_range = range,
                    #         parameterization = "matern")
                    # } 
                } else{
                    theta_model <- theta[(2+gap):(n_initial)]
                    alpha <- nu + model$d/2
                    alpha <- max(1e-5 + model$d/2, alpha)
                    model_tmp <- update.CBrSPDEobj(model_tmp,
                            user_theta = theta_model,
                            user_alpha = alpha,
                            parameterization = "spde")                        
                }
                
                if(n_cov > 0){
                    beta_cov <- theta[(n_initial+1):(n_initial+n_cov)]
                } else{
                    beta_cov <- NULL
                }

                loglik <- aux_lme_CBrSPDE.matern.loglike(object = model_tmp, y = y_resp, X_cov = X_cov, repl = repl,
                A_list = A_list, sigma_e = sigma_e, beta_cov = beta_cov)

            return(-loglik)
        }
    } else{
           likelihood <- function(theta){
                sigma_e <- exp(theta[1])
                n_cov <- ncol(X_cov)
                n_initial <- n_coeff_nonfixed                
                if(estimate_nu){
                    nu <- exp(theta[2])
                    if(nu %% 1 == 0){
                      nu <- nu - 1e-5
                    }
                    nu <- min(nu, nu_upper_bound)
                    gap <- 1
                } else{
                    gap <- 0
                }

                if(model$stationary){
                    # if(model_tmp$parameterization == "spde"){
                        alpha <- nu + model$d/2
                        alpha <- max(1e-5 + model$d/2, alpha)
                        tau <- exp(theta[2+gap])
                        kappa <- exp(theta[3+gap])
                        model_tmp <- update.rSPDEobj(model_tmp,
                            user_alpha = alpha, user_tau = tau,
                            user_kappa = kappa, parameterization = "spde")
                    # } else if(model_tmp$parameterization == "matern"){
                    #     sigma <- exp(theta[2+gap])
                    #     range <- exp(theta[3+gap])
                    #     model_tmp <- update.rSPDEobj(model_tmp,
                    #         user_nu = nu,
                    #         user_sigma = sigma, user_range = range,
                    #         parameterization = "matern")
                    # }
                } else{
                    theta_model <- theta[(2+gap):(n_initial)]
                    alpha <- nu + model$d/2
                    alpha <- max(1e-5 + model$d/2, alpha)
                    model_tmp <- update.rSPDEobj(model_tmp,
                            user_theta = theta_model,
                            user_alpha = alpha, parameterization = "spde")
                }
                
                if(n_cov > 0){
                    beta_cov <- theta[(n_initial+1):(n_initial+n_cov)]
                } else{
                    beta_cov <- NULL
                }

                loglik <- aux_lme_rSPDE.matern.loglike(object = model_tmp, y = y_resp, X_cov = X_cov, repl = repl,
                A_list = A_list, sigma_e = sigma_e, beta_cov = beta_cov)

            return(-loglik)
        }
    }

 if(ncol(X_cov)>0 && !is.null(model)){
    names_tmp <- colnames(X_cov)
    data_tmp <- cbind(y_resp, X_cov)
    data_tmp <- na.omit(data_tmp)
    temp_coeff <- lm(data_tmp[,1] ~ data_tmp[,-1] - 1)$coeff
    names(temp_coeff) <- names_tmp
    start_values <- c(start_values, temp_coeff)
    rm(data_tmp)
  }

  time_build_likelihood_end <- Sys.time()

  time_build_likelihood <- time_build_likelihood_end - time_build_likelihood_start 

hessian <- TRUE

if(improve_hessian){
  hessian <- FALSE
}

time_par <- NULL

if(parallel){
  start_par <- Sys.time()
  n_cores_lim <- Sys.getenv("_R_CHECK_LIMIT_CORES_", "")

  if (nzchar(n_cores_lim) && n_cores_lim == "TRUE") {
    n_cores <- 2L
  } 
  cl <- parallel::makeCluster(n_cores)
  parallel::setDefaultCluster(cl = cl)
  parallel::clusterExport(cl, "y_resp", envir = environment())
  parallel::clusterExport(cl, "model_tmp", envir = environment())
  parallel::clusterExport(cl, "A_list", envir = environment())
  parallel::clusterExport(cl, "X_cov", envir = environment())
  # parallel::clusterExport(cl, "y_list", envir = environment())  
  parallel::clusterExport(cl, "aux_lme_CBrSPDE.matern.loglike",
                 envir = as.environment(asNamespace("rSPDE")))
  parallel::clusterExport(cl, "aux_lme_rSPDE.matern.loglike",
                 envir = as.environment(asNamespace("rSPDE")))

  end_par <- Sys.time()
  time_par <- end_par - start_par

    start_fit <- Sys.time()
    res <- optimParallel::optimParallel(start_values, 
                  likelihood, method = optim_method,
                  control = optim_controls,
                  hessian = hessian,
                  parallel = list(forward = FALSE, cl = cl,
                      loginfo = FALSE))
  end_fit <- Sys.time()
  time_fit <- end_fit-start_fit
  parallel::stopCluster(cl)
} else{
  possible_methods <- c("CG", "BFGS", "L-BFGS-B", "Nelder-Mead")

  start_fit <- Sys.time()
      res <- tryCatch(optim(start_values, 
                  likelihood, method = optim_method,
                  control = optim_controls,
                  hessian = hessian), error = function(e){return(NA)})
  end_fit <- Sys.time()
  time_fit <- end_fit-start_fit
  if(is.na(res[1])){
    tmp_method <- optim_method
    while(length(possible_methods)>1){
      possible_methods <- setdiff(possible_methods, tmp_method)
      new_method <- possible_methods[1]
      warning(paste("optim method",tmp_method,"failed. Another optimization method was used."))
      start_fit <- Sys.time()
        res <- tryCatch(optim(start_values, 
                  likelihood, method = new_method,
                  control = optim_controls,
                  hessian = hessian), error = function(e){return(NA)})
      end_fit <- Sys.time()
      time_fit <- end_fit-start_fit
      tmp_method <- new_method
      if(!is.na(res[1])){
        optim_method <- new_method
        break
      }
    }
    if(length(possible_methods) == 1){
      stop("All optimization methods failed.")
    }
  }
}

  if(model$stationary){
    coeff <- exp(c(res$par[1:n_coeff_nonfixed]))
  } else{
    coeff <- res$par[1:n_coeff_nonfixed]
    if(estimate_nu){
      coeff[2] <- exp(coeff[2])
    }
  }
  coeff <- c(coeff, res$par[-c(1:n_coeff_nonfixed)])

  loglik <- -res$value

  n_fixed <- ncol(X_cov)
  n_random <- length(coeff) - n_fixed - 1  

  time_hessian <- NULL

  if(!improve_hessian){
    observed_fisher <- res$hessian
  } else{
    if(!is.list(hessian_args)){
      stop("hessian_controls must be a list")
    }

    start_hessian <- Sys.time()
    observed_fisher <- numDeriv::hessian(likelihood, res$par, method.args = hessian_args)
    end_hessian <- Sys.time()
    time_hessian <- end_hessian-start_hessian
  }

  if(model$stationary){
    par_change <- diag(c(exp(-c(res$par[1:n_coeff_nonfixed])), rep(1,n_fixed)))
    observed_fisher <- par_change %*% observed_fisher %*% par_change
  }

  inv_fisher <- tryCatch(solve(observed_fisher), error = function(e) matrix(NA, nrow(observed_fisher), ncol(observed_fisher)))
  
  std_err <- sqrt(diag(inv_fisher))

  coeff_random <- coeff[2:(n_coeff_nonfixed)]
  std_random <- std_err[2:(n_coeff_nonfixed)]

  if(model$stationary){
    # if(model$parameterization == "spde"){
        par_names <- c("tau", "kappa")
    # } else if(model$parameterization == "matern"){
    #     par_names <- c("sigma", "range")
    # }
  } else{
    par_names <- c("Theta 1") 
    if(ncol(model$B.tau)>2){
        for(i in 2:(ncol(model$B.tau)-1)){
            par_names <- c(par_names, paste("Theta",i))
        }
    }
  }

  if(estimate_nu){
      par_names <- c("nu", par_names)
  }

  names(coeff_random) <- par_names

  coeff_meas <- coeff[1]
  names(coeff_meas) <- "std. dev"

  std_meas <- std_err[1]

  coeff_fixed <- NULL
  if(n_fixed > 0){
    coeff_fixed <- coeff[(2+n_random):length(coeff)]
    std_fixed <- std_err[(2+n_random):length(coeff)]
  } else{
    std_fixed <- NULL
  }
  if(model$stationary){

    time_matern_par_start <- Sys.time()
    new_likelihood <- function(theta){
      new_par <- res$par
      if(estimate_nu){
        new_par[3:4] <- theta
      } else{
        new_par[2:3] <- theta
      }
      return(likelihood(new_par))
    }
    
    if(estimate_nu){
      coeff_random_nonnu <- coeff_random[-1]
      new_observed_fisher <- observed_fisher[3:4,3:4]
    } else{
      coeff_random_nonnu <- coeff_random
      new_observed_fisher <- observed_fisher[2:3,2:3]
    }
    change_par <- change_parameterization_lme(new_likelihood, model$d, coeff_random[1], coeff_random_nonnu,
                                            hessian = new_observed_fisher #,
                                            # improve_gradient = improve_gradient,
                                            # gradient_args = gradient_args
                                            )
    matern_coeff <- list()
    matern_coeff$random_effects <- coeff_random
    names(matern_coeff$random_effects) <- c("nu", "sigma", "range")
    matern_coeff$random_effects[2:3] <- change_par$coeff
    matern_coeff$std_random <- std_random
    matern_coeff$std_random[2:3] <- change_par$std_random
    time_matern_par_end <- Sys.time()
    time_matern_par <- time_matern_par_end - time_matern_par_start
  } else{
    matern_coeff <- NULL
    time_matern_par <- NULL
  }


  } else{ # If model is NULL
    coeff_random <- NULL
    time_matern_par <- NULL
    std_random <- NULL
    time_build_likelihood <- NULL
    start_values <- NULL
    matern_coeff <- NULL
    time_fit <- NULL
    time_hessian <- NULL
    time_par <- NULL
    A_list <- NULL

    if(ncol(X_cov) == 0){
      stop("The model does not have either random nor fixed effects.")
    }

    names_tmp <- colnames(X_cov)
    data_tmp <- cbind(y_resp, X_cov)
    data_tmp <- na.omit(data_tmp)
    res <- lm(data_tmp[,1] ~ data_tmp[,-1] - 1)
    coeff_fixed <- res$coeff
    names(coeff_fixed) <- names_tmp
    sm_temp <- summary(res)
    std_fixed <- sm_temp$coefficients
    rownames(std_fixed) <- names_tmp
    coeff_meas <- sm_temp$sigma
    names(coeff_meas) <- "std. dev"
    std_meas <- NULL
    loglik <- logLik(res)[[1]]

  }

  if(is.null(coeff_fixed) && is.null(coeff_random)){
    stop("The model does not have either random nor fixed effects.")
  }

  object <- list()
  object$coeff <- list(measurement_error = coeff_meas, 
  fixed_effects = coeff_fixed, random_effects = coeff_random)
  object$estimate_nu <- estimate_nu
  if(object$estimate_nu && !null_model){
    names(object$coeff$random_effects)[1] <- "alpha"
    object$coeff$random_effects[1] <- object$coeff$random_effects[1] + model$d/2
  }

  object$std_errors <- list(std_meas = std_meas,
        std_fixed = std_fixed, std_random = std_random) 
  object$call <- call_rspde_lme
  object$terms <- list(fixed_effects = X_cov)
  object$response <- list(y = y_resp)
  object$formula <- formula
  object$matern_coeff <- matern_coeff
  object$estimation_method <- optim_method
  object$parameterization_latent <- model$parameterization
  object$repl <- repl
  object$optim_controls <- optim_controls
  object$latent_model <- model
  object$null_model <- null_model
  object$start_values <- start_values
  object$loglik <- loglik
  object$niter <- res$counts
  object$response <- y_term
  object$covariates <- cov_term
  object$fitting_time <- time_fit
  object$rspde_order <- rspde_order
  object$time_matern_par <- time_matern_par
  object$improve_hessian <- improve_hessian
  object$time_hessian <- time_hessian
  object$parallel <- parallel
  object$time_par <- time_par
  object$time_data <- time_data
  object$optim_method <- optim_method
  object$time_likelihood <- time_build_likelihood
  object$A_list <- A_list
  object$has_graph <- model$has_graph
  object$which_repl <- which_repl
  object$stationary <- model$stationary
  # if(model_matrix){
    if(ncol(X_cov)>0){
      object$model_matrix <- cbind(y_resp, X_cov)
    } else{
      object$model_matrix <- y_resp
    }
  # }

  class(object) <- "rspde_lme"
  return(object)

}

#' @name logLik.rspde_lme
#' @title log-likelihood for \code{graph_lme} Objects
#' @description Gives the log-likelihood for a fitted mixed effects model with a Whittle-Matern latent model.
#' @param x object of class "rspde_lme" containing results from the fitted model.
#' @param ... further arguments passed to or from other methods.
#' @return log-likelihood at the fitted coefficients.
#' @noRd
#' @method logLik rspde_lme
#' @export 

logLik.rspde_lme <- function(object, ...){
  return(object$loglik)
}


#' @name print.rspde_lme
#' @title Print Method for \code{rspde_lme} Objects
#' @description Provides a brief description of results related to mixed effects with Whittle-Matern latent models.
#' @param x object of class "rspde_lme" containing results from the fitted model.
#' @param ... further arguments passed to or from other methods.
#' @return Called for its side effects.
#' @noRd
#' @method print rspde_lme
#' @export 

print.rspde_lme <- function(x, ...) {
  #
  if(!is.null(x$latent_model)){
    if(x$latent_model$stationary){
      call_name <- "Latent model - Whittle-Matern"
    } else{
      call_name <- "Latent model - Generalized Whittle-Matern"
    }
  } else{
    call_name <- "Linear regression model"
  }

  coeff_fixed <- x$coeff$fixed_effects
  coeff_random <- x$coeff$random_effects
  
  cat("\n")
  cat(call_name)
  cat("\n\n")
  cat("Call:\n", paste(deparse(x$call), sep = "\n", collapse = "\n"),
      "\n\n", sep = "")
  cat(paste0("Fixed effects:", "\n"))
  if(!is.null(coeff_fixed)){
    print(coeff_fixed)
  } else{
    message("No fixed effects")
  }
  cat("\n")
  cat(paste0("Random effects:", "\n"))
  if(!is.null(coeff_random)){
    print(coeff_random)
    if(x$stationary){
        cat(paste0("\n", "Random effects (Matern parameterization):", "\n"))
        print(x$matern_coeff$random_effects)
    }
  } else{
    message("No random effects")
  }
  cat("\n")
  cat(paste0("Measurement error:", "\n"))
  print(x$coeff$measurement_error)
}


#' @name summary.rspde_lme
#' @title Summary Method for \code{rspde_lme} Objects.
#' @description Function providing a summary of results related to mixed effects regression models with Whittle-Matern latent models.
#' @param object an object of class "rspde_lme" containing results from the fitted model.
#' @param all_times Show all computed times.
#' @param ... not used.
#' @return An object of class \code{summary_rspde_lme} containing several
#' informations of a *rspde_lme* object.
#' @method summary rspde_lme
#' @export
summary.rspde_lme <- function(object, all_times = FALSE,...) {
  ans <- list()

  nfixed <- length(object$coeff$fixed_effects)
  nrandom <- length(object$coeff$random_effects)
  model_type <- !object$null_model
  if(model_type){
    if(object$latent_model$stationary){
      call_name <- "Latent model - Whittle-Matern"
    } else{
      call_name <- "Latent model - Generalized Whittle-Matern"
    }
  } else{
    call_name <- "Linear regression model"
  }

  coeff_fixed <- object$coeff$fixed_effects
  coeff_random <- object$coeff$random_effects#
  coeff_meas <- object$coeff$measurement_error

  SEr_fixed <- object$std_errors$std_fixed
  SEr_random <- object$std_errors$std_random
  SEr_meas <- object$std_errors$std_meas

  if(object$stationary){
    coeff <- c(coeff_fixed, coeff_random, object$matern_coeff$random_effects, coeff_meas)
    SEr <- c(SEr_fixed,SEr_random, object$matern_coeff$std_random, SEr_meas)
  } else{
    coeff <- c(coeff_fixed, coeff_random, coeff_meas)
    SEr <- c(SEr_fixed,SEr_random, SEr_meas)
  }

  if(model_type){
    tab <- cbind(coeff, SEr, coeff / SEr, 2 * stats::pnorm(-abs(coeff / SEr)))
    colnames(tab) <- c("Estimate", "Std.error", "z-value", "Pr(>|z|)")
    rownames(tab) <- names(coeff)
    if(object$stationary){
        tab <- list(fixed_effects = tab[seq.int(length.out = nfixed), , drop = FALSE], random_effects = tab[seq.int(length.out = nrandom) + nfixed, , drop = FALSE], 
        random_effects_matern = tab[seq.int(length.out = nrandom) + nrandom + nfixed, , drop = FALSE], 
        meas_error = tab[seq.int(length.out = 1) + nfixed+2*nrandom, , drop = FALSE])
    } else{
      tab <- list(fixed_effects = tab[seq.int(length.out = nfixed), , drop = FALSE], random_effects = tab[seq.int(length.out = nrandom) + nfixed, , drop = FALSE], 
        meas_error = tab[seq.int(length.out = 1) + nfixed+nrandom, , drop = FALSE])
    }

  } else{
    tab <- list(fixed_effects = SEr_fixed, coeff_meas = coeff_meas)
  }

  ans$coefficients <- tab

  ans$model_type <- model_type

  ans$call_name <- call_name

  ans$call <- object$call

  ans$loglik <- object$loglik

  ans$optim_method <- object$optim_method

  ans$niter <- object$niter

  ans$all_times <- all_times

  ans$fitting_time <- object$fitting_time

  ans$improve_hessian <- object$improve_hessian

  ans$time_hessian <- object$time_hessian

  ans$parallel <- object$parallel

  ans$time_par <- object$time_par

  ans$time_data <- object$time_data

  ans$time_matern_par <- object$time_matern_par

  ans$time_likelihood <- object$time_likelihood

  class(ans) <- "summary_rspde_lme"
  ans
}

#' @name print.summary_rspde_lme
#' @title Print Method for \code{summary_rspde_lme} Objects
#' @description Provides a brief description of results related to mixed effects regression models with Whittle-Matern latent models.
#' @param x object of class "summary_rspde_lme" containing results of summary method applied to a fitted model.
#' @param ... further arguments passed to or from other methods.
#' @return Called for its side effects.
#' @noRd
#' @method print summary_rspde_lme
#' @export
print.summary_rspde_lme <- function(x,...) {
  tab <- x$coefficients

  #
  digits <- max(3, getOption("digits") - 3)
  #

  call_name <- x$call_name

  cat("\n")
  cat(call_name)

  cat("\n\n")
  cat("Call:\n", paste(deparse(x$call), sep = "\n", collapse = "\n"),
      "\n", sep = "")


  #
  model_type <- tolower(x$model_type)
  #
  if(model_type){
      if (NROW(tab$fixed_effects)) {
        cat(paste0("\nFixed effects:\n"))
        stats::printCoefmat(tab[["fixed_effects"]], digits = digits, signif.legend = FALSE)
      } else {
        message("\nNo fixed effects. \n")
      }
      #
      if (NROW(tab$random_effects)) {
        cat(paste0("\nRandom effects:\n"))
        stats::printCoefmat(tab[["random_effects"]][,1:3], digits = digits, signif.legend = FALSE)
      } else {
        cat(paste0("\nRandom effects:\n"))
        message("No random effects. \n")
      }
      if (NROW(tab$random_effects_matern)) {
        cat(paste0("\nRandom effects (Matern parameterization):\n"))
        stats::printCoefmat(tab[["random_effects_matern"]][,1:3], digits = digits, signif.legend = FALSE)
      }   
      #
      cat(paste0("\nMeasurement error:\n"))
        stats::printCoefmat(tab[["meas_error"]][1,1:3,drop = FALSE], digits = digits, signif.legend = FALSE)
  } else{
        cat(paste0("\nFixed effects:\n"))
        stats::printCoefmat(tab[["fixed_effects"]], digits = digits, signif.legend = FALSE)

        cat(paste0("\nRandom effects:\n"))
        message("No random effects. \n")
        cat(paste0("\nMeasurement error:\n"))
        print(tab$coeff_meas)

  }
  #
  if (getOption("show.signif.stars")) {
    cat("---\nSignif. codes: ", "0 '***' 0.001 '**' 0.01 '*' 0.05 '.' 0.1 ' ' 1", "\n\n")
  }
  #

  cat("Log-Likelihood: ", x$loglik,"\n")
  if(model_type){
    cat(paste0("Number of function calls by 'optim' = ", x$niter[1],"\n"))
    cat(paste0("Optimization method used in 'optim' = ", x$optim_method,"\n"))
    cat(paste0("\nTime used to:"))
    if(x$all_times){
      cat("\t prepare the data = ", paste(trunc(x$time_data[[1]] * 10^5)/10^5,attr(x$time_data, "units"),"\n"))
      cat("\t build the likelihood = ", paste(trunc(x$time_likelihood[[1]] * 10^5)/10^5,attr(x$time_likelihood, "units"),"\n"))
      cat("\t compute Matern parameterization = ", paste(trunc(x$time_matern_par[[1]] * 10^5)/10^5,attr(x$time_likelihood, "units"),"\n"))      
    }
    cat("\t fit the model = ", paste(trunc(x$fitting_time[[1]] * 10^5)/10^5,attr(x$fitting_time, "units"),"\n"))
    if(x$improve_hessian){
    cat(paste0("\t compute the Hessian = ", paste(trunc(x$time_hessian[[1]] * 10^5)/10^5,attr(x$time_hessian, "units"),"\n")))      
    }
    if(x$parallel){
    cat(paste0("\t set up the parallelization = ", paste(trunc(x$time_par[[1]] * 10^5)/10^5,attr(x$time_par, "units"),"\n")))      
    }

  }
}



#' @name predict.rspde_lme
#' @title Prediction of a mixed effects regression model on a metric graph.
#' @param object The fitted object with the `rspde_lme()` function 
#' @param data A `data.frame` or a `list` containing the covariates, the edge number and the distance on edge
#' for the locations to obtain the prediction.
#' @param loc Prediction locations. Can either be a `data.frame`, a `matrix` or a character vector, that contains the names of the columns of the coordinates of the locations. For models using `metric_graph` objects, plase use `edge_number` and `distance_on_edge` instead.
#' @param mesh Obtain predictions for mesh nodes? The graph must have a mesh, and either `only_latent` is set to TRUE or the model does not have covariates.
# @param repl Which replicates to obtain the prediction. If `NULL` predictions will be obtained for all replicates. Default is `NULL`.
#' @param which_repl Which replicates to use? If `NULL` all replicates will be used.
#' @param compute_variances Set to also TRUE to compute the kriging variances.
#' @param posterior_samples If `TRUE`, posterior samples will be returned.
#' @param n_samples Number of samples to be returned. Will only be used if `sampling` is `TRUE`.
#' @param only_latent Should the posterior samples and predictions be only given to the latent model?
#' @param edge_number Name of the variable that contains the edge number, the default is `edge_number`.
#' @param distance_on_edge Name of the variable that contains the distance on edge, the default is `distance_on_edge`.
#' @param normalized Are the distances on edges normalized?
#' @param return_as_list Should the means of the predictions and the posterior samples be returned as a list, with each replicate being an element?
#' @param return_original_order Should the results be return in the original (input) order or in the order inside the graph?
#' @param ... Not used.
#' @export
#' @method predict rspde_lme

predict.rspde_lme <- function(object, data = NULL, loc = NULL, mesh = FALSE, which_repl = NULL, compute_variances = FALSE, posterior_samples = FALSE,
                               n_samples = 100, only_latent = FALSE, edge_number = "edge_number",
                               distance_on_edge = "distance_on_edge", normalized = FALSE, return_as_list = FALSE, return_original_order = TRUE,
                               ...) {

  if(is.null(data)){
    if(!mesh){
      stop("If 'mesh' is false, you should supply data!")
    }
  }

  out <- list()

  coeff_fixed <- object$coeff$fixed_effects
  coeff_random <- object$coeff$random_effects
  coeff_meas <- object$coeff$measurement_error

  if(inherits(object, "graph_lme")){
    if(object$estimate_nu){
      coeff_random[1] <- coeff_random[1] - 0.5
    }
  }

  if(object$has_graph){
    loc <- cbind(data[[edge_number]], data[[distance_on_edge]])
  } else if(is.character(loc)){
    loc <- data[loc]
  }

  loc <- as.matrix(loc)

  X_cov_initial <- as.matrix(object$model_matrix)[,-1]
  if(ncol(X_cov_initial) > 0){
    if(mesh){
      stop("In the presence of covariates, you should provide the data, including the covariates at the prediction locations. If you only want predictions for the latent model, set 'only_latent' to TRUE.")
    }
  }

  X_cov_pred <- stats::model.matrix(object$covariates, data)
  if(nrow(X_cov_pred) != nrow(as.matrix(loc))){
    stop("Covariates not found in data.")
  }

  if(sum(duplicated(loc)) > 0){
    warning("There are duplicated locations for prediction, we will try to process the data to extract the unique locations,
    along with the corresponding covariates.")
    if(nrow(X_cov_pred) == nrow(loc)){
      data_tmp <- cbind(loc, X_cov_pred)
    }
    data_tmp <- unique(data_tmp) 
    if(sum(duplicated(cbind(data_tmp[,1:ncol(loc)]))) > 0){
      stop("Data processing failed, please provide a data with unique locations.")
    }
  }
  
  
  if(!mesh){
    n_prd <- length(loc[,1])
    # Convert data to normalized
    if(object$has_graph && !normalized){
      loc[,2] <- loc[,2] / object$graph$edge_lengths[loc[,1]]
    }
    Aprd <- object$latent_model$make_A(loc)
  } else{
    Aprd <- Matrix::Diagonal(nrow(object$latent_model$C))
  }

  ## 
  repl_vec <- object$repl
  if(is.null(which_repl)){
    u_repl <- unique(repl_vec)
  } else{
    u_repl <- unique(which_repl)
  }

  ##

  if(all(dim(X_cov_pred) == c(0,1))){
    X_cov_pred <- matrix(1, nrow = nrow(loc), ncol=1)
  }
  if(ncol(X_cov_pred) > 0){
    mu_prd <- X_cov_pred %*% coeff_fixed
  } else{
    mu_prd <- matrix(0, nrow = nrow(loc), ncol=1)
  }

  model_matrix_fit <- object$model_matrix

  model_matrix_fit <- as.matrix(model_matrix_fit)

  if(ncol(model_matrix_fit)>1){
    X_cov_fit <- model_matrix_fit[,2:ncol(model_matrix_fit)]
    mu <- X_cov_fit %*% coeff_fixed
  } else{
    mu <- 0
  }


  Y <- model_matrix_fit[,1] - mu

  model_type <- object$latent_model

  sigma.e <- coeff_meas[[1]]
  sigma_e <- sigma.e

  ## construct Q

  if(object$estimate_nu){
      alpha_est <- coeff_random[1]
      tau_est <- coeff_random[2]
      kappa_est <- coeff_random[3]
  } else{
      tau_est <- coeff_random[1]
      kappa_est <- coeff_random[2]
      alpha_est <- NULL
  }

  new_rspde_obj <- update(object$latent_model,
                        user_alpha = alpha_est,
                        user_kappa = kappa_est,
                        user_tau = tau_est,
                        parameterization = "spde")

                   
  Q <- new_rspde_obj$Q                   

  idx_obs_full <- as.vector(!is.na(Y))

  Aprd <- kronecker(matrix(1, 1, object$rspde_order + 1), Aprd)

  for(repl_y in u_repl){
    idx_repl <- repl_vec == repl_y

    idx_obs <- idx_obs_full[idx_repl]

    y_repl <- Y[idx_repl]
    y_repl <- y_repl[idx_obs]

    A_repl <- object$A_list[[repl_y]]

    Q_xgiveny <- t(A_repl) %*% A_repl/sigma_e^2 + Q

    mu_krig <- solve(Q_xgiveny,as.vector(t(A_repl) %*% y_repl / sigma_e^2))

    # mu_krig <- mu_krig[(gap+1):length(mu_krig)]

    mu_krig <- Aprd %*% mu_krig

    if(!only_latent){
      mu_krig <- mu_prd + mu_krig
    }

    mean_tmp <- as.vector(mu_krig)
        
    if(!return_as_list){
      out$mean <- c(out$mean, mean_tmp)
      out$repl <- c(out$repl, rep(repl_y,n_prd))
    } else{
      out$mean[[repl_y]] <- mean_tmp
    }

    if (compute_variances) {
        post_cov <- Aprd%*%solve(Q_xgiveny, t(Aprd))
        var_tmp <- pmax(diag(post_cov),0)

      if(!return_as_list){
        out$variance <- rep(var_tmp, length(u_repl))
      } else {
          for(repl_y in u_repl){
            out$variance[[repl_y]] <- var_tmp
          }
      }
    }

    if(posterior_samples){
      mean_tmp <- as.vector(mu_krig)
      post_cov <- Aprd%*%solve(Q_xgiveny, t(Aprd))
      Z <- rnorm(dim(post_cov)[1] * n_samples)
      dim(Z) <- c(dim(post_cov)[1], n_samples)
      LQ <- chol(forceSymmetric(post_cov))
      X <- LQ %*% Z
      X <- X + mean_tmp
      if(!only_latent){
        X <- X + matrix(rnorm(n_samples * length(mean_tmp), sd = sigma.e), nrow = length(mean_tmp))
      } else{
        X <- X - as.vector(mu_prd)
      }

      if(!return_as_list){
        out$samples <- rbind(out$samples, X)
      } else{
        out$samples[[repl_y]] <- X
      }
    }
  }

  return(out)
}


