% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apply_transformation_matrix.R
\name{apply_transformation_matrix}
\alias{apply_transformation_matrix}
\title{Apply transformation matrix to a set of columns}
\usage{
apply_transformation_matrix(
  data,
  mat,
  cols,
  origin = NULL,
  origin_fn = NULL,
  suffix = "_transformed",
  keep_original = TRUE,
  origin_col_name = ".origin",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{mat}{Transformation \code{matrix}. Must have the same number of columns as \code{`cols`}.}

\item{cols}{Columns to mutate values of. Must be specified when \code{`data`} is a \code{data.frame}.}

\item{origin}{Coordinates of the origin. \code{Vector} with the same number
of elements as \code{`cols`} (i.e. origin_x, origin_y, ...).
Ignored when \code{`origin_fn`} is not \code{NULL}.}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{suffix}{Suffix to add to the names of the generated columns.

Use an empty string (i.e. \code{""}) to overwrite the original columns.}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with the new, transformed columns and the origin coordinates.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Perform \link[base:matmult]{matrix multiplication} with a transformation matrix and a set of \code{data.frame} columns.

The data points in \code{`data`} are moved prior to the transformation, to bring
the origin to \code{0} in all dimensions. After the transformation, the
inverse move is applied to bring the origin back to its original position. See \code{`Details`} section.

The columns in \code{`data`} are transposed, making the operation (without the origin movement):
\deqn{mat · data[, cols]^T}

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and transforming around e.g. the centroid
of each group.
}
\details{
Example with 2 columns (\code{x}, \code{y}) and a 2x2 transformation matrix:

\itemize{
\item{Move origin to \code{(0, 0)}:

\code{x = x - origin_x}

\code{y = y - origin_y}}
\item{Convert to transposed matrix:

\code{data_mat = rbind(x, y)}}
\item{Matrix multiplication:

\code{transformed = mat \%*\% data_mat}}
\item{Move origin to original position (after extraction from \code{transformed}):

\code{x = x + origin_x}

\code{y = y + origin_y}}
}
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(ggplot2)

# Set seed
set.seed(3)

# Create a data frame
df <- data.frame(
  "x" = 1:12,
  "y" = 13:24,
  "z" = runif(12),
  "g" = c(
    1, 1, 1, 1, 2, 2,
    2, 2, 3, 3, 3, 3
  )
)

# Apply identity matrix
mat <- matrix(c(1, 0, 0, 0, 1, 0, 0, 0, 1), nrow = 3)
apply_transformation_matrix(
  data = df,
  mat = mat,
  cols = c("x", "y", "z"),
  origin = c(0, 0, 0)
)

# Apply rotation matrix
# 90 degrees around z-axis
# Origin is the most centered point
mat <- matrix(c(0, 1, 0, -1, 0, 0, 0, 0, 1), nrow = 3)
res <- apply_transformation_matrix(
  data = df,
  mat = mat,
  cols = c("x", "y", "z"),
  origin_fn = most_centered
)

# Plot the rotation
# z wasn't changed so we plot x and y
res \%>\%
  ggplot(aes(x = x, y = y)) +
  geom_point() +
  geom_point(aes(x = x_transformed, y = y_transformed)) +
  theme_minimal()

# Apply rotation matrix to grouped data frame
# Around centroids
# Same matrix as before
res <- apply_transformation_matrix(
  data = dplyr::group_by(df, g),
  mat = mat,
  cols = c("x", "y", "z"),
  origin_fn = centroid
)

# Plot the rotation
res \%>\%
  ggplot(aes(x = x, y = y, color = g)) +
  geom_point() +
  geom_point(aes(x = x_transformed, y = y_transformed)) +
  theme_minimal()
}
\seealso{
Other mutate functions: 
\code{\link{cluster_groups}()},
\code{\link{dim_values}()},
\code{\link{expand_distances_each}()},
\code{\link{expand_distances}()},
\code{\link{flip_values}()},
\code{\link{roll_values}()},
\code{\link{rotate_2d}()},
\code{\link{rotate_3d}()},
\code{\link{shear_2d}()},
\code{\link{shear_3d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{mutate functions}
