% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hexagonalize.R
\name{hexagonalize}
\alias{hexagonalize}
\title{Create x-coordinates so the points form a hexagon}
\usage{
hexagonalize(
  data,
  y_col = NULL,
  .min = NULL,
  .max = NULL,
  offset_x = 0,
  keep_original = TRUE,
  x_col_name = ".hexagon_x",
  edge_col_name = ".edge",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{y_col}{Name of column in \code{`data`} with y-coordinates to create x-coordinates for.}

\item{.min}{Minimum y-coordinate. If \code{NULL}, it is inferred by the given y-coordinates.}

\item{.max}{Maximum y-coordinate. If \code{NULL}, it is inferred by the given y-coordinates.}

\item{offset_x}{Value to offset the x-coordinates by.}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{x_col_name}{Name of new column with the x-coordinates.}

\item{edge_col_name}{Name of new column with the edge identifiers. If \code{NULL}, no column is added.

Numbering is clockwise and starts at the upper-right edge.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with the added x-coordinates and an identifier
for the edge the data point is a part of.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Create the x-coordinates for a \code{vector} of y-coordinates such that
they form a hexagon.

This will likely look most like a hexagon when the y-coordinates are somewhat equally distributed,
e.g. a uniform distribution.
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(purrr)
library(ggplot2)

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "y" = runif(200),
  "g" = factor(rep(1:5, each = 40))
)

# Hexagonalize 'y'
df_hex <- hexagonalize(df, y_col = "y")
df_hex

# Plot hexagon
df_hex \%>\%
  ggplot(aes(x = .hexagon_x, y = y, color = .edge)) +
  geom_point() +
  theme_minimal()

#
# Grouped hexagonalization
#

# Hexagonalize 'y' for each group
# First cluster the groups a bit to move the
# hexagons away from each other
df_hex <- df \%>\%
  cluster_groups(
    cols = "y",
    group_cols = "g",
    suffix = "",
    overwrite = TRUE
  ) \%>\%
  dplyr::group_by(g) \%>\%
  hexagonalize(
    y_col = "y",
    overwrite = TRUE
  )

# Plot hexagons
df_hex \%>\%
  ggplot(aes(x = .hexagon_x, y = y, color = g)) +
  geom_point() +
  theme_minimal()

#
# Specifying minimum value
#

# Specify minimum value manually
df_hex <- hexagonalize(df, y_col = "y", .min = -2)
df_hex

# Plot hexagon
df_hex \%>\%
  ggplot(aes(x = .hexagon_x, y = y, color = .edge)) +
  geom_point() +
  theme_minimal()

#
# Multiple hexagons by contraction
#

# Start by hexagonalizing 'y'
df_hex <- hexagonalize(df, y_col = "y")

# Contract '.hexagon_x' and 'y' towards the centroid
# To contract with multiple multipliers at once,
# we wrap the call in purrr::map_dfr
df_expanded <- purrr::map_dfr(
  .x = c(1, 0.75, 0.5, 0.25, 0.125),
  .f = function(mult) {
    expand_distances(
      data = df_hex,
      cols = c(".hexagon_x", "y"),
      multiplier = mult,
      origin_fn = centroid,
      overwrite = TRUE
    )
  }
)
df_expanded

df_expanded \%>\%
  ggplot(aes(
    x = .hexagon_x_expanded, y = y_expanded,
    color = .edge, alpha = .multiplier
  )) +
  geom_point() +
  theme_minimal()
}
\seealso{
Other forming functions: 
\code{\link{circularize}()},
\code{\link{square}()},
\code{\link{triangularize}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{forming functions}
