library(rearrr)
context("pair_extremes()")


test_that("rearrange() with method pair_extremes works", {

  # Create data frame
  xpectr::set_test_seed(1)
  df <- data.frame(
    "participant" = factor(c(1, 3, 5, 6, 7, 8, 9)),
    "score" = c(79, 85, 140, 69, 87, 92, 87),
    stringsAsFactors = FALSE
  )
  df <- df %>% dplyr::arrange(score)

  # with unequal number of rows
  df_rearranged <- pair_extremes(
    df,
    unequal_method = "first"
  )
  expect_equal(df_rearranged$.pair, factor(c(1, 2, 2, 3, 3, 4, 4)))
  expect_equal(df_rearranged$participant, factor(c(6, 1, 5, 3, 8, 7, 9)))
  expect_equal(df_rearranged$score, c(69, 79, 140, 85, 92, 87, 87))

  df_rearranged <- pair_extremes(
    df,
    unequal_method = "middle"
  )
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 4, 4)))
  expect_equal(df_rearranged$participant, factor(c(6, 5, 1, 8, 7, 3, 9)))
  expect_equal(df_rearranged$score, c(69, 140, 79, 92, 87, 85, 87))

  df_rearranged <- pair_extremes(
    df,
    unequal_method = "last"
  )
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 3, 4)))
  expect_equal(df_rearranged$participant, factor(c(6, 8, 1, 9, 3, 7, 5)))
  expect_equal(df_rearranged$score, c(69, 92, 79, 87, 85, 87, 140))


  # with equal number of rows

  df <- df %>%
    dplyr::filter(dplyr::row_number() != 5) %>%
    droplevels()

  df_rearranged <- pair_extremes(
    df,
    unequal_method = "first"
  )
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 3)))
  expect_equal(df_rearranged$participant, factor(c(6, 5, 1, 8, 3, 7)))
  expect_equal(df_rearranged$score, c(69, 140, 79, 92, 85, 87))

  df_rearranged <- pair_extremes(
    df,
    unequal_method = "middle"
  )
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 3)))
  expect_equal(df_rearranged$participant, factor(c(6, 5, 1, 8, 3, 7)))
  expect_equal(df_rearranged$score, c(69, 140, 79, 92, 85, 87))

  df_rearranged <- pair_extremes(
    df,
    unequal_method = "last"
  )
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 3)))
  expect_equal(df_rearranged$participant, factor(c(6, 5, 1, 8, 3, 7)))
  expect_equal(df_rearranged$score, c(69, 140, 79, 92, 85, 87))
})

test_that("rearrange() with method pair_extremes throws expected errors", {
  df <- data.frame("a" = c(1,2,3))

  ## Testing 'rearrange(df, method = "pair_extremes", uneq...'              ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19148 <- xpectr::capture_side_effects(
    pair_extremes(df,
                  unequal_method = "none"
  ), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unequal_method': Must be a subset of set {first,middle,last}."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19148[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)
  ## Finished testing 'rearrange(df, method = "pair_extremes", uneq...'     ####

})

test_that("create_rearrange_factor_pair_extremes_() works", {

  # unequal_method "middle"

  # Equal size
  expect_equal(create_rearrange_factor_pair_extremes_(40, unequal_method = "middle"), c(c(1:20), rev(c(1:20))))
  expect_equal(create_rearrange_factor_pair_extremes_(200, unequal_method = "middle"), c(c(1:100), rev(c(1:100))))

  # Unequal size
  expect_equal(create_rearrange_factor_pair_extremes_(41, unequal_method = "middle"), c(c(1:10, 12:21), 11, rev(c(1:10, 12:21))))
  expect_equal(create_rearrange_factor_pair_extremes_(201, unequal_method = "middle"), c(c(1:50, 52:101), 51, rev(c(1:50, 52:101))))

  # unequal_method "first"

  # Equal size
  expect_equal(create_rearrange_factor_pair_extremes_(40, unequal_method = "first"), c(c(1:20), rev(c(1:20))))
  expect_equal(create_rearrange_factor_pair_extremes_(200, unequal_method = "first"), c(c(1:100), rev(c(1:100))))

  # Unequal size
  expect_equal(create_rearrange_factor_pair_extremes_(41, unequal_method = "first"), c(1, c(c(1:20), rev(c(1:20))) + 1))
  expect_equal(create_rearrange_factor_pair_extremes_(201, unequal_method = "first"), c(1, c(c(1:100), rev(c(1:100))) + 1))

  # unequal_method "last"

  # Equal size
  expect_equal(create_rearrange_factor_pair_extremes_(40, unequal_method = "last"), c(c(1:20), rev(c(1:20))))
  expect_equal(create_rearrange_factor_pair_extremes_(200, unequal_method = "last"), c(c(1:100), rev(c(1:100))))

  # Unequal size
  expect_equal(create_rearrange_factor_pair_extremes_(41, unequal_method = "last"), c(c(c(1:20), rev(c(1:20))), 21))
  expect_equal(create_rearrange_factor_pair_extremes_(201, unequal_method = "last"), c(c(c(1:100), rev(c(1:100))), 101))
})

test_that("rearrange() with method pair_extremes works on vectors", {

  # Create data frame
  xpectr::set_test_seed(1)
  score <- sort(c(79, 85, 140, 69, 87, 92, 87))

  # with unequal number of rows
  df_rearranged <- pair_extremes(
    score,
    unequal_method = "first"
  ) %>%
    dplyr::rename(score = Value)
  expect_equal(df_rearranged$.pair, factor(c(1, 2, 2, 3, 3, 4, 4)))
  expect_equal(df_rearranged$score, c(69, 79, 140, 85, 92, 87, 87))

  df_rearranged <- pair_extremes(
    score,
    unequal_method = "middle"
  ) %>%
    dplyr::rename(score = Value)
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 4, 4)))
  expect_equal(df_rearranged$score, c(69, 140, 79, 92, 87, 85, 87))

  df_rearranged <- pair_extremes(
    score,
    unequal_method = "last"
  ) %>%
    dplyr::rename(score = Value)
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 3, 4)))
  expect_equal(df_rearranged$score, c(69, 92, 79, 87, 85, 87, 140))


  # with equal number of rows
  score <- c(head(score, 4), tail(score, 2))

  df_rearranged <- pair_extremes(
    score,
    unequal_method = "first"
  ) %>%
    dplyr::rename(score = Value)
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 3)))
  expect_equal(df_rearranged$score, c(69, 140, 79, 92, 85, 87))

  df_rearranged <- pair_extremes(
    score,
    unequal_method = "middle"
  ) %>%
    dplyr::rename(score = Value)
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 3)))
  expect_equal(df_rearranged$score, c(69, 140, 79, 92, 85, 87))

  df_rearranged <- pair_extremes(
    score,
    unequal_method = "last"
  ) %>%
    dplyr::rename(score = Value)
  expect_equal(df_rearranged$.pair, factor(c(1, 1, 2, 2, 3, 3)))
  expect_equal(df_rearranged$score, c(69, 140, 79, 92, 85, 87))
})

test_that("fuzz testing pair_extremes method for rearrange()", {
  xpectr::set_test_seed(42)

  # Create a data frame
  df <- data.frame(
    "index" = 1:9,
    "A" = sample(1:9),
    "B" = runif(9),
    "C" = LETTERS[1:9],
    stringsAsFactors = FALSE
  )

  # Generate expectations for 'rearrange' for pair_extremes method
  # Tip: comment out the gxs_function() call
  # so it is easy to regenerate the tests
  xpectr::set_test_seed(42)
  # xpectr::gxs_function(
  #   fn = pair_extremes,
  #   args_values = list(
  #     "data" = list(df, head(df, 8), c(1,2,3,4,5,6,7), c(1,2,3,4),
  #                   factor(c(1,2,3,4,5,6,7,1,2,3)), list(1,2,3), NA, 1),
  #     "col" = list(NULL, "A", "B", "C"),
  #     "unequal_method" = list("middle", "first", "last"),
  #     "shuffle_members" = list(FALSE, TRUE),
  #     "shuffle_pairs" = list(FALSE, TRUE),
  #     "factor_name" = list(NULL, ".pair", "A", 1, NA),
  #     "overwrite" = list(TRUE)
  #   ),
  #   extra_combinations = list(
  #     list("data" = c(1,2,3,4), "col" = "A"),
  #     list("data" = c(1,2,3,4), "factor_name" = "another_name"),
  #     list("shuffle_members" = TRUE, "shuffle_pairs" = TRUE),
  #     list("factor_name" = "A", "overwrite" = FALSE)
  #   ),
  #   indentation = 2
  # )


  ## Testing 'pair_extremes'                                                  ####
  ## Initially generated by xpectr
  # Testing different combinations of argument values

  # Testing pair_extremes(data = df, col = NULL, unequal...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19148 <- pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_19148),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19148[["index"]],
    c(1, 9, 2, 8, 5, 3, 7, 4, 6),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["A"]],
    c(9, 2, 8, 1, 4, 3, 6, 5, 7),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["B"]],
    c(0.70506, 0.11749, 0.45774, 0.97823, 0.25543, 0.71911, 0.94001,
      0.93467, 0.46229),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["C"]],
    c("A", "I", "B", "H", "E", "C", "G", "D", "F"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_19148),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19148),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19148),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19148),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19148)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = head(df, 8), col = NULL...
  # Changed from baseline: data = head(df, 8)
  xpectr::set_test_seed(42)
  # Assigning output
  output_19370 <- pair_extremes(data = head(df, 8), col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_19370),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19370[["index"]],
    c(1, 8, 2, 7, 3, 6, 4, 5),
    tolerance = 1e-4)
  expect_equal(
    output_19370[["A"]],
    c(9, 1, 8, 6, 3, 7, 5, 4),
    tolerance = 1e-4)
  expect_equal(
    output_19370[["B"]],
    c(0.70506, 0.97823, 0.45774, 0.94001, 0.71911, 0.46229, 0.93467,
      0.25543),
    tolerance = 1e-4)
  expect_equal(
    output_19370[["C"]],
    c("A", "H", "B", "G", "C", "F", "D", "E"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_19370),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19370),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19370),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19370),
    c(8L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19370)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = c(1, 2, 3, 4, 5, 6, 7),...
  # Changed from baseline: data = c(1, 2, 3, 4, ...
  xpectr::set_test_seed(42)
  # Assigning output
  output_12861 <- pair_extremes(data = c(1, 2, 3, 4, 5, 6, 7), col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_12861),
    "numeric",
    fixed = TRUE)
  # Testing type
  expect_type(
    output_12861,
    type = "double")
  # Testing values
  expect_equal(
    output_12861,
    c(1, 7, 2, 6, 4, 3, 5),
    tolerance = 1e-4)
  # Testing names
  expect_equal(
    names(output_12861),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(output_12861),
    7L)
  # Testing sum of element lengths
  expect_equal(
    sum(xpectr::element_lengths(output_12861)),
    7L)

  # Testing pair_extremes(data = c(1, 2, 3, 4), col = NU...
  # Changed from baseline: data = c(1, 2, 3, 4)
  xpectr::set_test_seed(42)
  # Assigning output
  output_18304 <- pair_extremes(data = c(1, 2, 3, 4), col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_18304),
    "numeric",
    fixed = TRUE)
  # Testing type
  expect_type(
    output_18304,
    type = "double")
  # Testing values
  expect_equal(
    output_18304,
    c(1, 4, 2, 3),
    tolerance = 1e-4)
  # Testing names
  expect_equal(
    names(output_18304),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(output_18304),
    4L)
  # Testing sum of element lengths
  expect_equal(
    sum(xpectr::element_lengths(output_18304)),
    4L)

  # Testing pair_extremes(data = factor(c(1, 2, 3, 4, 5,...
  # Changed from baseline: data = factor(c(1, 2,...
  xpectr::set_test_seed(42)
  # Assigning output
  output_16417 <- pair_extremes(data = factor(c(1, 2, 3, 4, 5, 6, 7, 1, 2, 3)), col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing is factor
  expect_true(
    is.factor(output_16417))
  # Testing values
  expect_equal(
    as.character(output_16417),
    c("1", "7", "1", "6", "2", "5", "2", "4", "3", "3"),
    fixed = TRUE)
  # Testing names
  expect_equal(
    names(output_16417),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(output_16417),
    10L)
  # Testing number of levels
  expect_equal(
    nlevels(output_16417),
    7L)
  # Testing levels
  expect_equal(
    levels(output_16417),
    c("1", "2", "3", "4", "5", "6", "7"),
    fixed = TRUE)

  # Testing pair_extremes(data = list(1, 2, 3), col = NU...
  # Changed from baseline: data = list(1, 2, 3)
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15190 <- xpectr::capture_side_effects(pair_extremes(data = list(1, 2, 3), col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15190[['error']]),
    xpectr::strip("1 assertions failed:\n * when 'data' is not a data.frame, it cannot be a list."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15190[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = NA, col = NULL, unequal...
  # Changed from baseline: data = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17365 <- xpectr::capture_side_effects(pair_extremes(data = NA, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17365[['error']]),
    xpectr::strip("Assertion failed. One of the following must apply:\n * checkmate::check_data_frame(data): Must be of type 'data.frame', not 'logical'\n * checkmate::check_vector(data): Contains missing values (element 1)\n * checkmate::check_factor(data): Contains missing values (element 1)"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17365[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = 1, col = NULL, unequal_...
  # Changed from baseline: data = 1
  xpectr::set_test_seed(42)
  # Assigning output
  output_11346 <- pair_extremes(data = 1, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_11346),
    "numeric",
    fixed = TRUE)
  # Testing type
  expect_type(
    output_11346,
    type = "double")
  # Testing values
  expect_equal(
    output_11346,
    1,
    tolerance = 1e-4)
  # Testing names
  expect_equal(
    names(output_11346),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(output_11346),
    1L)
  # Testing sum of element lengths
  expect_equal(
    sum(xpectr::element_lengths(output_11346)),
    1L)

  # Testing pair_extremes(data = NULL, col = NULL, unequ...
  # Changed from baseline: data = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16569 <- xpectr::capture_side_effects(pair_extremes(data = NULL, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16569[['error']]),
    xpectr::strip("Assertion failed. One of the following must apply:\n * checkmate::check_data_frame(data): Must be of type 'data.frame', not 'NULL'\n * checkmate::check_vector(data): Must be of type 'vector', not 'NULL'\n * checkmate::check_factor(data): Must be of type 'factor', not 'NULL'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16569[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = c(1, 2, 3, 4), col = "A...
  # Changed from baseline: data, col
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17050 <- xpectr::capture_side_effects(pair_extremes(data = c(1, 2, 3, 4), col = "A", unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17050[['error']]),
    xpectr::strip("1 assertions failed:\n * when 'data' is not a data.frame, 'col(s)' must be 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17050[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = c(1, 2, 3, 4), col = NU...
  # Changed from baseline: data, factor_name
  xpectr::set_test_seed(42)
  # Assigning output
  output_14577 <- pair_extremes(data = c(1, 2, 3, 4), col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = "another_name", overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_14577),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_14577[["Value"]],
    c(1, 4, 2, 3),
    tolerance = 1e-4)
  expect_equal(
    output_14577[["another_name"]],
    structure(c(1L, 1L, 2L, 2L), .Label = c("1", "2"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_14577),
    c("Value", "another_name"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_14577),
    c("numeric", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_14577),
    c("double", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_14577),
    c(4L, 2L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_14577)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = "C", unequal_...
  # Changed from baseline: col = "C"
  xpectr::set_test_seed(42)
  # Assigning output
  output_17191 <- pair_extremes(data = df, col = "C", unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_17191),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_17191[["index"]],
    c(1, 9, 2, 8, 5, 3, 7, 4, 6),
    tolerance = 1e-4)
  expect_equal(
    output_17191[["A"]],
    c(9, 2, 8, 1, 4, 3, 6, 5, 7),
    tolerance = 1e-4)
  expect_equal(
    output_17191[["B"]],
    c(0.70506, 0.11749, 0.45774, 0.97823, 0.25543, 0.71911, 0.94001,
      0.93467, 0.46229),
    tolerance = 1e-4)
  expect_equal(
    output_17191[["C"]],
    c("A", "I", "B", "H", "E", "C", "G", "D", "F"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_17191),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_17191),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_17191),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_17191),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_17191)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = "A", unequal_...
  # Changed from baseline: col = "A"
  xpectr::set_test_seed(42)
  # Assigning output
  output_19346 <- pair_extremes(data = df, col = "A", unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_19346),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19346[["index"]],
    c(8, 1, 9, 2, 4, 3, 6, 5, 7),
    tolerance = 1e-4)
  expect_equal(
    output_19346[["A"]],
    c(1, 9, 2, 8, 5, 3, 7, 4, 6),
    tolerance = 1e-4)
  expect_equal(
    output_19346[["B"]],
    c(0.97823, 0.70506, 0.11749, 0.45774, 0.93467, 0.71911, 0.46229,
      0.25543, 0.94001),
    tolerance = 1e-4)
  expect_equal(
    output_19346[["C"]],
    c("H", "A", "I", "B", "D", "C", "F", "E", "G"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_19346),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19346),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19346),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19346),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19346)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = "B", unequal_...
  # Changed from baseline: col = "B"
  xpectr::set_test_seed(42)
  # Assigning output
  output_12554 <- pair_extremes(data = df, col = "B", unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_12554),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_12554[["index"]],
    c(9, 8, 5, 7, 1, 2, 4, 6, 3),
    tolerance = 1e-4)
  expect_equal(
    output_12554[["A"]],
    c(2, 1, 4, 6, 9, 8, 5, 7, 3),
    tolerance = 1e-4)
  expect_equal(
    output_12554[["B"]],
    c(0.11749, 0.97823, 0.25543, 0.94001, 0.70506, 0.45774, 0.93467,
      0.46229, 0.71911),
    tolerance = 1e-4)
  expect_equal(
    output_12554[["C"]],
    c("I", "H", "E", "G", "A", "B", "D", "F", "C"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_12554),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_12554),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_12554),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_12554),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_12554)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: unequal_method = "first"
  xpectr::set_test_seed(42)
  # Assigning output
  output_14622 <- pair_extremes(data = df, col = NULL, unequal_method = "first", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_14622),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_14622[["index"]],
    c(1, 2, 9, 3, 8, 4, 7, 5, 6),
    tolerance = 1e-4)
  expect_equal(
    output_14622[["A"]],
    c(9, 8, 2, 3, 1, 5, 6, 4, 7),
    tolerance = 1e-4)
  expect_equal(
    output_14622[["B"]],
    c(0.70506, 0.45774, 0.11749, 0.71911, 0.97823, 0.93467, 0.94001,
      0.25543, 0.46229),
    tolerance = 1e-4)
  expect_equal(
    output_14622[["C"]],
    c("A", "B", "I", "C", "H", "D", "G", "E", "F"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_14622),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_14622),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_14622),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_14622),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_14622)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: unequal_method = "last"
  xpectr::set_test_seed(42)
  # Assigning output
  output_19400 <- pair_extremes(data = df, col = NULL, unequal_method = "last", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_19400),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19400[["index"]],
    c(1, 8, 2, 7, 3, 6, 4, 5, 9),
    tolerance = 1e-4)
  expect_equal(
    output_19400[["A"]],
    c(9, 1, 8, 6, 3, 7, 5, 4, 2),
    tolerance = 1e-4)
  expect_equal(
    output_19400[["B"]],
    c(0.70506, 0.97823, 0.45774, 0.94001, 0.71911, 0.46229, 0.93467,
      0.25543, 0.11749),
    tolerance = 1e-4)
  expect_equal(
    output_19400[["C"]],
    c("A", "H", "B", "G", "C", "F", "D", "E", "I"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_19400),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19400),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19400),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19400),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19400)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: unequal_method = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19782 <- xpectr::capture_side_effects(pair_extremes(data = df, col = NULL, unequal_method = NULL, shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19782[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unequal_method': Must be of type 'string', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19782[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: shuffle_members = TRUE
  xpectr::set_test_seed(42)
  # Assigning output
  output_11174 <- pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = TRUE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_11174),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_11174[["index"]],
    c(9, 1, 8, 2, 5, 3, 7, 6, 4),
    tolerance = 1e-4)
  expect_equal(
    output_11174[["A"]],
    c(2, 9, 1, 8, 4, 3, 6, 7, 5),
    tolerance = 1e-4)
  expect_equal(
    output_11174[["B"]],
    c(0.11749, 0.70506, 0.97823, 0.45774, 0.25543, 0.71911, 0.94001,
      0.46229, 0.93467),
    tolerance = 1e-4)
  expect_equal(
    output_11174[["C"]],
    c("I", "A", "H", "B", "E", "C", "G", "F", "D"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_11174),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11174),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11174),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11174),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11174)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: shuffle_members = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14749 <- xpectr::capture_side_effects(pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = NULL, shuffle_pairs = FALSE, factor_name = NULL, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14749[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'shuffle_members': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14749[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: shuffle_members, shuf...
  xpectr::set_test_seed(42)
  # Assigning output
  output_15603 <- pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = TRUE, shuffle_pairs = TRUE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_15603),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_15603[["index"]],
    c(5, 6, 4, 9, 1, 2, 8, 3, 7),
    tolerance = 1e-4)
  expect_equal(
    output_15603[["A"]],
    c(4, 7, 5, 2, 9, 8, 1, 3, 6),
    tolerance = 1e-4)
  expect_equal(
    output_15603[["B"]],
    c(0.25543, 0.46229, 0.93467, 0.11749, 0.70506, 0.45774, 0.97823,
      0.71911, 0.94001),
    tolerance = 1e-4)
  expect_equal(
    output_15603[["C"]],
    c("E", "F", "D", "I", "A", "B", "H", "C", "G"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_15603),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_15603),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_15603),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_15603),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_15603)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: shuffle_pairs = TRUE
  xpectr::set_test_seed(42)
  # Assigning output
  output_19040 <- pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = TRUE, factor_name = NULL, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_19040),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19040[["index"]],
    c(5, 4, 6, 1, 9, 2, 8, 3, 7),
    tolerance = 1e-4)
  expect_equal(
    output_19040[["A"]],
    c(4, 5, 7, 9, 2, 8, 1, 3, 6),
    tolerance = 1e-4)
  expect_equal(
    output_19040[["B"]],
    c(0.25543, 0.93467, 0.46229, 0.70506, 0.11749, 0.45774, 0.97823,
      0.71911, 0.94001),
    tolerance = 1e-4)
  expect_equal(
    output_19040[["C"]],
    c("E", "D", "F", "A", "I", "B", "H", "C", "G"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_19040),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19040),
    c("integer", "integer", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19040),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19040),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19040)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: shuffle_pairs = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_11387 <- xpectr::capture_side_effects(pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = NULL, factor_name = NULL, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11387[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'shuffle_pairs': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11387[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: factor_name = ".pair"
  xpectr::set_test_seed(42)
  # Assigning output
  output_19888 <- pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = ".pair", overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_19888),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19888[["index"]],
    c(1, 9, 2, 8, 5, 3, 7, 4, 6),
    tolerance = 1e-4)
  expect_equal(
    output_19888[["A"]],
    c(9, 2, 8, 1, 4, 3, 6, 5, 7),
    tolerance = 1e-4)
  expect_equal(
    output_19888[["B"]],
    c(0.70506, 0.11749, 0.45774, 0.97823, 0.25543, 0.71911, 0.94001,
      0.93467, 0.46229),
    tolerance = 1e-4)
  expect_equal(
    output_19888[["C"]],
    c("A", "I", "B", "H", "E", "C", "G", "D", "F"),
    fixed = TRUE)
  expect_equal(
    output_19888[[".pair"]],
    structure(c(1L, 1L, 2L, 2L, 3L, 4L, 4L, 5L, 5L), .Label = c("1",
      "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19888),
    c("index", "A", "B", "C", ".pair"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19888),
    c("integer", "integer", "numeric", "character", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19888),
    c("integer", "integer", "double", "character", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19888),
    c(9L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19888)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: factor_name = "A"
  xpectr::set_test_seed(42)
  # Assigning output
  output_19466 <- pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = "A", overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_19466),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19466[["index"]],
    c(1, 9, 2, 8, 5, 3, 7, 4, 6),
    tolerance = 1e-4)
  expect_equal(
    output_19466[["A"]],
    structure(c(1L, 1L, 2L, 2L, 3L, 4L, 4L, 5L, 5L), .Label = c("1",
      "2", "3", "4", "5"), class = "factor"))
  expect_equal(
    output_19466[["B"]],
    c(0.70506, 0.11749, 0.45774, 0.97823, 0.25543, 0.71911, 0.94001,
      0.93467, 0.46229),
    tolerance = 1e-4)
  expect_equal(
    output_19466[["C"]],
    c("A", "I", "B", "H", "E", "C", "G", "D", "F"),
    fixed = TRUE)
  # Testing column names
  expect_equal(
    names(output_19466),
    c("index", "A", "B", "C"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19466),
    c("integer", "factor", "numeric", "character"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19466),
    c("integer", "integer", "double", "character"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19466),
    c(9L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19466)),
    character(0),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: factor_name = 1
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10824 <- xpectr::capture_side_effects(pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = 1, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10824[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'factor_name': Must be of type 'string' (or 'NULL'), not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10824[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: factor_name = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15142 <- xpectr::capture_side_effects(pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NA, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15142[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'factor_name': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15142[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: factor_name, overwrite
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13902 <- xpectr::capture_side_effects(pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = "A", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13902[['error']]),
    xpectr::strip("1 assertions failed:\n * The column 'A' already exists and 'overwrite' is disabled."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13902[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing pair_extremes(data = df, col = NULL, unequal...
  # Changed from baseline: overwrite = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19057 <- xpectr::capture_side_effects(pair_extremes(data = df, col = NULL, unequal_method = "middle", shuffle_members = FALSE, shuffle_pairs = FALSE, factor_name = NULL, overwrite = NULL), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19057[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'overwrite': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19057[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  ## Finished testing 'pair_extremes'                                         ####
  #

})

test_that("testing pair_extremes_vec()", {
  xpectr::set_test_seed(42)


  ## Testing 'pair_extremes_vec(data = runif(9), unequal_m...'              ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Assigning output
  output_19148 <- pair_extremes_vec(data = runif(9), unequal_method = "first", shuffle_members = TRUE, shuffle_pairs = TRUE)
  # Testing class
  expect_equal(
    class(output_19148),
    "numeric",
    fixed = TRUE)
  # Testing type
  expect_type(
    output_19148,
    type = "double")
  # Testing values
  expect_equal(
    output_19148,
    c(0.64175, 0.83045, 0.28614, 0.93708, 0.91481, 0.5191, 0.65699,
      0.73659, 0.13467),
    tolerance = 1e-4)
  # Testing names
  expect_equal(
    names(output_19148),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(output_19148),
    9L)
  # Testing sum of element lengths
  expect_equal(
    sum(xpectr::element_lengths(output_19148)),
    9L)
  ## Finished testing 'pair_extremes_vec(data = runif(9), unequal_m...'     ####


})
