#' repo: The R objects resource manager
#'
#' @details
#' The repo package is meant to help with the management of R data
#' files. It builds one (or more) centralized repository where R
#' objects are stored together with corresponding annotations, tags,
#' dependency notes, provenance traces. It also provides navigation
#' tools to easily locate and load previously stored resources.
#' 
#' Create a new repository with \code{rp <- repo_open()}.
#'
#' Given the object rp of class \code{repo}, the \code{repo} function
#' \code{foo} can be called in two ways: 1) \code{repo_foo(rp)} or 2)
#' \code{rp$foo()}.
#'
#' For a complete list of functions, use \code{library(help = "repo")}.
#' 
#' @docType package
#' @name repo-package
#' @author Francesco Napolitano \email{franapoli@@gmail.com}
#' @aliases repo
NULL

#' Open an existing repository or create a new one.
#'
#' If a repository does not exist at the specified location, creates a
#' directory and stores the repository index in it. If a repository
#' exists, the index is loaded and a \code{repo} object is built.
#' 
#' @param root Path to store data in. Defaults to "~/.R_repo".
#' @param force Don't ask for confirmation.
#' @return An object of class \code{repo}.
#' @name repo_open
#' @examples
#' ## Creates a new repo in a temporary dir without asking for
#' ## confirmation.
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(0, "zero", "a random item", "a_tag")
#' repo$info()
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
NULL


#' Build and/or plots the repo's dependency graph
#'
#' Creates an weighted adjacency matrix, such that (i,j) = x means
#' that item i is in relation x with item j and possibly plots it.
#'
#' @details The relation between any two items i,j can be 1, 2 or
#' 3, respectively meaning:
#'
#' - depends on: to build item i, item j was necessary
#' 
#' - is attached to: item i is an attachment item and is attached to
#' item j.
#' 
#' - has been generated by: item i has been generated by a resource
#' described by the string of characters j, which is typically the
#' file name of a script.
#' 
#' @param repo An object of class repo.
#' @param depends If TRUE, show "depends" edges.
#' @param attached If TRUE, show "attached" edges.
#' @param generated If TRUE, show "generated" edges.
#' @param plot If TRUE (default), plot the dependency graph.
#' @return Adjacency matrix representing the graph, with edges labeled
#' 1, 2, 3 corresponding to "depends", "attached" and "generated" edge
#' types respectively.
#' @examples
#' ## Repository creation (or opening, if exists)
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#'
#' ## Producing some irrelevant data
#' data1 <- 1:10
#' data2 <- data1 * 2
#' data3 <- data1 / 2
#'
#' ## Putting the data in the database, specifying dependencies
#' repo$put(data1, "item1", "First item", "repo_dependencies", replace=TRUE)
#' repo$put(data2, "item2", "Item dependent on item1",
#'     "repo_dependencies", depends="item1", replace=TRUE)
#' repo$put(data3, "item3", "Item dependent on item1 and item2",
#'     "repo_dependencies", depends=c("item1", "item2"), replace=TRUE)
#'
#' ## Obtaining the dependency matrix
#' depmat <- repo$dependencies(plot=FALSE)
#' ## Simplifying labels
#' rownames(depmat) <- colnames(depmat) <- basename(rownames(depmat))
#' print(depmat)
#' ## The matrix can also be plotted as a graph (requires igraph package)
#' ## repo$dependencies(generated=F) # turning off "generated" edges
#' 
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)

repo_dependencies <- function(repo, depends=T, attached=T, generated=T, plot=T)
    repo$dependencies(depends, attached, generated, plot)

#' Check repo's integrity.
#'
#' Checks that all indexed files are present in the repository root
#' and that are not corrupt.
#'
#' @details Every time the object associated to an item is stored, an
#' MD5 checksum is saved to the repository index. \code{check} will
#' use those to verify that the object did not change since the last
#' \code{put} or \code{set} on the item.
#' @param repo An object of class repo.
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(0, "item1", "A sample item", "repo_check", replace=TRUE)
#' repo$check()
#' 
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)

repo_check <- function(repo) repo$check()

#' Copy items to another repo
#'
#' Copies an obj file from one repo to another and creates a new entry
#' in the index of the destination repository. Supports tags.
#' 
#' @param repo An object of class repo (will copy from it)
#' @param destrepo An object of class repo (will copy to it)
#' @param name The name of the object to copy
#' @param tags If not NULL, copy all items matching tags.
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo1")
#' repo1 <- repo_open(repo_path, TRUE)
#' repo1$put(0, "item1", "A sample item", "tag1", replace=TRUE)
#' repo_path2 <- file.path(tempdir(), "example_repo2")
#' repo2 <- repo_open(repo_path2, TRUE)
#' repo1$copy(repo2, "item1")
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)

repo_copy <- function(repo, destrepo, name, tags=NULL)
    repo$copy(destrepo, name, tags)


#' Provides simplified access to repo items.
#'
#' Creates a list of functions, each one associated with a repository
#' item, that can be used to access items more directly.
#'
#' @details Repository handlers are functions associated with
#' items. As opposed to item names, they can take advantage of IDE
#' autocompletion features and do not require quotation marks. An
#' handler to the \code{repo} object itself is provided in the list.
#' @param repo An object of class repo.
#' @return A list of functions.
#' @examples
#'
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' 
#' ## Putting some irrelevant data
#' repo$put(1, "item1", "Sample item 1", "repo_handlers", replace=TRUE)
#' repo$put(2, "item2", "Sample item 2", "repo_handlers", replace=TRUE)
#'
#' ## Getting item handlers
#' h <- repo$handlers()
#' ## handlers have the same names as the items in the repo (and they include
#' ## an handler to the repo itself).
#' names(h)
#'
#' ## Without arguments, function "item1" loads item named "item1".
#' i1 <- h$item1()
#'
#' ## Arguments can be used to call other repo functions on the item.
#' h$item1("info")
#'
#' ## After putting new data, the handlers must be refreshed.
#' repo$put(3, "item3", "Sample item 3", "repo_handlers", replace=TRUE)
#' h <- repo$handlers()
#' names(h)
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_handlers <- function(repo)repo$handlers()

#' List all tags
#'
#' Shows list of all unique tags associated with any item in the
#' repository.
#' 
#' @param repo An object of class repo.
#' @return Character vector of unique tags defined in the repo.
#' @seealso repo_put
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#'
#' ## Putting two items with a few tags
#' repo$put(1, "item1", "Sample item 1",
#'     c("repo_tags", "tag1"), replace=TRUE)
#' repo$put(2, "item2", "Sample item 2",
#'     c("repo_tags", "tag2"), replace=TRUE)
#'
#' ## Looking up tags
#' repo$tags()
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_tags <- function(repo) repo$tags()

#' Run system call on item
#'
#' Runs a system command passing as parameter the file name containing
#' the object associated with an item.
#' 
#' @param repo An object of class repo.
#' @param name Name of a repo item. The path to the file that contains
#' the item will be passed to the system program.
#' @param command Shell command
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' \dontrun{
#' ## Creating a PDF file with a figure.
#' pdffile <- file.path(tempdir(), "afigure.pdf")
#' pdf(pdffile)
#' ## Drawing a random plot in the figure
#' plot(runif(100), runif(100))
#' dev.off()
#' ## Attaching the PDF file to the repo
#' repo$attach(pdffile, "A plot of random numbers", "repo_sys", replace=TRUE)
#' ## don't need the original PDF file anymore
#' file.remove(pdffile)
#' ## Opening the stored PDF with Evince document viewer
#' repo$sys("afigure.pdf", "evince")
#'}
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_sys <- function(repo, name, command)
    repo$sys(name, command)

#' Show a summary of the repository contents.
#'
#' @param x An object of class repo.
#' @param tags A list of character tags. Only items matching all the
#' tags will be shown.
#' @param all Show also items tagged with "hide".
#' @param show Select columns to show.
#' @param ... Further arguments passed to or from other methods
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(1, "item1", "Sample item 1", c("tag1", "tag2"), replace=TRUE)
#' repo$put(2, "item2", "Sample item 2", c("tag1", "hide"), replace=TRUE)
#' repo$put(3, "item3", "Sample item 3", c("tag2", "tag3"), replace=TRUE)
#' repo$print()
#' repo$print(all=TRUE)
#' repo$print(show="tds", all=TRUE)
#' repo$print(show="tds", all=TRUE, tags="tag1")
#' ## wiping the temp repo
#' unlink(repo_path, TRUE)
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
print.repo <- function(x, tags=NULL, all=F, show="ds", ...)
    x$print(tags, all, show=show)

#' Export \code{repo} items to RDS file.
#' 
#' @param repo An object of class repo.
#' @param name Name of the item to export.
#' @param where Destination directory
#' @param tags List of tags: all items tagged with all the tags in the
#' list will be exported.
#' @return TRUE on success, FALSE otherwise.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(1, "item1", "Sample item 1", "export", replace=TRUE)
#' repo$export("item1", tempdir()) # creates item1.RDS in a tempdir
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_export <- function(repo, name, where=".", tags=NULL)
    repo$export(name, where, tags)

#' Provides detailed information about an item.
#' 
#' @param repo An object of class repo.
#' @param name Item name. If both name and tags are NULL, information
#' about the whole repo will be provided.
#' @param tags List of tags: info will run on all items matching the tag list.
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(1, "item1", "Sample item 1", "info", replace=TRUE)
#' repo$info("item1")
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_info <- function(repo, name = NULL, tags = NULL)
    repo$info(name, tags)

#' Remove item from the repo (and the disk).
#' 
#' @param repo An object of class repo.
#' @param name An item's name.
#' @param tags A list of tags: all items matching the list will be
#' removed.
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(1, "item1", "Sample item 1", "info", replace=TRUE)
#' repo$put(2, "item2", "Sample item 2", "info", replace=TRUE)
#' print(repo)
#' repo$rm("item1")
#' print(repo)
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_rm <- function(repo, name = NULL, tags = NULL)
    repo$rm(name, tags)

#' Retrieve an item from the repo.
#' 
#' @param repo An object of class repo.
#' @param name An item's name.
#' @return The previously stored object.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(1, "item1", "Sample item 1", "get", replace=TRUE)
#' print(repo$get("item1"))
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_get <- function(repo, name)repo$get(name)

#' Low-level list of item entries.
#' 
#' @param repo An object of class repo.
#' @return A detailed list of item entries.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(1, "item1", "Sample item 1", "entries", replace=TRUE)
#' repo$put(2, "item2", "Sample item 2", "entries", replace=TRUE)
#' repo$put(3, "item3", "Sample item 3", "entries", replace=TRUE)
#' print(repo$entries())
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_entries <- function(repo)repo$entries()

#' Add tags to an item.
#' 
#' @param repo An object of class repo.
#' @param name An item name.
#' @param newtags A list of tags that will be added to the item's tag
#' list.
#' @param tags A list of tags: newtags will be added to all items
#' matching the list.
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(1, "item1", "Sample item 1", "tag1", replace=TRUE)
#' repo$print(show="t")
#' repo$tag("item1", "tag2")
#' repo$print(show="t")
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_tag <- function(repo, name = NULL, newtags, tags = NULL)
    repo$tag(name, newtags, tags)

#' Remove tags from an item.
#' 
#' @param repo An object of class repo.
#' @param name An item name.
#' @param rmtags A list of tags that will be removed from the item's
#' tag list.
#' @param tags A list of tags: rmtags will be removed from all items
#' matching the list.
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(1, "item1", "Sample item 1", c("tag1", "tag2"), replace=TRUE)
#' repo$print(show="t")
#' repo$untag("item1", "tag2")
#' repo$print(show="t")
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_untag <- function(repo, name = NULL, rmtags, tags = NULL)
    repo$untag(name, rmtags, tags)

#' Edit an existing item.
#' 
#' @param repo An object of class repo.
#' @param name An item name.
#' @param obj An R object to replace the one currently associated with the item.
#' @param newname Newname of the item.
#' @param description Item's description.
#' @param tags New item's tags as a ist of character.
#' @param src New item's provenance as a list of character.
#' @param addtags Tags to be added to current item's tags. Can not be
#' used together with the parameter "tags".
#' @return Used for side effects.
#' @seealso repo_put
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' repo$put(1, "item1", "Sample item 1", c("tag1", "tag2"), replace=TRUE)
#' repo$set("item1", obj=2)
#' print(repo$get("item1"))
#' repo$set("item1", description="Modified description", tags="new_tag_set")
#' repo$info("item1")
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_set <- function(repo, name, obj=NULL, newname=NULL, description=NULL, tags=NULL, src=NULL, addtags=NULL)
    repo$set(name, obj, newname, description, tags, src, addtags)

#' Create a new item from an existing file.
#' 
#' @param repo An object of class repo.
#' @param filepath The path to the file to be stored in the repo.
#' @param description A character description of the item.
#' @param tags A list of tags to sort the item. Tags are useful for
#' selecting sets of items and run bulk actions.
#' @param src The item's provenance as a list of character. Usually
#' the name of the script producing the stored object, a website where
#' the object was downloaded, and so on. If one of the provenance
#' strings matches the name of a repo's item, this will create a
#' dependency link.
#' @param replace If the item exists, overwrite the specified fields.
#' @param to An existing item name to attach the file to.
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' 
#' \dontrun{
#' ## Creating a PDF file with a figure.
#' pdf("afigure.pdf")
#' ## Drawing a random plot in the figure
#' plot(runif(100), runif(100))
#' dev.off()
#' ## Attaching the PDF file to the repo
#' repo$attach("afigure.pdf", "A plot of random numbers", "repo_sys", replace=TRUE)
#' ## don't need the PDF file anymore
#' file.remove("afigure.pdf")
#' ## Opening the stored PDF with Evince document viewer
#' repo$sys("afigure.pdf", "evince")
#' }
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_attach <- function(repo, filepath, description, tags, src=NULL, replace=F, to=NULL)
    repo$attach(filepath, description, tags, src, replace, to)

#' Quickly store temporary data
#'
#' A very simplified call to put that only requires to specify
#' a variable name.
#'
#' @details
#' The name parameter is used to search the parent
#' (or a different specified) environment for the actual object to
#' store. Then it is also used as the item name. The reserved tags
#' "stash" and "hide" are set. In case a stashed item by the same name
#' already exists, it is automatically overwritten. In case a
#' non-stashed item by the same name already exists, an error is
#' raised. A different name can be specified through the rename
#' parameter in such cases.
#' @param repo An object of class repo.
#' @param name A character containing the name of the variable to store.
#' @param rename An optional character containing the new name for the
#' variable. Useful if a non-stash variable named "name" already
#' exists.
#' @param env Environment containing the variable by the specified
#' name. Resolves to parent frame by default.
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' tempdata <- runif(10)
#' repo$stash("tempdata")
#' repo$info("tempdata")
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_stash <- function(repo, name, rename, env=parent.frame())
    repo$stash(name, rename, env)

#' Remove all stashed data
#'
#' @param repo An object of class repo.
#' @param force If TRUE, no confirmation is asked.
#' @return Used for side effects.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' tempdata <- runif(10)
#' repo$stash("tempdata")
#' repo$print(all=TRUE)
#' repo$stashclear(TRUE)
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_stashclear <- function(repo, force=F)
    repo$stashclear(force)


#' Create a new item in the repo.
#'
#' Given an R object, stores it to an RDS file in the \code{repo} root
#' and add an associated item to the \code{repo} index, including
#' object name, description, tags and more.
#' 
#' @details The item \code{name} can be any string, however it should
#' be a concise identifier, possibly without special character (could
#' become mandatory soon). Some tags have a special meaning, like
#' "hide" (do not show the item by default), "attachment" (the item is
#' an attachment - this should never be set manually), "stash" (the
#' item is a stashed item, makes the item over-writable by other
#' "stash" items by default).
#' @param repo An object of class repo.
#' @param obj An R object to store in the repo.
#' @param name A character identifier for the new item.
#' @param description A character description of the item.
#' @param tags A list of tags to sort the item. Tags are useful for
#' selecting sets of items and run bulk actions.
#' @param src The item's provenance as a list of character. Usually
#' the name of the script producing the stored object, a website where
#' the object was downloaded, and so on. If one of the provenance
#' strings matches the name of a repo's item, this will create a
#' dependency link.
#' @param depends The name of an item on which this item depends.
#' @param replace If the item exists, overwrite the specified fields.
#' @param asattach Specifies that the item is to be trated as an
#' attachment (see attach).
#' @param to Optionally specifies which item this item is attached to.
#' @param addversion If TRUE and an item named name exists, create a
#' new version of the same item.
#' @return Used for side effects.
#' @examples
#' ## Repository creation (or opening, if exists)
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#'
#' ## As provenance string we will use the fully  name
#' ## of the file containing this script. Use "normalizePath"
#' ## for fully qualified name (causes problems if used in
#' ## the example).
#'
#' src <- "repoS3.R"
#' 
#' ## Producing some irrelevant data
#' data1 <- 1:10
#' data2 <- data1 * 2
#' data3 <- data1 / 2
#'
#' ## Putting the data in the database, specifying dependencies
#' repo$put(
#'     obj = data1,
#'     name = "item1",
#'     description = "First item",
#'     tags = c("repo_put", "a_random_tag"),
#'     src = src,
#'     replace=TRUE
#'     )
#' repo$put(data2, "item2", "Item dependent on item1",
#'     "repo_dependencies", src, "item1", replace=TRUE)
#' repo$put(data3, "item3", "Item dependent on item1 and item2",
#'     "repo_dependencies", src, c("item1", "item2"), replace=TRUE)
#'
#' print(repo)
#'
#' ## Creating another version of item1
#' data1.2 <- data1 + runif(10)
#' repo$put(data1.2, name = "item1", "First item with additional noise",
#'     tags = c("repo_put", "a_random_tag"), src, addversion=TRUE)
#' print(repo, all=TRUE)
#' repo$info("item1#1")
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_put <- function(repo, obj, name, description, tags, src=NULL,
                     depends = NULL, replace=F, asattach=F, to=NULL, addversion=F)
    repo$put(obj, name, description, tags, src,
             depends, replace, asattach, to, addversion)

#' Append text to an existing item content.
#'
#' This feature is experimental.
#' 
#' @param repo An object of class repo.
#' @param id The name of an item whose object is of class character.
#' @param txtorfunc Text to be appended to the item's object. It can
#' also be a on object of class function: in this case, its source is
#' appended.
#' @return Used for side effects.
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_append <- function(repo, id, txtorfunc)
    repo$append(id, txtorfunc)


#' Show path to repo root
#' 
#' @param repo An object of class repo.
#' @return character cotaining the path to the root of the repo.
#' @examples
#' repo_path <- file.path(tempdir(), "example_repo")
#' repo <- repo_open(repo_path, TRUE)
#' print(repo$root())
#'
#' ## wiping temporary repo
#' unlink(repo_path, TRUE)
repo_root <- function(repo)
    repo$root()

