% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reprex.R
\name{reprex}
\alias{reprex}
\title{Render a reprex}
\usage{
reprex(
  x = NULL,
  input = NULL,
  wd = NULL,
  venue = c("gh", "r", "rtf", "html", "slack", "so", "ds"),
  render = TRUE,
  advertise = NULL,
  session_info = opt(FALSE),
  style = opt(FALSE),
  comment = opt("#>"),
  tidyverse_quiet = opt(TRUE),
  std_out_err = opt(FALSE),
  html_preview = opt(TRUE),
  outfile = deprecated(),
  show = deprecated(),
  si = deprecated()
)
}
\arguments{
\item{x}{An expression. If not given, \code{reprex()} looks for code in
\code{input}. If \code{input} is not provided, \code{reprex()} looks on the clipboard.

When the clipboard is structurally unavailable, e.g., on RStudio Server or
RStudio Cloud, \code{reprex()} consults the current selection instead of the
clipboard.}

\item{input}{Character. If has length one and lacks a terminating newline,
interpreted as the path to a file containing reprex code. Otherwise,
assumed to hold reprex code as character vector. When \code{input} specifies a
filepath, it also determines the reprex working directory and the location
of all resulting files.}

\item{wd}{An optional filepath that is consulted when \code{input} is not a
filepath. (By default, all work is done, quietly, in a subdirectory of the
session temp directory.)

The most common use of \code{wd} is to set \code{wd = "."}, which means "reprex right
HERE in the current working directory". Do this if you really must
demonstrate something with local files.}

\item{venue}{Character. Must be one of the following (case insensitive):
\itemize{
\item "gh" for \href{https://github.github.com/gfm/}{GitHub-Flavored Markdown}, the
default
\item "r" for a runnable R script, with commented output interleaved. Also useful
for \href{https://slack.com/intl/en-ca/slack-tips/share-code-snippets}{Slack code snippets};
select "R" from the "Type" drop-down menu to enjoy nice syntax
highlighting.
\item "rtf" for
\href{https://en.wikipedia.org/wiki/Rich_Text_Format}{Rich Text Format}
(not supported for un-reprexing)
\item "html" for an HTML fragment suitable for inclusion in a larger HTML
document (not supported for un-reprexing)
\item "slack" for pasting into a Slack message. Optimized for people who opt out
of Slack's WYSIWYG interface. Go to
\strong{Preferences > Advanced > Input options} and select "Format messages with
markup". (If there is demand for a second Slack venue optimized for use
with WYSIWYG, please open an issue to discuss.)
\item "so" for
\href{https://stackoverflow.com/editing-help#syntax-highlighting}{Stack Overflow Markdown}.
Note: this is just an alias for "gh", since Stack Overflow started to
support CommonMark-style fenced code blocks in January 2019.
\item "ds" for Discourse, e.g.,
\href{https://community.rstudio.com}{community.rstudio.com}. Note: this is
currently just an alias for "gh".
}}

\item{render}{Logical. Whether to call \code{\link[rmarkdown:render]{rmarkdown::render()}} on the templated
reprex, i.e. whether to actually run the code. Defaults to \code{TRUE}. Exists
primarily for the sake of internal testing.}

\item{advertise}{Logical. Whether to include a footer that describes when and
how the reprex was created. If unspecified, the option \code{reprex.advertise}
is consulted and, if that is not defined, default is \code{TRUE} for venues
\code{"gh"}, \code{"html"}, \code{"so"}, \code{"ds"} and \code{FALSE} for \code{"r"}, \code{"rtf"}, \code{"slack"}.}

\item{session_info}{Logical. Whether to include
\code{\link[sessioninfo:session_info]{sessioninfo::session_info()}}, if available, or \code{\link[=sessionInfo]{sessionInfo()}} at the end
of the reprex. When \code{venue} is "gh", the session info is wrapped in a
collapsible details tag. Read more about \code{\link[=opt]{opt()}}.}

\item{style}{Logical. Whether to set the knitr chunk option \code{tidy = "styler"}, which re-styles code with the \href{https://styler.r-lib.org}{styler package}. Read more about \code{\link[=opt]{opt()}}.}

\item{comment}{Character. Prefix with which to comment out output, defaults
to \code{"#>"}. Read more about \code{\link[=opt]{opt()}}.}

\item{tidyverse_quiet}{Logical. Sets the options \code{tidyverse.quiet} and
\code{tidymodels.quiet}, which suppress (\code{TRUE}, the default) or include
(\code{FALSE}) the startup messages for the tidyverse and tidymodels packages.
Read more about \code{\link[=opt]{opt()}}.}

\item{std_out_err}{Logical. Whether to append a section for output sent to
stdout and stderr by the reprex rendering process. This can be necessary to
reveal output if the reprex spawns child processes or \code{system()} calls.
Note this cannot be properly interleaved with output from the main R
process, nor is there any guarantee that the lines from standard output and
standard error are in correct chronological order. See \code{\link[callr:r]{callr::r()}} for
more. Read more about \code{\link[=opt]{opt()}}.}

\item{html_preview}{Logical. Whether to show rendered output in a viewer
(RStudio or browser). Always \code{FALSE} in a noninteractive session. Read more
about \code{\link[=opt]{opt()}}.}

\item{outfile}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} in favor of \code{wd} or
providing a filepath to \code{input}. To reprex in current working directory,
use \code{wd = "."} now, instead of \code{outfile = NA}.}

\item{show}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} in favor of \code{html_preview},
for greater consistency with other R Markdown output formats.}

\item{si}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} in favor of \code{session_info}.}
}
\value{
Character vector of rendered reprex, invisibly.
}
\description{
Run a bit of R code using \code{\link[rmarkdown:render]{rmarkdown::render()}} and write the rendered result
to user's clipboard. If the clipboard is unavailable, the file containing
the rendered result is opened for manual copy. The goal is to make it easy to
share a small reproducible example ("reprex"), e.g., in a GitHub issue.
Reprex source can be
\itemize{
\item read from clipboard
\item provided directly as expression, character vector, or string
\item read from file
\item read from current selection or active document in RStudio
}

reprex can also be used for syntax highlighting (with or without rendering);
see below for more.
}
\section{Details}{

The usual "code + commented output" is returned invisibly, written to file,
and, whenever possible, put on the clipboard. An HTML preview displays in
RStudio's Viewer pane, if available, or in the default browser, otherwise.
Leading \code{"> "} prompts, are stripped from the input code. Read more at
\url{https://reprex.tidyverse.org/}.

reprex sets specific \href{https://yihui.org/knitr/options/}{knitr options}:
\itemize{
\item Chunk options default to \code{collapse = TRUE}, \code{comment = "#>"},
\code{error = TRUE}. Note that \code{error = TRUE}, because a common use case is bug
reporting.
\item reprex also sets knitr's \code{upload.fun}. It defaults to
\code{\link[knitr:imgur_upload]{knitr::imgur_upload()}} so figures produced by the reprex appear properly
on GitHub, Stack Overflow, Discourse, and Slack. Note that \code{imgur_upload()}
requires the packages httr and xml2. When \code{venue = "r"}, \code{upload.fun} is
set to \code{identity()}, so that figures remain local. In that case, you may
also want to provide a filepath to \code{input} or set \code{wd}, to control where
the reprex files are written.
You can supplement or override these options with special comments in your
code (see examples).
}
}

\section{Error backtraces}{


reprex sets the rlang option \code{rlang_backtrace_on_error_report = "full"}.
Combined with the knitr option \code{error = TRUE}, this means rlang errors are
displayed with a full backtrace. This basically eliminates the need to call
\code{\link[rlang:last_error]{rlang::last_error()}} or \code{\link[rlang:last_error]{rlang::last_trace()}} explicitly, although these
functions can be used in a reprex.

Insert a line containing the special comment \verb{#'} in between the
error-causing code and the \code{last_error()} or \code{last_trace()} call, to fulfill
the requirement of being in separate chunks:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f <- function() rlang::abort('foo')
f()
#'
rlang::last_error()
rlang::last_trace()
}\if{html}{\out{</div>}}

Read more in rlang's documentation: \href{https://rlang.r-lib.org/reference/rlang_backtrace_on_error.html#errors-in-rmarkdown}{Errors in RMarkdown}.
}

\section{Syntax highlighting}{


\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

A secondary use case for reprex is to produce syntax highlighted code
snippets, with or without rendering, to paste into applications like
Microsoft Word, PowerPoint, or Keynote. Use \code{venue = "rtf"} for this.

This feature is experimental and requires the installation of the
\href{http://www.andre-simon.de/doku/highlight/en/highlight.php}{highlight}
command line tool. The \code{"rtf"} venue is documented in \href{https://reprex.tidyverse.org/articles/articles/rtf.html}{its own article}
}

\examples{
\dontrun{
# put some code like this on the clipboard
# (y <- 1:4)
# mean(y)
reprex()

# provide code as an expression
reprex(rbinom(3, size = 10, prob = 0.5))
reprex({y <- 1:4; mean(y)})
reprex({y <- 1:4; mean(y)}, style = TRUE)

# note that you can include newlines in those brackets
# in fact, that is often a good idea
reprex({
  x <- 1:4
  y <- 2:5
  x + y
})

## provide code via character vector
reprex(input = c("x <- 1:4", "y <- 2:5", "x + y"))

## if just one line, terminate with '\n'
reprex(input = "rnorm(3)\n")

## customize the output comment prefix
reprex(rbinom(3, size = 10, prob = 0.5), comment = "#;-)")

# override a default chunk option
reprex({
  #+ setup, include = FALSE
  knitr::opts_chunk$set(collapse = FALSE)

  #+ actual-reprex-code
  (y <- 1:4)
  median(y)
})

# add prose, use general markdown formatting
reprex({
  #' # A Big Heading
  #'
  #' Look at my cute example. I love the
  #' [reprex](https://github.com/tidyverse/reprex#readme) package!
  y <- 1:4
  mean(y)
}, advertise = FALSE)

# read reprex from file and write resulting files to that location
tmp <- file.path(tempdir(), "foofy.R")
writeLines(c("x <- 1:4", "mean(x)"), tmp)
reprex(input = tmp)
list.files(dirname(tmp), pattern = "foofy")

# clean up
file.remove(list.files(dirname(tmp), pattern = "foofy", full.names = TRUE))

# write reprex to file AND keep figure local too, i.e. don't post to imgur
tmp <- file.path(tempdir(), "foofy")
dir.create(tmp)
reprex({
  #+ setup, include = FALSE
  knitr::opts_knit$set(upload.fun = identity)

  #+ actual-reprex-code
  #' Some prose
  ## regular comment
  (x <- 1:4)
  median(x)
  plot(x)
  }, wd = tmp)
list.files(dirname(tmp), pattern = "foofy")

# clean up
unlink(tmp, recursive = TRUE)

## target venue = R, also good for email or Slack snippets
ret <- reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, venue = "R")
ret

## target venue = html
ret <- reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, venue = "html")
ret

## include prompt and don't comment the output
## use this when you want to make your code hard to execute :)
reprex({
  #+ setup, include = FALSE
  knitr::opts_chunk$set(comment = NA, prompt = TRUE)

  #+ actual-reprex-code
  x <- 1:4
  y <- 2:5
  x + y
})

## leading prompts are stripped from source
reprex(input = c("> x <- 1:3", "> median(x)"))
}
}
