% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ee_download.R
\name{ee_drive_to_local}
\alias{ee_drive_to_local}
\title{Move results from Google Drive to a local directory}
\usage{
ee_drive_to_local(
  task,
  dsn,
  overwrite = TRUE,
  consider = TRUE,
  public = FALSE,
  metadata = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{task}{A generated list obtained after completing an Earth Engine task. See details.}

\item{dsn}{Character. Output filename. If missing, a temporary
file will be assigned.}

\item{overwrite}{A boolean argument that indicates
whether filename should be overwritten. By default TRUE.}

\item{consider}{Interactive. See details.}

\item{public}{Logical. If TRUE, a public link to the Google Drive resource is
created.}

\item{metadata}{Logical. If TRUE, the metadata related to the Google
Drive resource will be exported. See details.}

\item{quiet}{Logical. Suppress info message.}
}
\value{
If \code{metadata} is FALSE, will return the filename of the Google
Drive resource on their system. Otherwise, a list with two elements
(\code{dns} and \code{metadata}) is returned.
}
\description{
Move results of an EE task saved in Google Drive to a local directory.
}
\details{
The task argument requires a status of "COMPLETED" because
the parameters required to identify EE items in Google Drive
are retrieved from  \cr \code{ee$batch$Export$*$toDrive(...)$start()$status()}.

Due to the fact that Google Drive allows users to create files with the
same name, the \code{consider} argument is required. It use an interactive
R session by default to assist users in identifying the specific files
they wish to download. Additionally, "last" and "all" settings are
provided. "last" will only download the most recently saved file in
Google Drive, whereas "all" will download all files with the same name.

Finally, if the argument \code{metadata} is TRUE, a list containing the
following elements is exported and appended to the output filename (dsn):

\itemize{
\item{\bold{ee_id: }}{Name of the Earth Engine task.}
\item{\bold{drive_name: }}{Name of the Table in Google Drive.}
\item{\bold{drive_id: }}{Id of the Table in Google Drive.}
\item{\bold{drive_download_link: }}{Download link to the table.}
}
}
\examples{
\dontrun{
library(rgee)
library(stars)
library(sf)

ee_users()
ee_Initialize(drive = TRUE)

# Define study area (local -> earth engine)
# Communal Reserve Amarakaeri - Peru
rlist <- list(xmin = -71.13, xmax = -70.95,ymin = -12.89, ymax = -12.73)
ROI <- c(rlist$xmin, rlist$ymin,
         rlist$xmax, rlist$ymin,
         rlist$xmax, rlist$ymax,
         rlist$xmin, rlist$ymax,
         rlist$xmin, rlist$ymin)

ee_ROI <- matrix(ROI, ncol = 2, byrow = TRUE) \%>\%
  list() \%>\%
  st_polygon() \%>\%
  st_sfc() \%>\%
  st_set_crs(4326) \%>\%
  sf_as_ee()


# Get the mean annual NDVI for 2011
cloudMaskL457 <- function(image) {
  qa <- image$select("pixel_qa")
  cloud <- qa$bitwiseAnd(32L)$
    And(qa$bitwiseAnd(128L))$
    Or(qa$bitwiseAnd(8L))
  mask2 <- image$mask()$reduce(ee$Reducer$min())
  image <- image$updateMask(cloud$Not())$updateMask(mask2)
  image$normalizedDifference(list("B4", "B3"))
}

ic_l5 <- ee$ImageCollection("LANDSAT/LT05/C01/T1_SR")$
  filterBounds(ee$FeatureCollection(ee_ROI))$
  filterDate("2011-01-01", "2011-12-31")$
  map(cloudMaskL457)

# Create simple composite
mean_l5 <- ic_l5$mean()$rename("NDVI")
mean_l5 <- mean_l5$reproject(crs = "EPSG:4326", scale = 500)
mean_l5_Amarakaeri <- mean_l5$clip(ee_ROI)

# Move results from Earth Engine to Drive
task_img <- ee_image_to_drive(
  image = mean_l5_Amarakaeri,
  folder = "Amarakaeri",
  fileFormat = "GEO_TIFF",
  region = ee_ROI,
  fileNamePrefix = "my_image_demo"
)

task_img$start()
ee_monitoring(task_img)

# Move results from Drive to local
img <- ee_drive_to_local(task = task_img)
}
}
\seealso{
Other generic download functions: 
\code{\link{ee_gcs_to_local}()}
}
\concept{generic download functions}
