% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clustering.R
\name{calcNMI}
\alias{calcNMI}
\title{Calculate Normalized Mutual Information (NMI) by comparing two cluster
labeling variables}
\usage{
calcNMI(
  object,
  trueCluster,
  useCluster = NULL,
  useDatasets = NULL,
  verbose = getOption("ligerVerbose", TRUE)
)
}
\arguments{
\item{object}{A \linkS4class{liger} object, with the clustering result
present in cellMeta.}

\item{trueCluster}{Either the name of one variable in \code{cellMeta(object)}
or a factor object with annotation that matches with all cells being
considered.}

\item{useCluster}{The name of one variable in \code{cellMeta(object)}.
Default \code{NULL} uses default clusters.}

\item{useDatasets}{A character vector of the names, a numeric or logical
vector of the index of the datasets to be considered for the purity
calculation. Default \code{NULL} uses all datasets.}

\item{verbose}{Logical. Whether to show information of the progress. Default
\code{getOption("ligerVerbose")} or \code{TRUE} if users have not set.}
}
\value{
A numeric scalar of the NMI value
}
\description{
This function aims at calculating the Normalized Mutual Information for the
clustering result obtained with LIGER and the external clustering (existing
"true" annotation). NMI ranges from 0 to 1, with a score of 0 indicating no
agreement between clusterings and 1 indicating perfect agreement. The
mathematical definition of NMI is as follows:

\deqn{
H(X) = -\sum_{x \in X}P(X=x)\log_2 P(X=x)
}
\deqn{
H(X|Y) = -\sum_{y \in Y}P(Y=y)\sum_{x \in X}P(X=x|Y=y)\log_2 P(X=x|Y=y)
}
\deqn{
I(X;Y) = H(X) - H(X|Y)
}
\deqn{
NMI(X;Y) = \frac{I(X;Y)}{\sqrt{H(X)H(Y)}}
}

Where \eqn{X} is the cluster variable to be evaluated and \eqn{Y} is the true
cluster variable. \eqn{x} and \eqn{y} are the cluster labels in \eqn{X} and
\eqn{Y} respectively. \eqn{H} is the entropy and \eqn{I} is the mutual
information.

The true clustering annotation must be specified as the base line. We suggest
setting it to the object cellMeta so that it can be easily used for many
other visualization and evaluation functions.

The NMI can be calculated for only specified datasets, since true annotation
might not be available for all datasets. Evaluation for only one or a few
datasets can be done by specifying \code{useDatasets}. If \code{useDatasets}
is specified, the argument checking for \code{trueCluster} and
\code{useCluster} will be enforced to match the cells in the specified
datasets.
}
\examples{
# Assume the true cluster in `pbmcPlot` is "leiden_cluster"
# generate fake new labeling
fake <- sample(1:7, ncol(pbmcPlot), replace = TRUE)
# Insert into cellMeta
pbmcPlot$new <- factor(fake)
calcNMI(pbmcPlot, trueCluster = "leiden_cluster", useCluster = "new")

# Now assume we got existing base line annotation only for "stim" dataset
nStim <- ncol(dataset(pbmcPlot, "stim"))
stimTrueLabel <- factor(fake[1:nStim])
# Insert into cellMeta
cellMeta(pbmcPlot, "stim_true_label", useDatasets = "stim") <- stimTrueLabel
# Assume "leiden_cluster" is the clustering result we got and need to be
# evaluated
calcNMI(pbmcPlot, trueCluster = "stim_true_label",
        useCluster = "leiden_cluster", useDatasets = "stim")

# Comparison of the same labeling should always yield 1.
calcNMI(pbmcPlot, trueCluster = "leiden_cluster", useCluster = "leiden_cluster")
}
