context("Test coordinates")

test_that("Test grid references with different length", {

    expect_that(osg_parse("SN9491282412"),
                equals(structure(list(easting = 294912, northing = 282412),
                                 .Names = c("easting", "northing"))))

    expect_that(osg_parse("SN94918241"),
                equals(structure(list(easting = 294910, northing = 282410),
                                 .Names = c("easting", "northing"))))

    expect_that(osg_parse("SN949824"),
                equals(structure(list(easting = 294900, northing = 282400),
                                 .Names = c("easting", "northing"))))

    expect_that(osg_parse("SN9482"),
                equals(structure(list(easting = 294000, northing = 282000),
                                 .Names = c("easting", "northing"))))

    closeAllConnections()

})

test_that("Test grid references from various areas", {

    expect_that(osg_parse("TQ722213"),
                equals(structure(list(easting = 572200, northing = 121300),
                                 .Names = c("easting", "northing"))))

    expect_that(osg_parse(c("SN831869", "SN829838")),
                equals(structure(list(easting = c(283100, 282900),
                                      northing = c(286900, 283800)),
                                 .Names = c("easting", "northing"))))

    closeAllConnections()

})



# crs 4326
wgs <- matrix(c(18.5, 54.2,
                113.4, 46.78,
                16.9, 67.8),
              byrow = TRUE, ncol = 2, dimnames = list(NULL, c("x", "y")))

# correct values from https://mygeodata.cloud/cs2cs/
# crs 3034
lcceu <- matrix(c(4535410.16093, 3067447.25856,
                  9382256.98256, 6769270.46095,
                  4295683.24203, 4544089.18176),
                byrow = TRUE, ncol = 2, dimnames = list(NULL, c("x", "y")))

# crs 3416
lccat <- matrix(c(739233.098505, 1158123.41459,
                  6096512.30717, 4595909.41132,
                  562300.406341, 2719066.06187),
                byrow = TRUE, ncol = 2, dimnames = list(NULL, c("x", "y")))


test_that("conversion works for single crs", {
    tr <- .transform_crs(x = wgs[, "x"], y = wgs[, "y"], from = 4326, to = 3034)
    rownames(tr) <- NULL
    expect_equal(round(tr, 4), round(as.data.frame(lcceu), 4))
})


test_that("conversion works for multiple crs", {
    input <- data.frame(rbind(wgs, wgs),
                        crs = c(rep(3034, times = nrow(lcceu)),
                                rep(3416, times = nrow(lccat))))
    idx <- sample(nrow(input))
    input <- input[idx, ]

    tr <- .transform_crs(x = input$x, y = input$y, from = 4326, to = input$crs)
    rownames(tr) <- NULL
    expect_equal(round(tr, 4),
                 as.data.frame(round(rbind(lcceu, lccat)[idx, ], 4)))
})

test_that("vectorized version of parse_osg() returns values as before", {
    grid <- c("NJ9468430370", "NJ7567720150", "NO6348295613", "NO7973998317",
              "NO8264873370", "NO14753668", "NN9331316652", "NS7765895458",
              "NT4561868891", "IH4607173048", "IH3477995968", "IC4641115098",
              "IC9307615349", "IC9419136250", "IJ3287269316", "IJ4485350808",
              "NT4982333376", "NT8983647709", "NT8807755014", "NZ0385861700",
              "NY8565664671", "NZ2592313748", "SE4221547311", "SE5684555373",
              "SE7311258712", "SE7046885533", "SK62013996", "SK23742884",
              "SE64580122", "SK65294795", "SP54209732", "TF337879",
              "SK84204796", "TL0535349560", "TF7160200623", "TL8508984362",
              "TL1600453461", "TL9624926113", "TL3907409193", "TL2825613276",
              "TQ17786985", "SU73096478", "SU6495970828", "SU4703368219",
              "SP1214706228", "SU63457658", "SU4526669473", "TQ2624546209",
              "TQ7086553033", "TR1154955356", "TQ4290721374", "SU5874607460",
              "SU31780192", "SU35411886", "SU4612921122", "SU1510941387",
              "SZ1123396046", "SY9132287609", "SS9360201602", "SY0866588435",
              "SX7511265841", "SX1595167373", "SW8197545030", "SS6079523662",
              "ST2057625023", "ST5899431756", "SO78237616", "SP0400743741",
              "SJ41191446", "SO73495573", "SO8634939041", "SO5276911077",
              "SO3460605591", "SO0034830405", "ST0794095652", "SS6855399798",
              "SN4850920576", "NH6450742726", "SN2441941655", "SH7448901958",
              "SH8012958077",
              "SJ3482241459", "SH8735330798", "NJ1940262632", "SD5892230412",
              "SD5293565318", "NY3900057103", "NX9234685121", "NH9469119208",
              "NJ3185051788", "NX4126665330", "NS7043557942", "NS6718461676",
              "NS3940980307", "NS3563592880", "NN1159274257", "NJ5321746431",
              "NJ7055149833", "NG9418642885", "NG8600280330", "NC7136356765")

    lons <- c(-2.09, -2.405, -2.6036, -2.3358, -2.2861,
              -3.387, -3.727, -3.9702, -2.8715, -7.2879, -7.4598, -7.2759,
              -6.5471, -6.5228, -5.947, -5.7711, -2.7977, -2.1634, -2.1917,
              -1.9413, -2.2256, -1.6011, -1.3587, -1.1345, -0.8856, -0.9197,
              -1.0785, -1.6489, -1.0278, -1.028, -1.2017, 0.0081, -0.7462,
              -0.4622, 0.531, 0.7208, -0.3053, 0.851, 0.0139, -0.1413, -0.3077,
              -0.9512, -1.0669, -1.3251, -1.8254, -1.0876, -1.3504, -0.1944,
              0.447, 1.0305, 0.0341, -1.1666, -1.5503, -1.4971, -1.3442,
              -1.7853,
              -1.8421, -2.1242, -3.5113, -3.2943, -3.7615, -4.5953, -5.0606,
              -3.985, -3.1337, -2.5868, -2.3213, -1.9429, -2.8722, -2.3893,
              -2.2005, -2.6863, -2.9486, -3.4518, -3.3321, -3.9027, -4.2015,
              -4.2598, -4.5617, -3.8546, -3.7921, -2.9719, -3.6747, -3.3519,
              -2.6247, -2.7209, -2.9529, -3.6907, -3.747, -3.1404, -4.4805,
              -4.0683, -4.1219, -4.576, -4.6441, -5.0888, -2.7824, -2.4935,
              -5.43, -5.5991, -4.2077)
    lats <- c(57.3638, 57.2713, 57.0501,
              57.0754, 56.8514, 56.5144, 56.3302, 56.1361, 55.91, 54.6036,
              54.8104, 54.9812, 54.977, 55.1644, 54.5547, 54.3853, 55.5914,
              55.7227, 55.7883, 54.9499, 54.9764, 54.5184, 53.9202, 53.9913,
              54.0192, 54.2606, 52.9534, 52.8565, 53.5037, 53.0249, 52.571,
              53.3711, 53.0223, 52.1343, 52.5763, 52.4259, 52.1672, 51.899,
              51.7642, 51.8034, 51.4154, 51.3774, 51.4327, 51.411, 51.7546,
              51.4846, 51.4225, 51.2012, 51.2509, 51.2581, 50.9741, 50.8637,
              50.816, 50.9681, 50.9877, 51.1715, 50.7639, 50.6881, 50.8042,
              50.6884, 50.479, 50.4772, 50.2649, 50.9955, 51.0191, 51.0836,
              52.383, 52.092, 52.7246, 52.1992, 52.0496, 51.7963, 51.7452,
              51.9632, 51.6521, 51.6815, 51.8631, 57.4545, 52.0454, 52.6009,
              53.1064, 52.9665, 52.8628, 57.6464, 53.7684, 54.0816, 54.9049,
              55.1487, 57.2514, 57.551, 54.9573, 55.7973, 55.83, 55.9888,
              56.1003,
              56.8215, 57.5056, 57.5377, 57.4295, 57.7614, 58.4801)
    latslons <- structure(list(lons, lats), .Names = c("lon", "lat"))

    wgs <- osg_parse(grid, coord_system = "WGS84")
    rownames(wgs) <- NULL
    expect_equal(lapply(wgs, round, digits = 4), latslons)
})
