# --------------------------------------
# Author: Andreas Alfons
#         Erasmus Universiteit Rotterdam
# --------------------------------------

#' (Robustly) fit a mediation model
#'
#' (Robustly) estimate the effects in a mediation model.
#'
#' If \code{method} is \code{"covariance"} and \code{robust} is \code{TRUE}
#' (the default), the effects are estimated based on a Huber M-estimator of
#' location and scatter.
#'
#' A much more robust method based on robust regression will be available
#' soon.  Currently, least squares regression is always performed if
#' \code{method} is \code{"regression"}.
#'
#' @aliases print.fitMediation summary.regFitMediation summary.covFitMediation
#'
#' @param x  either a numeric vector containing the independent variable, or
#' (if \code{data} is supplied) a character string, an integer or a logical
#' vector specifying the corresponding column of \code{data}.
#' @param y  either a numeric vector containing the dependent variable, or
#' (if \code{data} is supplied) a character string, an integer or a logical
#' vector specifying the corresponding column of \code{data}.
#' @param m  either a numeric vector containing the proposed mediator variable,
#' or (if \code{data} is supplied) a character string, an integer or a logical
#' vector specifying the corresponding column of \code{data}.
#' @param covariates  optional; either a numeric vector or data frame
#' containing additional covariates to be used as control variables, or (if
#' \code{data} is supplied) a character, integer or logical vector specifying
#' the corresponding column of \code{data}.
#' @param data  an optional \code{data.frame}.
#' @param method  a character string specifying the method of
#' estimation.  Possible values are \code{"covariance"} (the default) to
#' estimate the effects via the covariance matrix, or \code{"regression"}
#' to estimate the effects via regressions.  Note that the effects are
#' always estimated via regressions if control variables are specified via
#' \code{covariates}.
#' @param robust  a logical indicating whether to robustly estimate the effects
#' (defaults to \code{TRUE}).  This is currently ignored if \code{method} is
#' \code{"regression"}.
#' @param control  if \code{robust} is \code{TRUE} and \code{method} is
#' \code{"covariance"}, a list of tuning parameters for \code{\link{covHuber}}
#' as generated by \code{\link{covControl}}.
#' @param \dots  additional arguments can be used to specify tuning parameters
#' directly instead of via \code{control}.
#'
#' @return An object inheriting from class \code{"fitMediation"} (class
#' \code{"regFitMediation"} if \code{method} is \code{"regression"} or
#' \code{"covFitMediation"} if \code{method} is \code{"covariance"}) with
#' the following components:
#' \item{a}{numeric; the point estimate of the effect of the independent
#' variable on the proposed mediator variable.}
#' \item{b}{numeric; the point estimate of the direct effect of the
#' proposed mediator variable on the dependent variable.}
#' \item{c}{numeric; the point estimate of the direct effect of the
#' independent variable on the dependent variable.}
#' \item{cPrime}{numeric; the point estimate of the total effect of the
#' independent variable on the dependent variable.}
#' \item{robust}{a logical indicating whether the effects were estimated
#' robustly.}
#' \item{fitMX}{an object of class \code{"\link[stats]{lm}"} containing the
#' estimation results from the regression of the proposed mediator variable
#' on the independent variable (only \code{"regFitMediation"}).}
#' \item{fitYMX}{an object of class \code{"\link[stats]{lm}"} containing the
#' estimation results from the regression of the dependent variable on the
#' proposed mediator and independent variables (only \code{"regFitMediation"}).}
#' \item{fitYX}{an object of class \code{"\link[stats]{lm}"} containing the
#' estimation results from the regression of the dependent variable on the
#' independent variable (only \code{"regFitMediation"}).}
#' \item{cov}{an object of class \code{"\link{covHuber}"} or
#' \code{"\link{covML}"} containing the covariance matrix estimates (only
#' \code{"covFitMediation"}).}
#' \item{data}{a data frame containing the independent, dependent and
#' proposed mediator variables.}
#'
#' @author Andreas Alfons
#'
#' @references
#' Zu, J. and Yuan, K.-H. (2010) Local influence and robust procedures for
#' mediation analysis. \emph{Multivariate Behavioral Research}, \bold{45}(1),
#' 1--44.
#'
#' @seealso \code{\link{testMediation}}
#'
#' \code{\link{covHuber}}, \code{\link{covML}}, \code{\link[stats]{lm}}
#'
#' @examples
#' # control parameters
#' n <- 250             # number of observations
#' a <- b <- c <- 0.2   # true effects
#' t <- 2               # number of observations to contaminate
#'
#' # draw clean observations
#' set.seed(20160911)
#' x <- rnorm(n)
#' m <- a * x + rnorm(n)
#' y <- b * m + c * x + rnorm(n)
#'
#' # contaminate the first t observations
#' m[1:t] <- m[1:t] - 6
#' y[1:t] <- y[1:t] + 6
#'
#' # fit mediation model
#' fitMediation(x, y, m)
#'
#' @keywords multivariate
#'
#' @import boot
#' @export

fitMediation <- function(x, y, m, covariates = NULL, data,
                         method = c("covariance", "regression"),
                         robust = TRUE, control, ...) {
  ## initializations
  # FIXME: make sure dimensions are correct
  if(missing(data)) {
    # prepare data frame containing all variables with original names
    x <- substitute(x)
    y <- substitute(y)
    m <- substitute(m)
    if(is.null(covariates)) {
      data <- eval.parent(call("data.frame", x, y, m))
    } else if(is.data.frame(covariates)) {
      data <- cbind(eval.parent(call("data.frame", x, y, m)), covariates)
    } else {
      covariates <- substitute(covariates)
      data <- eval.parent(call("data.frame", x, y, m, covariates))
    }
  } else {
    # prepare data set
    data <- as.data.frame(data)
    x <- data[, x, drop=FALSE]
    y <- data[, y, drop=FALSE]
    m <- data[, m, drop=FALSE]
    covariates <- data[, covariates, drop=FALSE]
    data <- cbind(x, y, m, covariates)
  }
  # extract names
  cn <- names(data)
  x <- cn[1]
  y <- cn[2]
  m <- cn[3]
  covariates <- cn[-(1:3)]
  # make sure that variables are numeric
  convert <- !sapply(data, is.numeric)
  data[convert] <- lapply(data[convert], as.numeric)
  # check if there are enough observations
  n <- nrow(data)
  if(n <= 3) stop("not enough observations")
  # check other arguments
  method <- match.arg(method)
  if(length(covariates) > 0 && method == "covariance") {
    method <- "regression"
    warning("using regression method")
  }
  if(method == "regression") robust <- FALSE
  else {
    robust <- isTRUE(robust)
    if(robust && missing(control)) control <- covControl(...)
  }
  ## estimate effects
  if(method == "regression") {
    regFitMediation(x, y, m, covariates, data=data,
                    robust=robust, control=control)
  } else covFitMediation(x, y, m, data=data, robust=robust, control=control)
}

## estimate the effects in a mediation model via regressions
regFitMediation <- function(x, y, m, covariates = character(), data,
                            robust = TRUE, control = NULL) {
  # construct formulas for regression models
  covariateTerm <- paste(c("", covariates), collapse="+")
  fMX <- as.formula(paste(m, "~", x, covariateTerm, sep=""))
  fYMX <- as.formula(paste(y, "~", m, "+", x, covariateTerm, sep=""))
  # compute regression models
  if(robust) stop("not implemented yet")
  else {
    # for the standard method, there is not much additional cost in performing
    # the regression for the total effect
    fitMX <- lm(fMX, data=data, model=FALSE)
    fitYMX <- lm(fYMX, data=data, model=FALSE)
    fYX <- as.formula(paste(y, "~", x, covariateTerm, sep=""))
    fitYX <- lm(fYX, data=data, model=FALSE)
  }
  # extract effects
  a <- unname(coef(fitMX)[2])
  b <- unname(coef(fitYMX)[2])
  c <- unname(coef(fitYMX)[3])
  cPrime <- unname(coef(fitYX)[2])
  # return results
  result <- list(a=a, b=b, c=c, cPrime=cPrime, fitMX=fitMX, fitYMX=fitYMX,
                 fitYX=fitYX, x=x, y=y, m=m, covariates=covariates,
                 data=data, robust=robust)
  class(result) <- c("regFitMediation", "fitMediation")
  result
}

## estimate the effects in a mediation model via the covariance matrix
covFitMediation <- function(x, y, m, data, robust = TRUE,
                            control = covControl()) {
  # compute scatter matrix (Huber M-estimator or MLE of covariance matrix)
  cov <- if(robust) covHuber(data, control=control) else covML(data)
  S <- cov$cov
  # compute coefficients of mediation model
  a <- S[m, x] / S[x, x]
  det <- S[x, x] * S[m, m] - S[m, x]^2
  b <- (-S[m, x] * S[y, x] + S[x, x] * S[y, m]) / det
  c <- (S[m, m] * S[y, x] - S[m, x] * S[y, m]) / det
  cPrime <- S[y, x] / S[x, x]
  # return results
  result <- list(a=a, b=b, c=c, cPrime=cPrime, cov=cov, x=x, y=y, m=m,
                 data=data, robust=robust)
  class(result) <- c("covFitMediation", "fitMediation")
  result
}
