#' Options for the rock package
#'
#' The `rock::opts` object contains three functions to set, get, and reset
#' options used by the rock package. Use `rock::opts$set` to set options,
#' `rock::opts$get` to get options, or `rock::opts$reset` to reset specific or
#' all options to their default values.
#'
#' It is normally not necessary to get or set `rock` options. The defaults implement
#' the Reproducible Open Coding Kit (ROCK) standard, and deviating from these defaults
#' therefore means the processed sources and codes are not compatible and cannot be
#' processed by other software that implements the ROCK. Still, in some cases this
#' degree of customization might be desirable.
#'
#' The following arguments can be passed:
#'
#' \describe{
#'   \item{...}{For `rock::opts$set`, the dots can be used to specify the options
#'   to set, in the format `option = value`, for example, `utteranceMarker = "\n"`. For
#'   `rock::opts$reset`, a list of options to be reset can be passed.}
#'   \item{option}{For `rock::opts$set`, the name of the option to set.}
#'   \item{default}{For `rock::opts$get`, the default value to return if the
#'   option has not been manually specified.}
#' }
#'
#' The following options can be set:
#'
#' \describe{
#'   \item{codeRegexes}{A named character vector with one or more regular
#'   expressions that specify how to extract the codes (that were used to code the
#'   sources). These regular expressions *must* each contain one capturing group
#'   to capture the codes.}
#'
#'   \item{idRegexes}{A named character vector with one or more regular
#'   expressions that specify how to extract the different types of
#'   identifiers. These regular expressions *must* each contain one capturing group
#'   to capture the identifiers.}
#'
#'   \item{sectionRegexes}{A named character vector with one or more regular
#'   expressions that specify how to extract the different types of sections.}
#'
#'   \item{autoGenerateIds}{The names of the `idRegexes` that, if missing, should receive
#'   autogenerated identifiers (which consist of 'autogenerated_' followed by an
#'   incrementing number).}
#'
#'   \item{persistentIds}{The names of the `idRegexes` for the identifiers which, once
#'   attached to an utterance, should be attached to all following utterances as well (until
#'   a new identifier with the same name is encountered, after which that identifier will be
#'   attached to all following utterances, etc).}
#'
#'   \item{noCodes}{This regular expression is matched with all codes after they have been
#'   extracted using the `codeRegexes` regular expression (i.e. they're matched against the
#'   codes themselves without, for example, the square brackets in the default code regex). Any
#'   codes matching this `noCodes` regular expression will be **ignored**, i.e., removed from the
#'   list of codes.}
#'
#'   \item{inductiveCodingHierarchyMarker}{For inductive coding, this marker is used to indicate
#'   hierarchical relationships between codes. The code at the left hand side of this marker will
#'   be considered the parent code of the code on the right hand side. More than two levels
#'   can be specified in one code (for example, if the `inductiveCodingHierarchyMarker` is '>',
#'   the code `grandparent>child>grandchild` would indicate codes at three levels.}
#'
#'   \item{attributeContainers}{The name of YAML fragments containing case attributes (e.g.
#'   metadata, demographic variables, quantitative data about cases, etc).}
#'
#'   \item{codesContainers}{The name of YAML fragments containing (parts of) deductive coding
#'   trees.}
#'
#'   \item{delimiterRegEx}{The regular expression that is used to extract the YAML fragments.}
#'
#'   \item{ignoreRegex}{The regular expression that is used to delete lines before any other
#'   processing. This can be used to enable adding comments to sources, which are then ignored
#'   during analysis.}
#'
#'   \item{utteranceMarker}{How to specify breaks between utterances in the source(s). The
#'   ROCK convention is to use a newline (`\\n`).}
#'
#'   \item{coderId}{A regular expression specifying the coder identifier, specified
#'   similarly to the codeRegexes.}
#'
#'   \item{idForOmittedCoderIds}{The identifier to use for utterances that do not
#'   have a coder id (i.e. utterance that occur in a source that does not specify
#'   a coder id, or above the line where a coder id is specified).}
#'
#'   \item{Two}{Second item}
#' }
#'
#' @aliases opts set get reset
#'
#' @usage opts
#'
#' @examples ### Get the default utteranceMarker
#' rock::opts$get(utteranceMarker);
#'
#' ### Set it to a custom version, so that every line starts with a pipe
#' rock::opts$set(utteranceMarker = "\n|");
#'
#' ### Check that it worked
#' rock::opts$get(utteranceMarker);
#'
#' ### Reset this option to its default value
#' rock::opts$reset(utteranceMarker);
#'
#' ### Check that the reset worked, too
#' rock::opts$get(utteranceMarker);
#'
#' @export
opts <- list();

opts$set <- function(...) {
  dots <- list(...);
  dotNames <- names(dots);
  names(dots) <-
    paste0("rock.", dotNames);
  if (all(dotNames %in% names(opts$defaults))) {
    do.call(options,
            dots);
  } else {
    stop("Option '", option, "' is not a valid (i.e. existing) option for the rock!");
  }
}

opts$get <- function(option, default=FALSE) {
  option <- as.character(substitute(option));
  if (!option %in% names(opts$defaults)) {
    stop("Option '", option, "' is not a valid (i.e. existing) option for the rock!");
  } else {
    return(getOption(paste0("rock.", option),
                     opts$defaults[[option]]));
  }
}

opts$reset <- function(...) {
  optionNames <-
    unlist(lapply(as.list(substitute(...())),
                  as.character));
  if (length(optionNames) == 0) {
    do.call(opts$set,
            opts$defaults);
  } else {
    prefixedOptionNames <-
      paste0("rock.", optionNames);
    if (all(optionNames %in% names(opts$defaults))) {
      do.call(opts$set,
              opts$defaults[optionNames]);
    } else {
      invalidOptions <-
        !(optionNames %in% names(opts$defaults));
      stop("Option(s) ", vecTxtQ(optionNames[invalidOptions]),
           "' is/are not a valid (i.e. existing) option for the rock!");
    }
  }
}

opts$defaults <-
  list(### Used throughout
       codeRegexes = c(codes = "\\[\\[([a-zA-Z0-9._>-]+)\\]\\]"),
       idRegexes = c(caseId = "\\[\\[cid[=:]([a-zA-Z0-9._-]+)\\]\\]",
                     stanzaId = "\\[\\[sid[=:]([a-zA-Z0-9._-]+)\\]\\]",
                     coderId = "\\[\\[coderId[=:]([a-zA-Z0-9._-]+)\\]\\]"),
       sectionRegexes = c(paragraphs = "---paragraph-break---",
                          secondary = "---<[a-zA-Z0-9]?>---"),
       uidRegex = "\\[\\[uid[=:]([a-zA-Z0-9._-]+)\\]\\]",
       inductiveCodingHierarchyMarker = ">",

       ### Used to parse sources
       autoGenerateIds = c('stanzaId'),
       persistentIds = c('caseId', 'coderId'),
       noCodes = "^uid:|^uid=|^dct:|^ci:",
       attributeContainers = c("rock_attributes"),
       codesContainers = c("codes", "dct"),
       delimiterRegEx = "^---$",
       ignoreRegex = "^#",

       ### Used to merge sources
       coderId = "\\[\\[coderId=([a-zA-Z0-9._-]+)\\]\\]",
       idForOmittedCoderIds = "noCoderId",

       ### Used for cleaning sources and adding UIDs
       codeDelimiters = c("[[", "]]"),
       uidPrefix = "uid=",
       utteranceMarker = "\n",
       fragmentDelimiter = "\n\n-----\n\n",
       replacementsPre = list(c("([^\\.])(\\.\\.)([^\\.])",
                                "\\1.\\3"),
                              c("([^\\.])(\\.\\.\\.\\.+)([^\\.])",
                                "\\1...\\3"),
                              c("(\\s*\\r?\\n){3,}",
                                "\n")),
       replacementsPost = list(c("([^\\,]),([^\\s])",
                                 "\\1, \\2")),
       utteranceSplits = c("([\\?\\!]+\\s?|\u2026\\s?|[[:alnum:]\\s?]\\.(?!\\.\\.)\\s?)"),

       ### Used for collecting sources
       utteranceGlue = "\n\n",
       sourceFormatting = "\n\n**Source: `%s`**\n\n",
       codeHeadingFormatting = "%s *(path: %s)*",

       ### Used for generating html
       codeClass = "code",
       idClass = "identifier",
       sectionClass = "sectionBreak",
       uidClass = "uid",
       utteranceClass = "utterance",

       ### Used throughout for working with files
       encoding = "UTF-8",
       preventOverwriting = TRUE,

       ### Used throughout for suppressing messages
       silent = TRUE);
