#' @import formatters
#' @importMethodsFrom formatters toString matrix_form nlines
NULL




# toString ----

## #' @export
## setGeneric("toString", function(x,...) standardGeneric("toString"))

## ## preserve S3 behavior
## setMethod("toString", "ANY", base::toString)

## #' @export
## setMethod("print", "ANY", base::print)


#' Convert an `rtable` object to a string
#'
#'
#' @param x table object
#' @param widths widths of row.name and columns columns
#' @param col_gap gap between columns
#' @param linesep character to create line separator
#' @exportMethod toString
#'
#' @return a string representation of \code{x} as it appears when printed.
#'
#' @examples
#' library(dplyr)
#'
#' iris2 <- iris %>%
#'   group_by(Species) %>%
#'   mutate(group = as.factor(rep_len(c("a", "b"), length.out = n()))) %>%
#'   ungroup()
#'
#' l <- basic_table() %>%
#'   split_cols_by("Species") %>%
#'   split_cols_by("group") %>%
#'   analyze(c("Sepal.Length", "Petal.Width"), afun = list_wrap_x(summary) , format = "xx.xx")
#'
#' tbl <- build_table(l, iris2)
#'
#' cat(toString(tbl, col_gap = 3))
#' @rdname tostring
#' @name tostring
#' @aliases toString,VTableTree-method
setMethod("toString", "VTableTree", function(x,
                                             widths = NULL,
                                             col_gap = 3,
                                             linesep = "\u2014") {
    toString(matrix_form(x, indent_rownames = TRUE),
             widths = widths, col_gap = col_gap,
             linesep = linesep)
})

#' Table shells
#'
#' A table shell is a rendering of the table which maintains the structure, but does not
#' display the values, rather displaying the formatting instructions for each cell.
#'
#' @inheritParams tostring
#' @inheritParams gen_args
#' @return for `table_shell_str` the string representing the table shell, for `table_shell`,
#' `NULL`, as the function is called for the side effect of printing the shell to the console
#' @export
#' @examples
#' library(dplyr)
#'
#' iris2 <- iris %>%
#'   group_by(Species) %>%
#'   mutate(group = as.factor(rep_len(c("a", "b"), length.out = n()))) %>%
#'   ungroup()
#'
#' l <- basic_table() %>%
#'   split_cols_by("Species") %>%
#'   split_cols_by("group") %>%
#'   analyze(c("Sepal.Length", "Petal.Width"), afun = list_wrap_x(summary) , format = "xx.xx")
#'
#' tbl <- build_table(l, iris2)
#' table_shell(tbl)
table_shell <- function(tt, widths = NULL, col_gap =3, linesep = "\u2014") {
    cat(table_shell_str(tt = tt, widths = widths, col_gap = col_gap, linesep = linesep))
}

#' @rdname table_shell
#' @export
table_shell_str <- function(tt, widths = NULL, col_gap =3, linesep = "\u2014") {

    matform <- matrix_form(tt, indent_rownames = TRUE)
    format_strs <- vapply(as.vector(matform$formats),
                          function(x) {
        if(inherits(x, "function"))
            "<fnc>"
        else if(inherits(x, "character"))
            x
        else
            stop("Don't know how to make a shell with formats of class: ", class(x))
    }, "")

    matform$strings <- matrix(format_strs, ncol = ncol(matform$strings),
                              nrow = nrow(matform$strings))
    toString(matform, widths = widths, col_gap = col_gap, linesep = linesep)
}


#' Transform rtable to a list of matrices which can be used for outputting
#'
#' Although rtables are represented as a tree data structure when outputting the table to ASCII or HTML it is useful to
#' map the rtable to an in between state with the formatted cells in a matrix form.
#'
#' @inheritParams gen_args
#' @param indent_rownames logical(1), if TRUE the column with the row names in the `strings` matrix of has indented row
#'   names (strings pre-fixed)
#'
#' @export
#'
#' @details
#'
#' The strings in the return object are defined as follows: row labels are those determined by \code{summarize_rows} and cell values are determined using \code{get_formatted_cells}. (Column labels are calculated using a non-exported internal funciton.
#'
#'@return A list with the following elements:
#' \describe{
#' \item{strings}{The content, as it should be printed, of the top-left material, column headers, row labels , and cell values of \code{tt}}
#' \item{spans}{The column-span information for each print-string in the strings matrix}
#' \item{aligns}{The text alignment for each print-string in the strings matrix}
#' \item{display}{Whether each print-string in the strings matrix should be printed or not}.
#' \item{row_info}{the data.frame generated by \code{summarize_rows(tt)}}
#' }
#'
#' With an additional \code{nrow_header} attribute indicating the number of pseudo "rows"  the
#' column structure defines.
#' @examples
#' library(dplyr)
#'
#' iris2 <- iris %>%
#'   group_by(Species) %>%
#'   mutate(group = as.factor(rep_len(c("a", "b"), length.out = n()))) %>%
#'   ungroup()
#'
#' l <- basic_table() %>%
#'   split_cols_by("Species") %>%
#'   split_cols_by("group") %>%
#'   analyze(c("Sepal.Length", "Petal.Width"), afun = list_wrap_x(summary) , format = "xx.xx")
#'
#' l
#'
#' tbl <- build_table(l, iris2)
#'
#' matrix_form(tbl)
setMethod("matrix_form", "VTableTree",
          function(obj, indent_rownames = FALSE) {

    stopifnot(is(obj, "VTableTree"))

    header_content <- .tbl_header_mat(obj) # first col are for row.names

    ##sr <- summarize_rows(obj)
    sr <- make_row_df(obj)

    body_content_strings <- if (NROW(sr) == 0) {
                                character()
                            } else {
                                cbind(as.character(sr$label), get_formatted_cells(obj))
                            }

    formats_strings <- if (NROW(sr) == 0) {
                           character()
                       } else {
                           cbind(as.character(sr$label), get_formatted_cells(obj, shell = TRUE))
                       }

  tsptmp <- lapply(collect_leaves(obj, TRUE, TRUE), function(rr) {
    sp <- row_cspans(rr)
    rep(sp, times = sp)
  })

  ## the 1 is for row labels
  body_spans <- if (nrow(obj) > 0) {
    cbind(1L, do.call(rbind, tsptmp))
  } else {
    matrix(1, nrow = 0, ncol = ncol(obj) + 1)
  }

    body_aligns <- if(NROW(sr) == 0) {
                              character()
                          } else {
                              cbind("left", get_cell_aligns(obj))
                          }

    body <- rbind(header_content$body, body_content_strings)
    if(disp_ccounts(obj)) {
        formats <- rbind(head(header_content$body, -1), c("", rep(colcount_format(obj), ncol(obj))),
                         formats_strings)
    } else {
        formats <- rbind(header_content$body, formats_strings)
    }
  spans <- rbind(header_content$span, body_spans)
  row.names(spans) <- NULL

  space <- matrix(rep(0, length(body)), nrow = nrow(body))
    aligns <- rbind(matrix(rep("center", length(header_content$body)), nrow = nrow(header_content$body)),
                    body_aligns)

  aligns[, 1] <- "left" # row names and topleft (still needed for topleft)

  ## if (any(apply(body, c(1, 2), function(x) grepl("\n", x, fixed = TRUE))))
  ##   stop("no \\n allowed at the moment")


    nr_header <- nrow(header_content$body)
    if (indent_rownames) {
        body[, 1] <- indent_string(body[, 1], c(rep(0, nr_header), sr$indent))
        formats[,1] <- indent_string(formats[, 1], c(rep(0, nr_header), sr$indent))
    }

    col_ref_strs <- matrix(vapply(header_content$footnotes, function(x) {
        if(length(x) == 0)
            ""
        else
            paste(vapply(x, format_fnote_ref, ""), collapse = " ")
    }, ""), ncol = ncol(body))
    body_ref_strs <- get_ref_matrix(obj)

    body <- matrix(paste0(body,
                         rbind(col_ref_strs,
                               body_ref_strs)),
                   nrow = nrow(body),
                   ncol = ncol(body))

    ref_fnotes <- get_formatted_fnotes(obj)
    MatrixPrintForm(strings = body,
                    spans = spans,
                    aligns = aligns,
                    formats = formats,
                    ## display = display, purely a function of spans, handled in constructor now
                    row_info = sr,
                    ## line_grouping handled internally now line_grouping = 1:nrow(body),
                    ref_fnotes = ref_fnotes,
                    nlines_header = nr_header, ## this is fixed internally
                    nrow_header = nr_header,
                    expand_newlines = TRUE, ## incase the default ever changes
                    has_rowlabs = TRUE,
                    has_topleft = TRUE,
                    main_title = main_title(obj),
                    subtitles = subtitles(obj),
                    page_titles = page_titles(obj),
                    main_footer = main_footer(obj),
                    prov_footer = prov_footer(obj)
                    )

    ## ret <- structure(
    ##     list(
    ##         strings = body,
    ##         spans = spans,
    ##         aligns = aligns,
    ##         display = display,
    ##         row_info = sr,
    ##         line_grouping = 1:nrow(body), # this is done for real in .do_mat_expand now
    ##         ref_footnotes = ref_fnotes
    ##     ),
    ##     nlines_header = nr_header, ## this is done for real in .do_mat_expand nownlines_header,
    ##     nrow_header = nr_header,
    ##     class = c("MatrixPrintForm", "list"))
    ## ## .do_mat_expand(ret)
    ## mform_handle_newlines(ret, has_topleft = TRUE)
})





format_fnote_ref <- function(fn) {
    if(length(fn) == 0 || (is.list(fn) && all(vapply(fn, function(x) length(x) == 0, TRUE))))
        return("")
    else if(is.list(fn) && all(vapply(fn, is.list, TRUE)))
        return(vapply(fn, format_fnote_ref, ""))
    if(is.list(fn)) {
        inds <- unlist(lapply(unlist(fn), function(x) if(is(x, "RefFootnote")) x@index else NULL))
    } else {
        inds <- fn@index
    }
    if(length(inds) > 0) {
        paste0(" {", paste(inds, collapse = ", "), "}")
    } else {
        ""
    }
}


format_fnote_note <- function(fn) {
    if(length(fn) == 0 || (is.list(fn) && all(vapply(fn, function(x) length(x) == 0, TRUE))))
        return(character())
    if(is.list(fn)) {
        return(unlist(lapply(unlist(fn), format_fnote_note)))
    }

    if(is(fn, "RefFootnote")) {
        paste0("{", ref_index(fn), "} - ", ref_msg(fn))
    } else {
        NULL
    }
}

.fn_ind_extractor <- function(strs) {
    res <- suppressWarnings(as.numeric(gsub("\\{([[:digit:]]+)\\}.*", "\\1", strs)))
    if(!(sum(is.na(res)) %in% c(0L, length(res))))
        stop("Got NAs mixed with non-NAS for extracted footnote indices. This should not happen")
    res
}

.colref_mat_helper <- function(vals, span) {
        val <- paste(lapply(vals, format_fnote_ref), collapse = " ")
        if(length(val) == 0)
            val <- ""
        rep(val, times = span)
}

get_colref_matrix <- function(tt) {
    cdf <- make_col_df(tt, visible_only=FALSE)
    objs <- cdf$col_fnotes
    spans <- cdf$total_span
    vals <- mapply(.colref_mat_helper,
                   vals = objs,
                   span = spans)
    vals
}

get_ref_matrix <- function(tt) {
    if(ncol(tt) == 0 || nrow(tt) == 0) {
        return(matrix("", nrow = nrow(tt), ncol = ncol(tt) + 1L))
    }
    rows <- collect_leaves(tt, incl.cont = TRUE, add.labrows = TRUE)
    lst <- unlist(lapply(rows, cell_footnotes), recursive = FALSE)
    cstrs <- unlist(lapply(lst, format_fnote_ref))
    bodymat <- matrix(cstrs,
                      byrow = TRUE,
                      nrow = nrow(tt),
                      ncol = ncol(tt))
    cbind(vapply(rows, function(rw) format_fnote_ref(row_footnotes(rw)), ""), bodymat)
}

get_formatted_fnotes <- function(tt) {
    colresfs <- unlist(make_col_df(tt, visible_only = FALSE)$col_fnotes)
    colstrs <- unlist(lapply(colresfs, format_fnote_note))
    rows <- collect_leaves(tt, incl.cont = TRUE, add.labrows = TRUE)
    lst <- unlist(lapply(rows, cell_footnotes), recursive = FALSE)
    cellstrs <- unlist(lapply(lst, format_fnote_note))
    rstrs <- unlist(lapply(rows, function(rw) format_fnote_note(row_footnotes(rw))))
    allstrs <- c(colstrs, rstrs, cellstrs)
    inds <- .fn_ind_extractor(allstrs)
    allstrs[order(inds)]
}



## ## print depths (not to be confused with tree depths)
## .cleaf_depths <- function(ctree = coltree(cinfo), depth = 1, cinfo) {
##     if(is(ctree, "LayoutColLeaf"))
##         return(depth)
##     unlist(lapply(tree_children(ctree), .cleaf_depths, depth = depth + 1))
## }


## .do_tbl_h_piece <- function(ct, padding = 0, span = 1) {

##     if(is(ct, "LayoutColLeaf")) {
##       ##   padcells <- if(padding > 0) list(rep(list(rcell("", colspan = 1)), padding))

##         return(c(list(rcell(obj_label(ct), colspan = 1))))
##     }


##     nleafs <- length(collect_leaves(ct))
##     padcells <- if(padding > 0) list(rep(list(rcell("", colspan = 1)), padding))

##     kids <- tree_children(ct)
##     cdepths <- vapply(kids, function(k) max(.cleaf_depths(k)), 1)
##     pieces <- mapply(.do_tbl_h_piece,
##                      ct = kids, padding = max(cdepths) - cdepths,
##                      SIMPLIFY= FALSE)
##     ## listpieces <- vapply(pieces, function(x) {
##     ##     any(vapply(x, function(y) is.list(y) && !is(y, "CellValue"), NA))
##     ##     }, NA)
##     ## if(any(listpieces))
##     ##     pieces[listpieces] <- lapply(pieces[listpieces], unlist)
##     lpieces <- vapply(pieces, length, 1L)

##     nmcell <- list(rcell(obj_label(ct), colspan = nleafs))

##     stopifnot(length(unique(lpieces)) == 1)
##     rowparts <- lapply(1:max(lpieces),
##                        function(i) {
##         res = lapply(pieces, `[[`, i = i)
##         if(!are(res, "CellValue"))
##             res = unlist(res, recursive = FALSE)
##         res
##     })


##     c(padcells,
##       nmcell,
##       rowparts)
## }

.do_tbl_h_piece2 <- function(tt) {
    coldf <- make_col_df(tt, visible_only = FALSE)
    remain <- seq_len(nrow(coldf))
    chunks <- list()
    cur <- 1
    retmat <- NULL

    ## each iteration of this loop identifies
    ## all rows corresponding to one top-level column
    ## label and its children, then processes those
    ## with .do_header_chunk
    while(length(remain) > 0) {
        rw <- remain[1]
        inds <- coldf$leaf_indices[[rw]]
        endblock <- which(coldf$abs_pos == max(inds))

        stopifnot(endblock >= rw)
        chunks[[cur]] <- .do_header_chunk(coldf[rw:endblock,])
        remain <- remain[remain > endblock]
        cur <- cur + 1
    }
    chunks <- .pad_tops(chunks)
    lapply(seq_len(length(chunks[[1]])),
           function(i) {
        DataRow(unlist(lapply(chunks, `[[`, i), recursive = FALSE))
    })
}
.pad_end <- function(lst, padto, ncols) {
    curcov <- sum(vapply(lst, cell_cspan, 0L))
    if(curcov == padto)
        return(lst)

    c(lst, list(rcell("", colspan = padto - curcov)))
}


.pad_tops <- function(chunks) {
    lens <- vapply(chunks, length, 1L)
    padto <- max(lens)
    needpad <- lens != padto
    if(all(!needpad))
        return(chunks)

    chunks[needpad] <- lapply(chunks[needpad],
                              function(chk) {
        span <- sum(vapply(chk[[length(chk)]], cell_cspan, 1L))
        needed <- padto - length(chk)
        c(replicate(rcell("", colspan = span),
                    n = needed),
          chk)
    })
    chunks

}

.do_header_chunk <- function(coldf) {
    ## hard assumption that coldf is a section
    ## of a column dataframe summary that was
    ## created with visible_only=FALSE
    nleafcols <- length(coldf$leaf_indices[[1]])

    spldfs <- split(coldf, lengths(coldf$path))
    toret <- lapply(seq_along(spldfs),
                    function(i) {
        rws <- spldfs[[i]]

        thisbit <- lapply(seq_len(nrow(rws)),
                          function(ri) {
            rcell(rws[ri, "label", drop = TRUE], colspan = rws$total_span[ri],
                  footnotes = rws[ri, "col_fnotes", drop = TRUE][[1]])
        })
        .pad_end(thisbit, nleafcols)
    })

    toret
}



.tbl_header_mat <- function(tt) {

    clyt <- coltree(tt)
    rows <- .do_tbl_h_piece2(tt) ##(clyt)
    cinfo <- col_info(tt)

    nc <- ncol(tt)
    body <- matrix(rapply(rows, function(x) {
        cs <- row_cspans(x)
        if (is.null(cs)) cs <- rep(1, ncol(x))
        rep(row_values(x), cs)
    }), ncol = nc, byrow = TRUE)

    span <- matrix(rapply(rows, function(x) {
        cs <- row_cspans(x)
        if (is.null(cs)) cs <- rep(1, ncol(x))
        rep(cs, cs)
    }), ncol = nc, byrow = TRUE)

    fnote <- do.call(rbind,
                           lapply(rows, function(x) {
                              cell_footnotes(x)
                           }))



    if (disp_ccounts(cinfo)) {
        counts <- col_counts(cinfo)
        cformat <- colcount_format(cinfo)
        body <- rbind(body, vapply(counts, format_rcell, character(1), cformat))
        span <- rbind(span, rep(1, nc))
        fnote <- rbind(fnote, rep(list(list()), nc))
    }

    tl <- top_left(cinfo)
    lentl <- length(tl)
    nli <- nrow(body)
    if(lentl == 0)
        tl <- rep("", nli)
    else if(lentl > nli) {
        npad <- lentl - nli
        body <- rbind(matrix("", nrow = npad, ncol = ncol(body)), body)
        span <- rbind(matrix(1, nrow = npad, ncol = ncol(span)), span)
        fnote <- rbind(matrix(list(), nrow = npad, ncol = ncol(body)), fnote)
    } else if (lentl < nli)
        tl <- c(tl, rep("", nli - lentl))

    list(body = cbind(tl, body, deparse.level = 0), span = cbind(1, span),
         footnotes = cbind(list(list()), fnote))
}



# get formatted cells ----

#' get formatted cells
#'
#' @return the formatted print-strings for all (body) cells in \code{obj}.
#' @export
#' @inheritParams gen_args
#' @param shell logical(1). Should the formats themselves be returned instead of the
#' values with formats applied. Defaults to \code{FALSE}.
#' @examples
#'
#' library(dplyr)
#'
#' iris2 <- iris %>%
#'   group_by(Species) %>%
#'   mutate(group = as.factor(rep_len(c("a", "b"), length.out = n()))) %>%
#'   ungroup()
#'
#' tbl <- basic_table() %>%
#'   split_cols_by("Species") %>%
#'   split_cols_by("group") %>%
#'   analyze(c("Sepal.Length", "Petal.Width"), afun = list_wrap_x(summary) , format = "xx.xx") %>%
#'   build_table(iris2)
#'
#' get_formatted_cells(tbl)
#' @rdname gfc
setGeneric("get_formatted_cells", function(obj, shell = FALSE) standardGeneric("get_formatted_cells"))
#' @rdname gfc
setMethod("get_formatted_cells", "TableTree",
          function(obj, shell = FALSE) {

    lr <- get_formatted_cells(tt_labelrow(obj), shell = shell)

    ct <- get_formatted_cells(content_table(obj), shell = shell)

    els <- lapply(tree_children(obj), get_formatted_cells, shell = shell)

    ## TODO fix ncol problem for rrow()
    if (ncol(ct) == 0 && ncol(lr) != ncol(ct)) {
        ct <- lr[NULL, ]
    }

    do.call(rbind, c(list(lr), list(ct),  els))
})

#' @rdname gfc
setMethod("get_formatted_cells", "ElementaryTable",
          function(obj, shell = FALSE) {
    lr <- get_formatted_cells(tt_labelrow(obj), shell = shell)
    els <- lapply(tree_children(obj), get_formatted_cells, shell = shell)
    do.call(rbind, c(list(lr), els))
})

#' @rdname gfc
setMethod("get_formatted_cells", "TableRow",
          function(obj, shell = FALSE) {
            default_format <- if (is.null(obj_format(obj))) "xx" else obj_format(obj)
            format <- lapply(row_cells(obj), function(x) {
                format <- obj_format(x)
                if (is.null(format))
                    default_format
                else
                    format
            })

            matrix(unlist(Map(function(val, format, spn) {
                stopifnot(is(spn, "integer"))
                val <- if(shell) format else paste(format_rcell(val, format), collapse = ", ")
                rep(list(val), spn)
            }, row_values(obj), format, row_cspans(obj))), ncol = ncol(obj))
})

#' @rdname gfc
setMethod("get_formatted_cells", "LabelRow",
          function(obj, shell = FALSE) {
    nc <- ncol(obj) # TODO note rrow() or rrow("label") has the wrong ncol
    vstr <- if(shell) "-" else ""
    if (labelrow_visible(obj)) {
        matrix(rep(vstr, nc), ncol = nc)
    } else {
        matrix(character(0), ncol = nc)
    }
})


#' @rdname gfc
setGeneric("get_cell_aligns", function(obj) standardGeneric("get_cell_aligns"))
#' @rdname gfc
setMethod("get_cell_aligns", "TableTree",
          function(obj) {

    lr <- get_cell_aligns(tt_labelrow(obj))

    ct <- get_cell_aligns(content_table(obj))

    els <- lapply(tree_children(obj), get_cell_aligns)

    ## TODO fix ncol problem for rrow()
    if (ncol(ct) == 0 && ncol(lr) != ncol(ct)) {
        ct <- lr[NULL, ]
    }

    do.call(rbind, c(list(lr), list(ct),  els))
})

#' @rdname gfc
setMethod("get_cell_aligns", "ElementaryTable",
          function(obj) {
    lr <- get_cell_aligns(tt_labelrow(obj))
    els <- lapply(tree_children(obj), get_cell_aligns)
    do.call(rbind, c(list(lr), els))
})

#' @rdname gfc
setMethod("get_cell_aligns", "TableRow",
          function(obj) {
    als <- vapply(row_cells(obj), cell_align, "")
    spns <- row_cspans(obj)

    matrix(rep(als, times = spns),
           ncol = ncol(obj))
})

#' @rdname gfc
setMethod("get_cell_aligns", "LabelRow",
          function(obj) {
    nc <- ncol(obj) # TODO note rrow() or rrow("label") has the wrong ncol
    if (labelrow_visible(obj)) {
        matrix(rep("center", nc), ncol = nc)
    } else {
        matrix(character(0), ncol = nc)
    }
})





## #' Propose Column Widths of an `rtable` object
## #'
## #' The row names are also considered a column for the output
## #'
## #' @param x `rtable` object
## #' @param mat_form object as created with `matrix_form`
## #'
## #' @export
## #' @return a vector of column widths based on the content of \code{x} (or \code{mat_form} if explictly provided)
## #' for use in printing and, in the future, in pagination.
## #' @examples
## #' library(dplyr)
## #'
## #' iris2 <- iris %>%
## #'   group_by(Species) %>%
## #'   mutate(group = as.factor(rep_len(c("a", "b"), length.out = n()))) %>%
## #'   ungroup()
## #'
## #' l <- basic_table() %>%
## #'   split_cols_by("Species") %>%
## #'   split_cols_by("group") %>%
## #'   analyze(c("Sepal.Length", "Petal.Width"), afun = list_wrap_x(summary) , format = "xx.xx")
## #'
## #' tbl <- build_table(l, iris2)
## #'
## #' propose_column_widths(tbl)
## propose_column_widths <- function(x, mat_form = matrix_form(x, indent_rownames = TRUE)) {

##     ##stopifnot(is(x, "VTableTree"))

##   body <- mat_form$strings
##   spans <- mat_form$spans
##   aligns <- mat_form$aligns
##   display <- mat_form$display

##   chars <- nchar(body)

##   # first check column widths without colspan
##   has_spans <- spans != 1
##   chars_ns <- chars
##   chars_ns[has_spans] <- 0
##   widths <- apply(chars_ns, 2, max)

##   # now check if the colspans require extra width
##   if (any(has_spans)) {
##     has_row_spans <- apply(has_spans, 1, any)

##     chars_sp <- chars[has_row_spans, , drop = FALSE]
##     spans_sp <- spans[has_row_spans, , drop = FALSE]
##     disp_sp <- display[has_row_spans, , drop = FALSE]

##     nc <- ncol(spans)
##     for (i in seq_len(nrow(chars_sp))) {
##       for (j in seq_len(nc)) {
##         if (disp_sp[i, j] && spans_sp[i, j] != 1) {
##           i_cols <- seq(j, j + spans_sp[i, j] - 1)

##           nchar_i <- chars_sp[i, j]
##           cw_i <- widths[i_cols]
##           available_width <- sum(cw_i)

##           if (nchar_i > available_width) {
##             # need to update widths to fit content with colspans
##             # spread width among columns
##             widths[i_cols] <- cw_i + spread_integer(nchar_i - available_width, length(cw_i))
##           }
##         }
##       }
##     }
##   }
##   widths
## }

# utility functions ----

#' from sequence remove numbers where diff == 1
#'
#' numbers need to be sorted
#'
#' @noRd
#'
#' @examples
#' remove_consecutive_numbers(x = c(2, 4, 9))
#' remove_consecutive_numbers(x = c(2, 4, 5, 9))
#' remove_consecutive_numbers(x = c(2, 4, 5, 6, 9))
#' remove_consecutive_numbers(x = 4:9)
remove_consecutive_numbers <- function(x) {

  # actually should be integer
  stopifnot(is.wholenumber(x), is.numeric(x), !is.unsorted(x))

  if (length(x) == 0) return(integer(0))
  if (!is.integer(x)) x <- as.integer(x)

  sel <- rep(TRUE, length(x))

  x[c(TRUE, diff(x)  != 1)]
}


#' insert an empty string
#'
#' @noRd
#'
#' @examples
#' empty_string_after(letters[1:5], 2)
#' empty_string_after(letters[1:5], c(2, 4))
empty_string_after <- function(x, indices) {

  if (length(indices) > 0) {
    offset <- 0
    for (i in sort(indices)) {
      x <- append(x, "", i + offset)
      offset <- offset + 1
    }
  }
  x
}

#' Indent Strings
#'
#' Used in rtables to indent row names for the ASCII output.
#'
#' @param x a character vector
#' @param indent a vector of length \code{length(x)} with non-negative integers
#' @param incr non-negative integer: number of spaces per indent level
#' @param including_newline boolean: should newlines also be indented
#'
#' @export
#' @return \code{x} indented by left-padding with code{indent*incr} white-spaces.
#' @examples
#' indent_string("a", 0)
#' indent_string("a", 1)
#' indent_string(letters[1:3], 0:2)
#' indent_string(paste0(letters[1:3], "\n", LETTERS[1:3]), 0:2)
#'
indent_string <- function(x, indent = 0, incr = 2, including_newline = TRUE) {

  if (length(x) > 0) {
    indent <- rep_len(indent, length.out = length(x))
    incr <- rep_len(incr, length.out = length(x))
  }

  indent_str <- strrep(" ", (indent > 0) * indent * incr)

  if (including_newline) {
    x <- unlist(mapply(function(xi, stri) {
      gsub("\n", stri, xi, fixed = TRUE)
    }, x, paste0("\n", indent_str)))
  }

  paste0(indent_str, x)
}

## .paste_no_na <- function(x, ...) {
##   paste(na.omit(x), ...)
## }


## #' Pad a string and align within string
## #'
## #' @param x string
## #' @param n number of character of the output string, if `n < nchar(x)` an error is thrown
## #'
## #' @noRd
## #'
## #' @examples
## #'
## #' padstr("abc", 3)
## #' padstr("abc", 4)
## #' padstr("abc", 5)
## #' padstr("abc", 5, "left")
## #' padstr("abc", 5, "right")
## #'
## #' if(interactive()){
## #' padstr("abc", 1)
## #' }
## #'
## padstr <- function(x, n, just = c("center", "left", "right")) {

##   just <- match.arg(just)

##   if (length(x) != 1) stop("length of x needs to be 1 and not", length(x))
##   if (is.na(n) || !is.numeric(n) || n < 0) stop("n needs to be numeric and > 0")

##   if (is.na(x)) x <- "<NA>"

##   nc <- nchar(x)

##   if (n < nc) stop("\"", x, "\" has more than ", n, " characters")

##   switch(
##     just,
##     center = {
##       pad <- (n - nc)/2
##       paste0(spaces(floor(pad)), x, spaces(ceiling(pad)))
##     },
##     left = paste0(x, spaces(n - nc)),
##     right = paste0(spaces(n - nc), x)
##   )
## }

## spaces <- function(n) {
##   strrep(" ", n)
## }


#' Convert Matrix of Strings into a String with Aligned Columns
#'
#' Note that this function is intended to print simple rectangular
#' matrices and not rtables.
#'
#' @param mat a matrix of strings
#' @param nheader number of header rows
#' @param colsep string that separates the columns
#' @param linesep character to build line separator
#'
#' @noRd
#'
#' @return a string
#'
#' @examples
#'
#' mat <- matrix(c("A", "B", "C", "a", "b", "c"), nrow = 2, byrow = TRUE)
#' cat(rtables:::mat_as_string(mat)); cat("\n")
mat_as_string <- function(mat, nheader = 1, colsep = "    ", linesep = "\u2014") {
  colwidths <- apply(apply(mat, c(1, 2), nchar), 2, max)

  rows_formatted <- apply(mat, 1, function(row) {
    paste(unlist(mapply(padstr, row, colwidths, "left")), collapse = colsep)
  })

  header_rows <- seq_len(nheader)
  nchwidth <- nchar(rows_formatted[1])
  paste(c(rows_formatted[header_rows],
          substr(strrep(linesep, nchwidth), 1, nchwidth),
          rows_formatted[-header_rows]), collapse = "\n")
}
