\name{ndl}
\alias{ndl}
\title{No-Decompression Limit}
\description{
  Compute the No-Decompression Limit (NDL) for a dive 
  to a specified depth, under a Haldane-type
  decompression model.
}
\usage{
ndl(depth, g=air, prevstate=NULL, model = "DSAT")
}
\arguments{
  \item{depth}{
    Depth (in metres) for which the NDL should be computed.
    A single number or a numeric vector.
  }
  \item{g}{
    Breathing gas for the dive. An object of class \code{"gas"}.
  }
  \item{prevstate}{
    Optional. Initial state of the diver.
    A data frame containing the tissue saturations 
    for each tissue compartment in the model, at the start of the dive.
    Defaults to the state of a diver with no previous dive history.
  }
  \item{model}{
    The decompression model. Either an object of class \code{"hm"},
    or a character string giving the name of one of the standard
    Haldane-type models. Options are \code{"DSAT"},
    \code{"USN"}, \code{"Workman"} and \code{"ZH-L16A"}.
  }
}
\details{
  The No-Decompression Limit (NDL) for a given depth \eqn{d} is the
  maximum duration of a No-Decompression dive to depth \eqn{d}.

  A No-Decompression dive is one which can (theoretically) be aborted
  at any time without requiring staged decompression.
  Equivalently, in a No-Decompression Dive, the Nitrogen saturation
  in each of the diver's tissue
  compartments never exceeds the maximum Nitrogen saturation
  tolerated at sea level
  (known as the surfacing M-value for that compartment).

  This algorithm computes the NDL for a dive to the given depth,
  or for several possible alternative dives to the given depths,
  based on the specified decompression
  model. The NDL is determined by the halftimes and the surfacing
  M-values for all the compartments in the model.

  If the breathing gas \code{g} contains Helium, then \code{model}
  must include compartments for Helium. Similar calculations apply in
  this case.

  The argument \code{prevstate} represents the tissue saturation of the
  diver at the start of the dive. It should be a data frame, with one row for
  each compartment of the decompression model, and one column for each
  inert gas (\code{N2} and/or \code{He}). Such data frames
  are usually generated by \code{\link{saturated.state}} or
  \code{\link{haldane}}.
}
\section{Warning}{
  \bold{This is not intended for use as a scuba diving table!}
  The NDL calculated here is not the same as the NDL indicated in any
  published scuba diving table. For example,
  the NDL calculated by this algorithm
  using the halftimes and M-values of the DSAT model,
  may be longer than the NDL published in the DSAT Recreational Dive Planner.
  Infinite values of NDL may be returned in some cases.
}
\value{
  Numeric vector, of the same length as \code{depth},
  giving the No-Decompression Limit in minutes for each depth.
  Infinite values are possible.

  The value also has an attribute called \code{"controlling"} which
  identifies the controlling tissue (the tissue for which the M-value
  is reached).
}
\author{Adrian Baddeley
  \email{adrian@maths.uwa.edu.au}
  \url{http://www.maths.uwa.edu.au/~adrian/}
}
\seealso{
  \code{\link{scuba.disclaimer}},
  \code{\link{haldane}},
  \code{\link{showstates}},
  \code{\link{saturated.state}}
}
\examples{
  # NDL for a dive to 15 metres
  ndl(15)

  # NDL for a dive to 24 metres on EANx 32
  ndl(15, g=nitrox(0.32))

  # NDL for a dive to 60 metres on Trimix 15/50 under ZH-L16A model
  ndl(70, g=trimix(0.15, 0.5), model="Z")

  # NDL for a repetitive dive on air to 15 metres, following an 
  # 18-minute air dive to 30 metres and half-hour surface interval
  firstdive <- dive(c(30,18), c(5,3), c(0,30))
  ndl(15, prevstate=haldane(firstdive))
}
\keyword{utilities}
